#!/usr/bin/env python

# This file is part of Atabake
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Authors: Artur Duque de Souza <artur.souza@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.

__author__ = "Artur Duque de Souza / Leonardo Sobral Cunha"
__author_email__ = "artur.souza@openbossa.org / leonardo.cunha@openbossa.org"

import os
import sys
import imp
import logging

from atabake.lib.utils import which

log = logging.getLogger("atabake.player_set")

__all__ = ("PlayerSet", "load_from_directory")


class PlayerSet(object):
    """Class to hold and organize a set of players"""

    def __init__(self, basetype, *items):
        self.basetype = basetype
        self.map = {}
        self.list = []

        for item in items:
            self._add(item)
        self._sort()

    def _add(self, item):
        self.map[item.name] = item
        self.list.append(item)

    def add(self, item):
        self._add(item)
        self._sort()

    def __getitem__(self, spec):
        if isinstance(spec, basestring):
            return self.map[spec]
        elif isinstance(spec, int):
            return self.list[spec]
        else:
            msg = "Expected str/int, %s found" % type(spec)
            raise TypeError(msg)

    def get(self, name, default=None):
        return self.map.get(name, default)

    def __iter__(self):
        return self.list.__iter__()

    def __len__(self):
        return len(self.list)

    def _sort(self):
        self.list.sort(lambda a, b: cmp(b.priority, a.priority))

    def update(self, playerset):
        self.map.update(playerset.map)
        self.list.extend(playerset.list)
        self._sort()

    def load_from_directory(self, directory):
        for player in load_from_directory(directory, self.basetype):
            self._add(player)
        self._sort()
        log.debug("Loaded players: %s", self.list)

    def __str__(self):
        lst = ['"%s"(%s)' % (player.__name__, player.name) \
               for player in self.list]
        return "%s(basetype=%s, items=[%r])" % \
               (self.__class__.__name__, self.basetype.__name__, lst)


def _load_module(pathlist, name):
    """Own function to load a module (player)"""
    fp, path, desc = imp.find_module(name, pathlist)
    try:
        module = imp.load_module(name, fp, path, desc)
        return module
    finally:
        if fp:
            fp.close()


def load_from_directory(directory, basetype):
    """Function to load players from a given directory"""
    pn = basetype.__name__
    log.debug("Loading players from %s, type=%s", directory, pn)

    s_players = set()
    for d in os.listdir(directory):
        name, ext = os.path.splitext(d)
        if ext in (".py", ".pyo") and name != "__init__":
            s_players.add(name)

    players = []
    for name in s_players:
        try:
            mod = _load_module([directory], name)
        except Exception, e:
            log.warning("Could not load plugin %s/%s: %s",
                        directory, name, e, exc_info=True)
            continue

        for sym in dir(mod):
            cls = getattr(mod, sym)
            if isinstance(cls, type) and issubclass(cls, basetype) and \
               cls != basetype:
                if (which(cls.exec_name) or os.path.exists(cls.exec_name)) and \
                   cls.enabled:
                    players.append(cls)
                    log.info("Loaded %s(%s) from %s",
                             cls.__name__, pn, os.path.join(directory, name))
                else:
                    log.warning("Could not load %s(%s).",
                                cls.__name__, cls.exec_name)
    return players
