#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Gstmp3 Atabake Plugin
# Copyright (C) 2008 Instituto Nokia de Tecnologia
# Author: Adriano Rezende <adriano.rezende@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import pygst
pygst.require("0.10")
import gst
import logging

from atabake.lib.player import Player
from atabake.lib.errors import AtabakeException
from atabake.lib.player_session import PlayerSession

log = logging.getLogger("atabake.player.gstreamer-mp3")


class GstMp3Backend(Player):
    name = "gstreamer-mp3"
    exec_name = "atabake"
    priority = -1
    enabled = True
    max_volume = 65535

    error_bindings = {
        # 1 = Internal data flow error
        1: AtabakeException.ERROR_PLAYER_GENERIC,
        # 3 = Resource not found
        3: AtabakeException.ERROR_FILE_NOT_FOUND,
        # 4 - Resource busy
        4: AtabakeException.ERROR_PLAYING,
        # 9 = Could not read from resource
        9: AtabakeException.ERROR_PLAYING,
        # 12 = Synchronize on the resource failed
        12: AtabakeException.ERROR_PLAYING
        }

    def __init__(self, url, xid, state, eos_cb, error_cb, buff_cb):
        Player.__init__(self, url, xid, state, eos_cb, error_cb, buff_cb)

        self.pipeline = gst.Pipeline('pipeline')
        self.sink = gst.element_factory_make('dspmp3sink', 'sink')
        self.source = gst.element_factory_make('gnomevfssrc', 'source')

        self.pipeline.add(self.sink)
        self.pipeline.add(self.source)
        gst.element_link_many(self.source, self.sink)

        self.volume = None

        self.reset_player()

        bus = self.pipeline.get_bus()
        bus.add_signal_watch()
        self.watch_id = bus.connect('message', self._handle_bus_message)

    def reset_player(self):
        self.info = {}
        self.position = 0
        self.duration = 0
        self.get_media_details()

        return True

    def play(self):
        self.pipeline.set_state(gst.STATE_PLAYING)
        return True

    def pause(self):
        if self.get_state() == PlayerSession.STATE_PAUSED:
            self.pipeline.set_state(gst.STATE_PLAYING)
        else:
            self.pipeline.set_state(gst.STATE_PAUSED)
        return True

    def stop(self):
        self.pipeline.set_state(gst.STATE_NULL)
        return True

    def is_seekable(self):
        return True

    def seek(self, position):
        event = gst.event_new_seek(1.0, gst.FORMAT_TIME,
                                   gst.SEEK_FLAG_FLUSH | gst.SEEK_FLAG_ACCURATE,
                                   gst.SEEK_TYPE_SET, position * gst.MSECOND,
                                   gst.SEEK_TYPE_NONE, 0)

        if not self.pipeline.send_event(event):
            log.error("error while trying to seek position %r" % position)
            return False

        return True

    def get_position(self):
        try:
            position, format = self.pipeline.query_position(gst.FORMAT_TIME)
            position = int(position / gst.MSECOND)
            log.debug("returned position from gstreamer: %d miliseconds", position)
        except Exception, e:
            log.error("error while trying to get position from gstreamer: %s", e)
            position = 0

        return position

    def get_duration(self):
        try:
            self.duration, format = self.pipeline.query_duration(gst.FORMAT_TIME)
            self.duration = int(self.duration/ gst.MSECOND)
            log.debug("returned duration from gstreamer: %d miliseconds",
                      self.duration)
        except Exception, e:
            log.error("error while trying to get duration from gstreamer.: %s", e)
            self.duration = 0

        return self.duration

    def get_media_details(self):
        # TODO
        return self.info

    def set_volume(self, value):
        self.volume = value
        volume = float(value) / 100
        volume = int(volume * self.max_volume)

        log.debug("trying to set volume in gstreamer to: %f", value)
        self.sink.set_property('volume', volume)
        return True

    def set_uri(self, uri):
        Player.set_uri(self, uri)

        self.reset_player()

        if not uri:
            return False

        if uri.find('://') < 0:
            uri = 'file://' + uri

        log.debug("trying to set uri in gstreamer to: %s", uri)
        self.source.set_property('location', uri)
        self.pipeline.set_state(gst.STATE_READY)

        # When setting uri, the volume goes back to maximum.
        # So we need to set again to the last volume played.
        # The problem is when setting the same volume it doesn't
        # come back to normal. So we need to do the hack below.
        if self.volume is not None:
            if self.volume != 0:
                self.set_volume(self.volume - 1)
            else:
                self.set_volume(self.volume + 1)

            self.set_volume(self.volume)

        return True

    def set_fullscreen(self, status):
        return True

    def set_video_window(self, xid):
        return False

    def _handle_bus_message(self, bus, message):
        log.debug("bus message: %s" % message)

        if message.type == gst.MESSAGE_EOS:
            log.debug('received EOS from bus in gstreamer')
            if self.eos_callback:
                self.eos_callback()
        elif message.type == gst.MESSAGE_ERROR:
            err, dbg = message.parse_error()
            log.debug("received error from bus in gstreamer: (%d) %s",
                      err.code, err.message)

            err_code = self.error_bindings.get(err.code,
                                               AtabakeException.ERROR_UNKNOWN)

            if self.error_callback:
                self.error_callback(err_code)
        elif message.type == gst.MESSAGE_SEGMENT_DONE:
            log.debug('received SEGMENT_DONE from bus in gstreamer')

    def delete(self):
        Player.delete(self)
        log.debug("Deleting player")

        bus = self.pipeline.get_bus()
        bus.disconnect(self.watch_id)
        bus.remove_signal_watch()
