/* 
 * File:   scene.cpp
 * Author: pavel
 */

#include "Scene.h"

#ifdef NO_ES
#include <GL/gl.h>
#else
#include <GLES2/gl2.h>
#endif

#include <QGLShader>
#include <QMatrix3x3>
#include <iostream>
#include <cmath>
#include <algorithm>
#include "ObjReader.h"

using namespace std;

#ifdef NO_ES
const int vidWidth = 640 / 2;
const int vidHeight = 480 / 2;
#else
const int vidWidth = 848 / 2;
const int vidHeight = 480 / 2;
#endif

#define PI 3.14159265

static bool byDistFromCamera(Object& a, Object& b) {
    if (!a.alive) // only look at valid matrices
        return false;

    return a.matMV.column(3).length() < b.matMV.column(3).length();
}

static float getYRotation(QMatrix4x4& mat) {
    QVector4D v(1, 0, 0, 0);
    v = mat * v;
    v.setZ(0); // actually y(up)
    v.normalize();

    float sign = v.y() < 0 ? -1.0 : 1.0; // determines quadrant
    return sign * acos(v.x()) * 180.0f / PI;
}

static float getXRotation(QMatrix4x4& mat) {
    QVector4D u(0, 1, 0, 0);
    u = mat * u;
    u.setX(0);
    u.normalize();

    float r = asin(u.y()) * 180.0f / PI;
    return 90-abs(r);
}

Scene::Scene(bool _drawShadows) :
    drawShadows(_drawShadows), fixLight(false), annoMode(false), vq(vidWidth, vidHeight), tr(vidWidth, vidHeight) {

    // AR Tracker Setup
    if (!tr.init("Logitech_Notebook_Pro.cal", 1, 1000))
        cerr << "could not load camera file, will crash :)" << endl;

    tr.setUndistortionMode(ARToolKitPlus::UNDIST_LUT);
    tr.activateAutoThreshold(true);

    angle = 0;

    // Rendering related Setup
    lightPos = QVector3D(0, 0, 3);
    lightAim = QVector3D(0, 0, 0);

    // set up Projection Matrix
    for (int i = 0; i < 16; i++) {
        matProj.data()[i] = tr.getProjectionMatrix()[i];
    }

    //matProj.perspective(33.0, 640/480, 1, 1000);
    //matProj.flipCoordinates(); // convert ARToolkit ModelView Matrix

    QMatrix4x4 matBias;
    matBias.translate(0.5, 0.5, 0.5);
    matBias.scale(0.5);

    matLightPMV.perspective(60.0, 640.0 / 480, 1, 1000);
    matShadowTexProj = matBias * matLightPMV;

    CanvasPlane::setMatFill();
}

/**
 * setup scene objects
 * you can allocate OpenGL data here
 */
void Scene::setupObjects() {
    objects[0].setup(ObjReader("models/casa/casa.obj")); // 7 fps
    objects[0].id = 0;
    objects[1].setup(ObjReader("models/farm/farm.obj")); // 10fps
    objects[1].id = 1; // TODO max id is 8 due to hacky object picking

    activeOb = &objects[0];

    vq.setup();
    sq.setup();
    intf.setup();
}

void Scene::setupShaders() {
    // build Shadow Shader - used for rendering the scene
    shadowProg.addShaderFromSourceFile(QGLShader::Fragment, drawShadows ? "shaders/shadowmap.fs" : "shaders/phong.fs");
    shadowProg.addShaderFromSourceFile(QGLShader::Vertex, "shaders/shadowmap.vs");
    shadowProg.link();
    shadowProg.bind();
    shadowProg.setUniformValue("myTex", 0);
    shadowProg.setUniformValue("depthTex", 1);

    // build Simple Shader - used for rendering depth into the FBO
    simpleProg.addShaderFromSourceFile(QGLShader::Fragment, "shaders/depth.fs");
    simpleProg.addShaderFromSourceFile(QGLShader::Vertex, "shaders/simple.vs");
    simpleProg.link();

    // build Texturing Shader for VideoQuad
    videoProg.addShaderFromSourceFile(QGLShader::Fragment, "shaders/videotex.fs");
    videoProg.addShaderFromSourceFile(QGLShader::Vertex, "shaders/videotex.vs");
    videoProg.link();

    // build Depth Test Shader for DropShadowQuad
    dropShadowProg.addShaderFromSourceFile(QGLShader::Fragment, "shaders/dropshadow.fs");
    dropShadowProg.addShaderFromSourceFile(QGLShader::Vertex, "shaders/shadowmap.vs");
    dropShadowProg.link();
    dropShadowProg.bind();
    dropShadowProg.setUniformValue("depthTex", 1);

    // build Texturing Shader for Tags & Interface
    texturedProg.addShaderFromSourceFile(QGLShader::Fragment, "shaders/textured.fs");
    texturedProg.addShaderFromSourceFile(QGLShader::Vertex, "shaders/textured.vs");
    texturedProg.link();
}

void Scene::setupFBO() {
    /*
     * Create FBO only with DEPTH_COMPONENT for Shadow Mapping
     * this actually does not work as the SGX530 does not support DEPTH_TEXTURE as advertised
     * resort to rendering to RGBA and packing/ unpacking
     */
    unsigned int depth;
    glGenRenderbuffers(1, &depth);
    glBindRenderbuffer(GL_RENDERBUFFER, depth);
    glRenderbufferStorage(GL_RENDERBUFFER, GL_DEPTH_COMPONENT16, shadowMapSize, shadowMapSize);
    glBindRenderbuffer(GL_RENDERBUFFER, 0);

    glGenFramebuffers(1, &fbo);
    glBindFramebuffer(GL_FRAMEBUFFER, fbo);

    glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, (*objects.begin()).depthTex, 0);

    glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, depth);

    if (glCheckFramebufferStatus(GL_FRAMEBUFFER) != GL_FRAMEBUFFER_COMPLETE) {
        cout << "FBO not set up correctly" << endl;
    }

    glBindFramebuffer(GL_FRAMEBUFFER, 0);
}

void Scene::setup() {
    setupObjects();
    setupShaders();
    setupFBO();

    // Set background color
    glClearColor(0.6, 0.8, 1, 1);
    bg = QColor(0, 0, 0, 0);

    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
}

void Scene::cleanup() {
    vq.cleanup();
    sq.cleanup();
    //ob.cleanup();

    glDeleteProgram(shadowProg.programId());
    glDeleteProgram(simpleProg.programId());
    glDeleteProgram(videoProg.programId());
}

void Scene::grabLightPos() {
    /*for (QHash<unsigned int, Object>::iterator i = objects.begin(); i != objects.end(); ++i) {
     Object& ob = i.value();
     // this is the global position, but we do lighting in model space
     //lightPos = QVector3D(ob.matMV.column(3));
     angle = getYRotation(ob.matMV);
     }*/
}

void Scene::putCanvas() {
    annoMode = !annoMode;

    if (!annoMode)
        // unfreeze action
        return;

    Object& ob = min_element(objects.begin(), objects.end(), byDistFromCamera).value();

    if (!ob.alive) {
        // no object visible
        annoMode = false;
        return;
    }

    ob.tags.resize(ob.tags.size() + 1);
    CanvasPlane& cp = ob.tags.last();
    cp.setup(bg);
    cp.matMV = QMatrix4x4(ob.matMV.normalMatrix().transposed());
    cp.matMV.translate(-QVector3D(ob.matMV.column(3)) * 0.7);
    activeCp = &cp;
}

/*
 * Update the ModelView matrices of the visible objects
 */
void Scene::updateObjectMV() {
    vector<int> detected = tr.calc(vq.vid.data);

    for (unsigned int j = 0; j < detected.size(); j++) {
        if (annoMode)
            continue;

        int id = detected[j];

        if (!objects.contains(id))
            continue;

        Object& ob = objects[id];

        tr.selectDetectedMarker(detected[j]);

        // copy data over
        for (int i = 0; i < 16; i++) {
            ob.matMV.data()[i] = tr.getModelViewMatrix()[i];
        }

        //ob.matMV.setToIdentity();
        //ob.matMV.lookAt(QVector3D(3,5,-5), QVector3D(0,0,0), QVector3D(0,1,0));

        ob.matMV.rotate(90, 1, 0, 0); // y and z axes are still swapped
        ob.matAge.start();
    }
}

void Scene::render() {
    QMatrix4x4 rot;
    QMatrix4x4 matLightPMV;
    QVector3D lightPos;

    // caches
    vector<Object*> to_render;
    vector<QMatrix4x4> matPMV;
    vector<QMatrix4x4> matShadowTexProj;
    vector<QVector3D> lightDir;

    updateObjectMV(); // calc -3fps (21)

    // Check which objects we should render
    for (QHash<unsigned int, Object>::iterator i = objects.begin(); i != objects.end(); ++i) {
        Object& ob = i.value();

        ob.alive = ob.matAge.elapsed() <= 300;

        // only render alive objects
        if (ob.alive) {
            to_render.push_back(&ob);

            if (annoMode)
                ob.matAge.start(); // keep visible objects alive while frozen
        }
    }

    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glDisable(GL_DEPTH_TEST);

    // Render the Video Quad (background)
    videoProg.bind();
    vq.render(videoProg, annoMode); // video render limit 24 fps

    // adjust caches
    matPMV.resize(to_render.size());
    matShadowTexProj.resize(to_render.size(), this->matShadowTexProj);
    lightDir.resize(to_render.size());

    // compute matrices and render depth textures
    if(drawShadows) {

        glEnable(GL_DEPTH_TEST);
        glBindFramebuffer(GL_FRAMEBUFFER, fbo);
        glViewport(0, 0, shadowMapSize, shadowMapSize);
        simpleProg.bind();
    }

    for (unsigned int i = 0; i < to_render.size(); i++) {
        Object& ob = *to_render[i];

        rot.setToIdentity();
        matLightPMV = this->matLightPMV;
        lightPos = this->lightPos;

        if (!fixLight) {
            ob.angleY = getYRotation(ob.matMV);
            ob.angleX = getXRotation(ob.matMV);
        }

        float angle = fixLight ? this->angle : 0;

        rot.rotate(ob.angleY + angle, 0, 1, 0);
        rot.rotate(-ob.angleX, 1, 0, 0);
        lightPos = rot * lightPos;

        matLightPMV.lookAt(lightPos, lightAim, QVector3D(0, 1, 0));
        matShadowTexProj[i].lookAt(lightPos, lightAim, QVector3D(0, 1, 0));
        matPMV[i] = matProj * ob.matMV; // matrices -1 fps (20)

        rot.setToIdentity();
        rot.rotate(180 + ob.angleY + angle, 0, 1, 0); // TODO looks like base conversion above is still not right
        lightPos = rot * this->lightPos;
        lightDir[i] = (lightAim - lightPos).normalized();

        if(!drawShadows)
            continue;

        // Render Object from light POV to shadow tex (modelspace)
        glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, ob.depthTex, 0);
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

        simpleProg.setUniformValue("ProjModelView", matLightPMV);

        ob.render(simpleProg);
        sq.render(simpleProg);
    }

    if(drawShadows) {
        glBindFramebuffer(GL_FRAMEBUFFER, 0); // shadowmap -2 (18)
        glViewport(0, 0, width, height);

        glEnable(GL_BLEND);

        // render Drop Shadow
        dropShadowProg.bind();
        glDisable(GL_DEPTH_TEST);
        for (unsigned int i = 0; i < to_render.size(); i++) {
            glActiveTexture(GL_TEXTURE1);
            glBindTexture(GL_TEXTURE_2D, to_render[i]->depthTex);
            glActiveTexture(GL_TEXTURE0);

            dropShadowProg.setUniformValue("scale", to_render[i]->scale);
            dropShadowProg.setUniformValue("ProjModelView", matPMV[i]);
            dropShadowProg.setUniformValue("ShadowTexProj", matShadowTexProj[i]);

            sq.render(dropShadowProg); // -5fps
        }
    }

    glDisable(GL_BLEND);
    glEnable(GL_DEPTH_TEST);
    // Render Objects using Shadowmap
    shadowProg.bind();
    for (unsigned int i = 0; i < to_render.size(); i++) {
        shadowProg.setUniformValue("scale", to_render[i]->scale);
        shadowProg.setUniformValue("lightDir", lightDir[i]);
        shadowProg.setUniformValue("ProjModelView", matPMV[i]);
        shadowProg.setUniformValue("ShadowTexProj", matShadowTexProj[i]);
        shadowProg.setUniformValue("InverseTrans", matProj.normalMatrix()); // dont transform normals
        shadowProg.setUniformValue("id", float(to_render[i]->id + 1) / 255); // convert object id to image id

        to_render[i]->render(shadowProg); // -1 (15)
    }

    // render Annotations
    texturedProg.bind();
    glDisable(GL_DEPTH_TEST);
    glEnable(GL_BLEND);
    for (unsigned int i = 0; i < to_render.size(); i++) {
        Object& ob = *to_render[i];
        for (int j = 0; j < ob.tags.size(); j++) {
            ob.tags[j].render(texturedProg, matPMV[i], annoMode and &ob.tags[j] == activeCp);
        }
    }

    intf.render(texturedProg);

    glDisable(GL_BLEND);
}
