#include "storefood.h"
#include "ui_storefood.h"
#include <QMessageBox>
#include <QTextStream>
#include <QDesktopServices>
#include <QUrl>

StoreFood::StoreFood(QWidget *parent) :
    QDialog(parent),
    ui(new Ui::StoreFood)
{
    ui->setupUi(this);
    parseUDF(true);
}

StoreFood::~StoreFood()
{
    delete ui;
}

void StoreFood::changeEvent(QEvent *e)
{
    QDialog::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void StoreFood::setPoints(QString sNewPts)
{
    ui->txtCost->setText(sNewPts);
}


void StoreFood::setMode(QString sNewMode)
{
    sMode = sNewMode;
}

QString StoreFood::parseFields(bool validateMode)
{
    QString sFormattedStr = "";
    // return empty string if ok, otherwise return error
    // Check Name
    if (ui->txtName->text().length() < 2)
    {
        ui->txtName->setFocus();
        return "Name must be meaningful";
    }

    sFormattedStr = ui->txtName->text();
    sFormattedStr += ",";
    // Check detail
    if (ui->txtDetail->text().length() < 2)
    {
        ui->txtDetail->setFocus();
        return "Detail must be meaningful";
    }
    sFormattedStr += ui->txtDetail->text();
    sFormattedStr += ",";

    // Check weight
    QString sWeight = ui->txtWeight->text();
    // Strip off any trailing weight characters
    QString sTest = sWeight.right(1);
    bool ok;
    int iTestLastChar;
    iTestLastChar = sTest.toInt(&ok);
    while (!ok)
    {
        if (sWeight.length()<=1)
        {
            ui->txtWeight->setFocus();
            return "Weight must be a valid number followed by weight type e.g. 150g";
        }
        sWeight.chop(1);
        sTest = sWeight.right(1);
        int iTestLastChar;
        iTestLastChar = sTest.toInt(&ok);
    }

    double dWeight =  sWeight.toDouble(&ok);
    if (!ok)
    {
        ui->txtWeight->setFocus();
        dWeight =  0;
        return "Weight must be a valid number followed by weight type e.g. 150g";
    }
    sFormattedStr += ui->txtWeight->text();
    sFormattedStr += ",";

    QString sPV = ui->txtCost->text();
    double dPV = sPV.toDouble(&ok);
    if (!ok)
    {
        dPV = 0;
        ui->txtCost->setFocus();
        return "Error getting the cost value. The cost must be numeric";
    }
    sFormattedStr += ui->txtCost->text();
    sFormattedStr += ",";

    QString sCore = ui->txtCore->text().toUpper();
    if (sCore != "N" && sCore != "Y")
    {
        ui->txtCore->setFocus();
        return "The Core value must be either Y or N";
    }
    sFormattedStr += ui->txtCore->text().toUpper();

    if (validateMode)
    {
        return "";
    }
    else
    {
        return sFormattedStr;
    }
}

void StoreFood::on_btnSave_clicked()
{
    // Validate the data
    QString sValid = parseFields(true);
    if (sValid.length()>0)
    {
        QMessageBox::warning(this, tr("Data Validation."),
        sValid);
        return;
    }

    // get the formatted string
    QString formattedStr = parseFields(false);

    // Check for existing key
    QString sKey = getExistingUDFKey(formattedStr);

    // No key and edit mode.
    if (sKey == "" && sMode == "Edit")
    {
        QMessageBox::warning(this, tr("Edit Data."),
        tr("Name not found, cannot edit Name field."));
        return;
    }

    // Found key and new mode.
    if (sKey != "" && sMode != "Edit")
    {
        QMessageBox::warning(this, tr("Add Data."),
        tr("Name already exists in the database, cannot store duplicate name."));
        return;
    }

    if (sMode != "Edit") // Get new key
    {
        sKey = parseUDF(false);
    }
    // Add the formatted string to the Settings
    mSelFood.setValue(sKey, formattedStr);
    sMode = "Edit";

    sKey = parseUDF(true);
}

/* Search the config file for the next
   available key */
QString StoreFood::parseUDF(bool populateList)
{
    QStringList files;
    int rows=1;
    QString sKey;
    QTextStream(&sKey) << "UDF/UDF" << rows;    

    QString line = mSelFood.value(sKey).toString();

    if (populateList)
    {
        ui->lvExisting->clear();
    }


    while (line!="" && line.left(2) != "--")
    {
        if (populateList)
        {
            ui->lvExisting->addItem(line);
        }
        rows++;
        sKey="";
        QTextStream(&sKey) << "UDF/UDF" << rows;
        line = mSelFood.value(sKey).toString();
    }
    return sKey;
}

/* Search the config file for an existing key */
QString StoreFood::getExistingUDFKey(QString sName)
{
    QStringList files;
    int rows=1;
    int gap=0;
    QString sKey;
    QTextStream(&sKey) << "UDF/UDF" << rows;

    QString line = mSelFood.value(sKey).toString();

    while (line!="" || gap < 10)
    {
      //  QMessageBox::warning(this, tr("Key"), sKey + " " + line + " " + sName);

        if (line!="")
        {
            QStringList qslItems;
            qslItems = line.split(",",QString::KeepEmptyParts );
            QStringList::Iterator it = qslItems.begin();
            QString sUDFName = *it;

            if (sUDFName.toUpper() == sName.toUpper())
            {
    //            QMessageBox::warning(this, tr("Key"), "That was the one!");

                return sKey;
            }

            rows++;
            gap=0;
        }
        else
        {
            rows++;
            gap++;
        }
        sKey = "";
        QTextStream(&sKey) << "UDF/UDF" << rows;
        line = mSelFood.value(sKey).toString();
    }
    return "";
}

/* Search the config file for an existing key */
QString StoreFood::getExistingUDFLine(QString sCompLine)
{
    QStringList files;
    int rows=1;
    int gap=0;
    QString sKey;
    QTextStream(&sKey) << "UDF/UDF" << rows;

    QString line = mSelFood.value(sKey).toString();

    while (line!="" || gap < 10)
    {
//        QMessageBox::warning(this, tr("Key"), sKey + " " + line + " " + sCompLine);

        if (line!="")
        {
            if (line.toUpper() == sCompLine.toUpper())
            {
  //              QMessageBox::warning(this, tr("Key"), "That was the one!");

                return sKey;
            }

            rows++;
            gap=0;
        }
        else
        {
            rows++;
            gap++;
        }
        sKey = "";
        QTextStream(&sKey) << "UDF/UDF" << rows;
        line = mSelFood.value(sKey).toString();
    }
    return "";
}

void StoreFood::setUpTextBoxes(QString line)
{
    QStringList qslItems;
    qslItems = line.split(",",QString::KeepEmptyParts );
    QStringList::Iterator it = qslItems.begin();
    int column=0;

    while(it != qslItems.end())
    {
        switch (column)
        {
            case 0:
            {
                ui->txtName->setText(*it);
                break;
            }
            case 1:
            {
                ui->txtDetail->setText(*it);break;
            }
            case 2:
            {
                ui->txtWeight->setText(*it);break;
            }
            case 3:
            {
                ui->txtCost->setText(*it);break;
            }
            case 4:
            {
                ui->txtCore->setText(*it);break;
            }
        }

        it++;
        column++;
    }
}


void StoreFood::on_btnEdit_clicked()
{
    QList<QListWidgetItem*> selItems = ui->lvExisting->selectedItems();
    if (selItems.count()!=1)
    {
        QMessageBox::warning(this, tr("Edit Data."),
        tr("You must select a single item first."));
        return;
    }

    ui->txtName->setEnabled(false);
    ui->txtDetail->setEnabled(false);
    setMode("Edit");

    // Get the line to edit
    QListWidgetItem* item = selItems.first();
    QString line =  item->text();

    // Now parse the line into it's parts
    setUpTextBoxes(line);

}


void StoreFood::setDisableName()
{
    ui->txtName->setEnabled(false);
    ui->txtDetail->setEnabled(false);
}

void StoreFood::setEnableName()
{
    ui->txtName->setEnabled(true);
    ui->txtDetail->setEnabled(true);
}


void StoreFood::on_btnDelete_clicked()
{
    QList<QListWidgetItem*> selItems = ui->lvExisting->selectedItems();
    if (selItems.count()!=1)
    {
        QMessageBox::warning(this, tr("Delete Data"),
        tr("You must select a single item first."));
        return;
    }

    setMode("Delete");

    // Get the line to edit
    QListWidgetItem* item = selItems.first();
    QString line =  item->text();
    QString sKey = getExistingUDFLine(line);
    QMessageBox::warning(this, tr("Key"), sKey);

    mSelFood.setValue(sKey, "--,--,0,0,N");

//    mSelFood.remove(sKey);
    parseUDF(true);

}

void StoreFood::on_btnNew_clicked()
{
    setMode("New");
    ui->txtName->setEnabled(true);
    ui->txtDetail->setEnabled(true);
    setUpTextBoxes(",,,,");
}

void StoreFood::on_btnCancel_clicked()
{

}

void StoreFood::on_btnClose_clicked()
{
    this->close();
}

void StoreFood::on_txtCost_textChanged(QString )
{
    QString qNewValue = ui->txtCost->text();
    bool ok;
    double dTest = qNewValue.toDouble(&ok);
    int dotCnt=0;
    if (!ok)
    {
        QString qStripped;

        for (int i=0; i< qNewValue.length(); i++ )
        {
            QString subStr = qNewValue.at(i);
            QString sValid = "1234567890.";
            if (subStr.toLower() == ":")
            {
                subStr = ".";
            }
            if (subStr.toLower() == "?")
            {
                subStr = ".";
            }

            if (subStr==".")
            {
                dotCnt++;
            }
            if (subStr.toLower() == "q")
            {
                subStr = "1";
            }
            if (subStr.toLower() == "w")
            {
                subStr = "2";
            }
            if (subStr.toLower() == "e")
            {
                subStr = "3";
            }
            if (subStr.toLower() == "r")
            {
                subStr = "4";
            }
            if (subStr.toLower() == "t")
            {
                subStr = "5";
            }
            if (subStr.toLower() == "y")
            {
                subStr = "6";
            }
            if (subStr.toLower() == "u")
            {
                subStr = "7";
            }
            if (subStr.toLower() == "i")
            {
                subStr = "8";
            }
            if (subStr.toLower() == "o")
            {
                subStr = "9";
            }
            if (subStr.toLower() == "p")
            {
                subStr = "0";
            }
            if ( (sValid.contains( subStr)) && (dotCnt <= 1))
            {
                qStripped.append(subStr);
            }
        }
        ui->txtCost->setText(qStripped);
    }
}

void StoreFood::on_btnHelp_clicked()
{
    QUrl url;
    url.setUrl("https://garage.maemo.org/plugins/wiki/index.php?FoodFinder&id=1325&type=g");
    QDesktopServices::openUrl(url);
}
