/*
 * Some parts of this file may be from zeemoted, which is also under GPL.
 *
 * All original portions are
 * Copyright (C) 2009 <davidfalkayn@gmail.com>.
 *
 * Accelemymote is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Accelemymote is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with accelemymote.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "accelemymote.h"
#include "uinput.h"
#include "inih/ini.h" // BSD licensed utility routine

#include <linux/uinput.h>
#include <sys/stat.h>

inline float deg_to_rad(int degrees)
{
  return PI*degrees/180.0;
}

static int handler(void* setting, const char* section, const char* name, const char* value) // Ini file parse handler
{ // Requires global debug variable as-is
    configuration* config = (configuration*)setting;
    #define MATCH(s, n) strcasecmp(section, s) == 0 && strcasecmp(name, n) == 0
    if (MATCH("format", "version")) {
        config->version = atoi(value);
    } else if (MATCH("sampling", "samples")) {
        if (atoi(value) < 11 && atoi(value) > 0) config->samples = atoi(value);
    } else if (MATCH("sampling", "frame_rate")) {
        if (atoi(value) < 31 && atoi(value) > 0) config->frame_rate = atoi(value);
    } else if (MATCH("control", "offset_pitch")) {
        config->offset_pitch = deg_to_rad(atoi(value));
    } else if (MATCH("control", "flip_pitch")) {
        config->flip_pitch = atoi(value)%2;
        if (!config->flip_pitch) config->flip_pitch = -1;
    } else if (MATCH("control", "max_roll")) {
        config->max_roll = deg_to_rad(atoi(value));
    } else if (MATCH("control", "max_pitch")) {
        config->max_pitch = deg_to_rad(atoi(value));
    } else if (MATCH("control", "threshold_roll")) {
        config->threshold_roll = deg_to_rad(atoi(value));
    } else if (MATCH("control", "threshold_pitch")) {
        config->threshold_pitch = deg_to_rad(atoi(value));
    } else if (MATCH("control", "profile")) {
        config->profile = strdup(value);
    } else if (MATCH("control", "turtle")) {
        config->turtle = atoi(value);
    } else if (MATCH("control", "proximity")) {
        config->proximity = atoi(value);
    } else if (MATCH("control", "autofire0")) {
        config->autofire0 = atoi(value);
    }
}

inline int read_accel(int *x, int *y, int *z)
{
  FILE *fd;
  int results;

  int count = 1;
  while(count)
  {
    fd = fopen(accel_filename, "r");
    if(!fd) return 0;	
    results = fscanf((FILE*) fd,"%i %i %i",x,y,z);	
    fclose(fd);	
    if(results != 3) return 0;
    count--;
  }
  return 1;
}

char read_prox()
{
  FILE *fd;
  char proximity;
  fd = fopen(prox_filename,"r");
  proximity = fgetc(fd); // 'o'pen or 'c'losed
  fclose(fd);
  return proximity;
}

// Abstracts the raw accelerometer readings as follows:
// * creates an average of samples# of readings sample_interval microseconds apart
// * returns 0 or number of failed reads if it fails during any accelerometer reads
inline int do_poll(int samples, const struct timespec* sample_interval, int* x, int* y, int* z) 
{
    int error = 0, i;
    long x_tot = 0, y_tot = 0, z_tot = 0;
    for (i=samples; i > 0; i--)
    {
      nanosleep(sample_interval, 0); // sleep first so that the lag is minimized between collection and response
      if(!read_accel(x, y, z)) { error++; }
      x_tot += *x, y_tot += *y, z_tot += *z;
    }
    *x = x_tot/samples;
    *y = y_tot/samples;
    *z = z_tot/samples;
    return error;
}

inline void clamp(float *angle, float limit)
{
  if (*angle > limit) *angle=limit;
  else if (*angle < -limit) *angle=-limit;
}

int init(char* config_file, void* config) // Returns filedescriptor for uinput device (nonzero on success)
{
  int x,y,z, uinput_fd;
  printf("accelemymote V%s\n",VERSION);

  printf("Trying to access accelerometer ...\n");
  if(!read_accel(&x, &y, &z)) { printf("failed.\n"); return -1; }
  else printf("  opened accelerometer at %s.\n", accel_filename);

  uinput_fd = init_uinput_device();
  if (uinput_fd > 0) printf("Accelemymote driver successfully installed for use as joystick.\n");
  else return 0;

  // TODO: set defaults here
  if (ini_parse(config_file, handler, config) < 0){
    printf("Error: can't load configuration from %s!\n", config_file);
    return 0;
  }
  ((configuration*)config)->freshness = time(0);
  printf("Config loaded from %s. Profile=%s and poll rate=%d/s.\n",
    config_file, ((configuration*)config)->profile, ((configuration*)config)->frame_rate*((configuration*)config)->samples);

  /* If no trigger_filename is set, the process will run indefinitely.*/
  if (trigger_filename[0])
  {
    printf("Trigger file %s specified; ", trigger_filename);
    if (access(trigger_filename, F_OK)) 
    {
      printf("file not found... shutting down.\n");
      return 0;
    }
    else printf("starting service and watching file.\n");
  } else {
    printf("No trigger file specified. Kill service process when finished.\n");
  }

  return uinput_fd;
}


int main(int argc, char **argv)
{
  char proximity;
  int button0 = 0;
  int uinput_fd; 
  int ax, ay, az;
  float acc; // Total acceleration (Sqrt of sum of squares)

  // Roll and pitch in radians
  orientation current, previous;

  unsigned int read_errors = 0;
  unsigned long frames = 1, events = 0, turtle_timer = 0, status_poll = 0;

  struct timespec sample_interval;
  struct stat cfg_stat;

  configuration config;
  if (argc > 1) config_file = argv[1]; // Only valid command line parameter is alternative cfg file
  uinput_fd = init(config_file, &config);
  if (!uinput_fd) return -1;

  sample_interval.tv_sec = 0;
  sample_interval.tv_nsec = ONE_SECOND / (config.frame_rate * config.samples);

  if (strcasecmp(config.profile, "debug") == 0) debug = 1;

  while(1) { // Begin main loop. This only exits if a trigger file that was specified is deleted.

    read_errors+= do_poll(config.samples, &sample_interval, &ax, &ay, &az);
    if (config.proximity){
      proximity = read_prox();
    }
    frames++;

    // Monitor the state of the config file and trigger file infrequently
    if (!(frames % 10) && status_poll < time(0)-5){
      status_poll = time(0);
      if (trigger_filename[0] && access(trigger_filename, F_OK)) break; // if trigger file is gone, quit...
      stat(config_file, &cfg_stat);
      if (cfg_stat.st_mtime > config.freshness) { // or if configuration has been modified...
        if (ini_parse(config_file, handler, &config) < 0){
          printf("Error: can't refresh configuration from %s!\n", config_file);
          return 0;
        }
        if (debug) printf("Reconfigured using new %s file.\n", config_file);
        config.freshness = time(0);
        sample_interval.tv_nsec = ONE_SECOND / (config.frame_rate * config.samples);
      }
    }

    /* Handle turtle mode if:
         user has flipped phone upside down
         or proximity is covered and that is enabled*/
    if (config.turtle && az > 750 || config.proximity == 1 && proximity == 'c' ) { // TODO: may need filtering to avoid accidental activation during motion if poll time is short
      turtle_timer = time(0); // Only used to decide whether to toggle (<10 s), but could e.g. shutdown service if left upside down
      sample_interval.tv_nsec = ONE_SECOND * 2lu;
      while (!do_poll(1, &sample_interval, &ax, &ay, &az) && az > 100 || config.proximity ==1 && read_prox() == 'c') // Poll slowly until flipped face-up again and proximity sensor is uncovered.
        if (trigger_filename[0] && access(trigger_filename, F_OK)) break; // Don't forget to check for shutdown
      if (config.turtle < 0 || turtle_timer && time(0) - turtle_timer > 10 ) {
        turtle_timer = 0; 
        config.turtle = 1; // Turtle now enabled but inactive
        sample_interval.tv_nsec = ONE_SECOND / (config.frame_rate * config.samples);
      } else { 
        // Center joystick before turtling
        do_uinput(uinput_fd, ABS_X, 0, EV_ABS);
        do_uinput(uinput_fd, ABS_Y, 0, EV_ABS);
        config.turtle = -1; // Turtle active now; time to sleep
        sample_interval.tv_nsec = ONE_SECOND / 2.5;
      }
    }
    if (config.turtle == -1) continue; //skip events in turtle mode
 
    // Calculate tilt and fire joystick events
    //
    // TODO: Keep state and/or watch total acceleration to subtract out "gestures". Also, the math could probably use some work.
    acc = sqrtf(((long)ax)*ax + ((long)ay)*ay + ((long)az)*az);  //Use long to avoid overflow when squaring ~1000 to get vector length!

    current.pitch = asinf(ay / acc)+config.offset_pitch;
    current.roll = -asinf(ax / acc);
    if (debug && !(frames%24)) 
      printf("Frame: acceleration(%d,%d,%d)=%.0f. Tilt (%0.2f,%0.2f) with offset %0.2f included.\n",ax,ay,az,acc,current.pitch, current.roll, config.offset_pitch);

    clamp(&current.pitch, config.max_pitch), clamp(&current.roll, config.max_roll);

    // Handle proximity sensor as a button
    if (config.proximity == 2) {
      if (proximity == 'c') {
        if (!button0 || config.autofire0){
          button0++;
          do_uinput(uinput_fd, BTN_A, 1, EV_KEY); // Send simulated button press to UINPUT
        }
      } else {
        if (button0){
          button0 = 0;
          do_uinput(uinput_fd, BTN_A, 0, EV_KEY); // Send simulated button release to UINPUT
        }
      }
    }

    // Send significantly changed control states to UINPUT
    if ((current.roll-previous.roll) * (current.roll-previous.roll) > config.threshold_roll*config.threshold_roll) // abs()
    {
      do_uinput(uinput_fd, ABS_X, (current.roll/config.max_roll) * (MAX_AXIS/2), EV_ABS);
      previous.roll = current.roll;
    }
    if ((current.pitch-previous.pitch) * (current.pitch-previous.pitch) > config.threshold_pitch*config.threshold_pitch) // abs()
    {
      do_uinput(uinput_fd, ABS_Y, ((current.pitch/config.max_pitch) * (MAX_AXIS/2) * config.flip_pitch), EV_ABS);
      previous.pitch = current.pitch;
    }

  } // End main loop

  // Center joystick before quitting
  do_uinput(uinput_fd, ABS_X, 0, EV_ABS);
  do_uinput(uinput_fd, ABS_Y, 0, EV_ABS);
  printf("Accelemymote service shutting down. The program made %ik reads of the accelerometer with %i errors.\n",config.samples*frames/1000, read_errors);
  return 0;
}
