/*
 * Some parts of this file may be from zeemoted, which is also under GPL.
 *
 * All original portions are
 * Copyright (C) 2009 <davidfalkayn@gmail.com>.
 *
 * Accelemymote is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Accelemymote is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with accelemymote.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "accelemymote.h"
#include "uinput.h"
#include "inih/ini.h" // BSD licensed utility routine

#include <linux/uinput.h>

static int handler(void* setting, const char* section, const char* name, const char* value) // Ini file parse handler
{
    configuration* config = (configuration*)setting;
    config->flip_pitch = -1;
    config->poll_rate = 24;
    config->offset_pitch = 10;
    config->profile = "";

    #define MATCH(s, n) strcasecmp(section, s) == 0 && strcasecmp(name, n) == 0
    if (MATCH("format", "version")) {
        config->version = atoi(value);
    } else if (MATCH("sampling", "samples")) {
        config->samples = atoi(value);
    } else if (MATCH("sampling", "poll_rate")) {
        config->poll_rate = atoi(value);
    } else if (MATCH("control", "offset_pitch")) {
        config->offset_pitch = atoi(value);
    } else if (MATCH("control", "flip_pitch")) {
        config->flip_pitch = atoi(value)%2;
	if (!config->flip_pitch) config->flip_pitch = -1;
    } else if (MATCH("control", "max_roll")) {
        config->max_roll = atoi(value);
    } else if (MATCH("control", "max_pitch")) {
        config->max_pitch = atoi(value);
    } else if (MATCH("control", "threshold_roll")) {
        config->threshold_roll = atoi(value);
    } else if (MATCH("control", "threshold_pitch")) {
        config->threshold_pitch = atoi(value);
    } else if (MATCH("control", "profile")) {
        config->profile = strdup(value);
    }
}

inline float deg_to_rad(int degrees)
{
  return PI*degrees/180.0;
}

inline int read_accel(int *x, int *y, int *z)
{
  FILE *fd;
  int results;

  int count = 1;
  while(count)
  {
    fd = fopen(accel_filename, "r");
    if(!fd) return 0;	
    results = fscanf((FILE*) fd,"%i %i %i",x,y,z);	
    fclose(fd);	
    if(results != 3) return 0;
    count--;
  }
  return 1;
}

inline void clamp(float *angle, float limit)
{
  if (*angle > limit) *angle=limit;
  else if (*angle < -limit) *angle=-limit;
}

int init(char* config_file, void* config) // Returns filedescriptor for uinput device (nonzero on success)
{
  int x,y,z, uinput_fd;
  printf("accelemymote V%.2f\n",VERSION);

  printf("Trying to access accelerometer ...\n");
  if(!read_accel(&x, &y, &z)) { printf("failed.\n"); return -1; }
  else printf("  opened accelerometer at %s.\n", accel_filename);

  uinput_fd = init_uinput_device();
  if (uinput_fd > 0) printf("Accelemymote driver successfully installed for use as joystick.\n");
  else return 0;

  /* If no trigger_filename is set, the process will run indefinitely.*/
  if (trigger_filename[0])
  {
    printf("Trigger file %s specified; ", trigger_filename);
    if (access(trigger_filename, F_OK)) 
    {
      printf("file not found... shutting down.\n");
      return 0;
    }
    else printf("starting service and watching file.\n");
  } else {
    printf("No trigger file specified. Kill service process when finished.\n");
  }

  if (ini_parse(config_file, handler, config) < 0) {
    printf("Error: can't load configuration from %s!\n", config_file);
    return 0;
  }
  printf("Config loaded from %s. Profile=%s.\n",
    config_file, ((configuration*)config)->profile);
  return uinput_fd;
}

int main(int argc, char **argv)
{
  int uinput_fd;
  int ax, ay, az;
  int oax, oay, oaz;
  long ax_bar, ay_bar, az_bar; //Avoid overflow when squaring ~1000 to normalize
  float normalize; // Sqrt of sum of squares
  float offset_pitch; //Radians to neutral

  // Roll and pitch in radians
  orientation current, previous;
  orientation max, threshold;

  unsigned int read_errors = 0;
  unsigned long reads = 1;

  configuration config;
  if (argc > 1) config_file = argv[1]; // Only valid command line parameter is alternative cfg file
  uinput_fd = init(config_file, &config);
  if (!uinput_fd) return -1;

  max.pitch = deg_to_rad(config.max_pitch);
  max.roll = deg_to_rad(config.max_roll);
  threshold.pitch = deg_to_rad(config.threshold_pitch);
  threshold.roll = deg_to_rad(config.threshold_roll);
  offset_pitch = deg_to_rad(config.offset_pitch);

  if (strcasecmp(config.profile, "debug") == 0) debug = 1;

  while(1) { // Begin main loop. This only exits if a trigger file was specified and it is deleted.

    // Average new accelerations with previous accelerations (poor man's smoothing function)
    oax = ax, oay = ay, oaz = az;
    if(!read_accel(&ax, &ay, &az)) { read_errors++; }
    reads++;
    if (!(reads % 500) && trigger_filename[0] && access(trigger_filename, F_OK)) break; // trigger file is gone, so quit
 
    ax_bar = (oax+ax)/2;
    ay_bar = (oay+ay)/2;
    az_bar = (oaz+az)/2;
    
    // TODO: Keep state and/or watch total acceleration to subtract out "gestures".
    // Also, the math could probably use some work.
    normalize = sqrtf( ax_bar*ax_bar + ay_bar*ay_bar + az_bar*az_bar);
    current.pitch = asinf( ay_bar / normalize)+offset_pitch;
    current.roll = -asinf( ax_bar / normalize);

    clamp(&current.pitch, max.pitch), clamp(&current.roll, max.roll);
    
    // Send significantly changed control states to UINPUT
//    if ((current.roll-previous.roll) * (current.roll-previous.roll) > threshold.roll*threshold.roll) // abs()
//    {
      do_uinput(uinput_fd, ABS_X, (current.roll/max.roll) * (MAX_AXIS/2), EV_ABS); //Currently only emulates keypresses ("hat" profile)
      previous.roll = current.roll;
//    }
//    if ((current.pitch-previous.pitch) * (current.pitch-previous.pitch) > threshold.pitch*threshold.pitch) // abs()
//    {
      do_uinput(uinput_fd, ABS_Y, ((current.pitch/max.pitch) * (MAX_AXIS/2) * config.flip_pitch), EV_ABS);
      previous.pitch = current.pitch;
//    }

    usleep(1000000u / config.poll_rate);
  } // End main loop

  printf("Accelemymote service shutting down. The program made %ik reads of the accelerometer with %i errors.\n",reads/1000, read_errors);
  return 0;
}
