#!/usr/bin/env python2.5
# -*- coding: utf-8 -*-
#
# accDisplay
#
# Copyright (c) 2009 cpscotti (Clovis Peruchi Scotti)
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
#
# How the whole thing works?
#   When you open the app it is in "live" mode and does nothing more than display data from the accelerometers
#   Pressing on "Record" imediately starts to record accelerometer data (not smooth; raw data) until the user presses:
#              "stop", it then opens a dialog to select where to save the data
#              "clear", sweeps the recording buffer and continues recording
#   Pressing on "open" will trigger an Open File dialog to open a file previously saved.. after succesful loading the user may press:
#               "Play/Pause" in order to view a playback of the loaded data
#               "Close", to close the file and go back to the main screen
#               Slide the TimeScale to change the display frequency
#   the "smoothing" checkbox applies (first order) Exponential Smoothing to the accelerometer data BEFORE displaying ONLY (the recorded data is always raw)
#        alpha = 0.6
#
# [cpscotti, roboscotti, scotti@ieee.org]

import hildon
import gtk
import osso
import time

import gobject

import math

#import gconf
import accFileIF
import os

class AppState:
	(LIVE, RECORDING, PLAYBACK) = range(3)

class accDisplay (hildon.Window):
	
	def __init__(self):
		hildon.Window.__init__ (self)

		self.running = True
		self.last_read = [0,0,0]

		self.appState = AppState.LIVE
		self.data_storage = []
		self.data_ptr = []
		self.tstamp = ""
		self.period = 1.0/20
		self.paused = True
		self.smoothFlag = True

		self.set_title("Accelerometer Viewer")
		self.build_gui()
		self.show()
		self.connect("delete-event",self.on_close)

		while gtk.events_pending():
			gtk.main_iteration(False)
		
		#self.be = jumpBE.JumpBe()
		self.main_loop()

	def main_loop(self):
		while self.running:
			if (not self.paused) or (not self.appState == AppState.PLAYBACK):
				time.sleep(self.period);
				if self.smoothFlag:
					t = self.filter(self.acquisition())
				else:
					t = self.acquisition()
				self.x_label.set_text("X: " + str(math.ceil(t[0]*0.1)*0.01))
				self.y_label.set_text("Y: " + str(math.ceil(t[1]*0.1)*0.01))
				self.z_label.set_text("Z: " + str(math.ceil(t[2]*0.1)*0.01))
				self.last_read = t
				self.expose_handler(self.canvas,0)
			while gtk.events_pending():
				gtk.main_iteration(False)
			
			if not self.running:
				break

	def fix_x(self, cx):
		return int(cx+self.w/2)

	def fix_y(self, cy):
		return int(cy+self.h/2)

	def filter(self, newM):
		out = []
		out.append(0.4*self.last_read[0]+0.6*newM[0])
		out.append(0.4*self.last_read[1]+0.6*newM[1])
		out.append(0.4*self.last_read[2]+0.6*newM[2])
		return out

	def expose_handler(self, widget, event):
		self.w, self.h = widget.window.get_size()
		xgc = widget.window.new_gc()
		xgc.set_rgb_fg_color(gtk.gdk.color_parse("black"))
		widget.window.draw_rectangle(xgc, True, 0,0, 800, 400)
		xgc.set_rgb_fg_color(gtk.gdk.color_parse("brown"))
		widget.window.draw_line(xgc, self.fix_x(-400), self.fix_y(0),self.fix_x(400),self.fix_y(0))
		widget.window.draw_line(xgc, self.fix_x(0), self.fix_y(-200),self.fix_x(0),self.fix_y(200))
		widget.window.draw_line(xgc, self.fix_x(-430), self.fix_y(250),self.fix_x(430),self.fix_y(-250))
		xgc.line_width = 3
		xgc.set_rgb_fg_color(gtk.gdk.color_parse("yellow"))
		widget.window.draw_line(xgc, self.fix_x(0), self.fix_y(0),self.fix_x(-0.1*self.last_read[0]),self.fix_y(0.0))
		widget.window.draw_line(xgc,self.fix_x(0),self.fix_y(0),self.fix_x(0),self.fix_y(-0.1*self.last_read[1]))
		widget.window.draw_line(xgc,self.fix_x(0),self.fix_y(0),self.fix_x(-0.086*self.last_read[2]),self.fix_y(0.05*self.last_read[2]))
		#x = math.cos(math.pi/6)*0.5 = 0.43
		#y = math.sin(math.pi/6)*0.5 = 0.25

		xgc.set_rgb_fg_color(gtk.gdk.color_parse("blue"))
		xgc.line_width = 4
		sumx = self.fix_x(-0.1*self.last_read[0] - 0.086*self.last_read[2])
		sumy = self.fix_y(-0.1*self.last_read[1] + 0.05*self.last_read[2])
		widget.window.draw_line(xgc,self.fix_x(0),self.fix_y(0),sumx,sumy)
		

	def build_gui(self):
		menu = self.create_menu()
		menu.show_all()
		self.set_app_menu(menu)

		self.box_v = gtk.VBox(False,0)
		self.add(self.box_v)

		self.box_h = gtk.HBox(False,0)
		self.box_v.pack_start(self.box_h)

		self.x_label = gtk.Label("X: NaN")
		self.y_label = gtk.Label("Y: NaN")
		self.z_label = gtk.Label("Z: NaN")
		self.box_h.pack_start(self.x_label,True, True, 0)
		self.box_h.pack_start(self.z_label,True, True, 0)
		self.box_h.pack_start(self.y_label,True, True, 0)
		self.x_label.show()
		self.y_label.show()
		self.z_label.show()
		
		self.canvas = gtk.DrawingArea()
		self.canvas.set_size_request(300, 200)
		self.box_v.pack_start(self.canvas)
		self.canvas.connect("expose-event",self.expose_handler)
		self.canvas.show()

		self.box_state_h = gtk.HBox(False,0)
		self.box_v.pack_start(self.box_state_h)

		self.innerVbox = gtk.VBox(False,0)
		self.box_state_h.pack_start(self.innerVbox,True,True,0)


		self.state_label = gtk.Label("Status")
		self.innerVbox.pack_start(self.state_label,True, True, 0) #expand, fill, padding
		self.state_label.show()

		self.smooth_box = gtk.CheckButton(label="Smoothing")
		self.smooth_box.connect("toggled", self.on_smooth_chng)
		self.innerVbox.pack_start(self.smooth_box,True, True, 0) #expand, fill, padding
		self.smooth_box.show()
		
		self.innerVbox.show()

		self.time_scale = gtk.HScale()
		self.time_scale.set_range(1,41)
		self.time_scale.set_value(20)
		self.time_scale.set_sensitive(False)
		self.time_scale.connect("change-value",self.update_period)
		self.box_state_h.pack_start(self.time_scale,True, True, 0) #expand, fill, padding
		self.time_scale.show()

		self.box_state_h.show()

		self.box_end_h = gtk.HBox(False,0)
		self.box_v.pack_start(self.box_end_h)

		self.left_bt = gtk.Button("Left")
		self.left_bt.connect("clicked", self.on_left_bt)
		self.box_end_h.pack_start(self.left_bt,True, True, 0)
		self.left_bt.show()

		self.right_bt = gtk.Button("Right")
		self.right_bt.connect("clicked", self.on_right_bt)
		self.box_end_h.pack_start(self.right_bt,True, True, 0)
		self.right_bt.show()

		self.box_end_h.show()
		self.box_h.show()
		self.box_v.show()
		self.setup_gui()

		#hildon.hildon_banner_show_information(self, '', "Information not accurate")

	def acquisition(self):
		if self.appState == AppState.LIVE:
			return acc_read()
		elif self.appState == AppState.RECORDING:
			rd = acc_read()
			self.data_storage.append(rd)
			return rd
		elif self.appState == AppState.PLAYBACK:
			if len(self.data_storage) > self.data_ptr:
				ret = self.data_storage[self.data_ptr]
				self.data_ptr += 1
				return ret
			else:
				self.data_ptr = 0
				self.paused = True
				return [0,0,0]

	def setup_gui(self):
		if self.appState == AppState.LIVE:
			self.right_bt.set_label("Open")
			self.left_bt.set_label("Record")
			self.state_label.set_text("Live Playback")
		elif self.appState == AppState.RECORDING:
			self.right_bt.set_label("Clear")
			self.left_bt.set_label("Stop")
			self.state_label.set_text("Recording..")
		elif self.appState == AppState.PLAYBACK:
			self.right_bt.set_label("Play/Pause")
			self.left_bt.set_label("Close")
			self.state_label.set_text("Time Scale(Hz): ") # + self.tstamp)
		while gtk.events_pending():
			gtk.main_iteration(False)

	def on_left_bt(self, *args):
		if self.appState == AppState.LIVE:
			self.appState = AppState.RECORDING
		elif self.appState == AppState.RECORDING:
			filename = self.get_save_fn()
			if filename != "":
				fw = accFileIF.AccFileIF()
				fw.Save(self.data_storage,filename)
			##going back to LIVE even if user aborts the saving process
			self.appState = AppState.LIVE
		elif self.appState == AppState.PLAYBACK:
			self.data_storage = []
			self.data_ptr = 0
			self.period = 1.0/20
			self.time_scale.set_sensitive(False)
			self.appState = AppState.LIVE
		self.setup_gui()
	
	def on_right_bt(self, *args):
		if self.appState == AppState.LIVE:
			filename = self.get_load_fn()
			if filename != "":
				fw = accFileIF.AccFileIF()
				self.tstamp,self.data_storage = fw.Load(filename)
				self.time_scale.set_value(20)
				self.time_scale.set_sensitive(True)
				self.data_ptr = 0
				self.appState = AppState.PLAYBACK
			##only going to PLAYBACK if open process was successful
		elif self.appState == AppState.RECORDING:
			self.data_storage = []
			#print "Clear"
		elif self.appState == AppState.PLAYBACK:
			self.paused = not self.paused
			#print "Rewind"
		self.setup_gui()

	def update_period(self, *args):
		if self.appState == AppState.PLAYBACK:
			self.period = 1/self.time_scale.get_value()

	def on_smooth_chng(self, *args):
		self.smoothFlag = self.smooth_box.get_active()

	####################################################
	################ top menu creation #################
	####################################################
	def create_menu(self):
		menu = hildon.AppMenu()
		self.ab_menu_button = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		label = 'About'
		self.ab_menu_button.set_label(label)
		self.ab_menu_button.connect("clicked", self.on_about)
		menu.append(self.ab_menu_button)

		self.quit_menu_button = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
		label = 'Quit'
		self.quit_menu_button.set_label(label)
		self.quit_menu_button.connect("clicked", self.on_close)
		menu.append(self.quit_menu_button)
		return menu

	#####################################################
	############## Normal Callbacks #####################
	#####################################################
	def on_close(self, *args):
		self.running = False

	def on_about(self, *args):
		dialog = gtk.AboutDialog()
		dialog.set_name("accDisplay")
		
		dialog.set_comments('Accelerometers View/Dataloggin')
		dialog.set_version("0.1")
		dialog.set_copyright("By Clovis Peruchi Scotti (aka cpscotti)")
		dialog.set_website("www.das.ufsc.br/~scotti")
		dialog.connect ("response", lambda d, r: d.destroy())
		dialog.show()


	###################################################
	############### file choser dialogs.. #############
	###################################################
	def get_save_fn(self):
		fc = gobject.new(hildon.FileChooserDialog, action=gtk.FILE_CHOOSER_ACTION_SAVE)
		#fc = hildon.FileChooserDialog(self, gtk.FILE_CHOOSER_ACTION_SAVE,fsm)# bug #6254
		
		fc.set_show_hidden(False)
		fc.set_do_overwrite_confirmation(True)
		
		fc.set_property('autonaming',False)
		fc.set_property('show-files',True)
		#fc.set_current_folder(os.path.expanduser("~"))
		fc.set_current_name("AccReading.xml")
		#fc.set_extension('xml')
		filename = ""
		if fc.run() == gtk.RESPONSE_OK:
			filename = fc.get_filename()
		fc.destroy()
		return filename

	def get_load_fn(self):
		fc = gobject.new(hildon.FileChooserDialog, action=gtk.FILE_CHOOSER_ACTION_OPEN)
		fc.set_property('show-files',True)
		fc.set_current_name("AccReading.xml")
		filename = ""
		if fc.run()==gtk.RESPONSE_OK:
			filename = fc.get_filename()
		fc.destroy()
		return filename

###################################################
############## Accel reading function #############
###################################################
def acc_read():
	f = open("/sys/class/i2c-adapter/i2c-3/3-001d/coord", 'r' )
	coords = [int(w) for w in f.readline().split()]
	f.close()
	return coords

if __name__ == "__main__":
	fend = accDisplay()
