/*
 * Copyright (C) 2009, 2010 Floriano Scioscia.
 *
 * This file is part of 100 Boxes.
 *
 * Conboy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Conboy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with 100 Boxes. If not, see <http://www.gnu.org/licenses/>.
 */

#include <mce/dbus-names.h>
#include <mce/mode-names.h>
#include "orientation.h"
#include "100boxes.h"

static void set_orientation(const gchar* orientation, gpointer app_data) {
	struct game_data* d = (struct game_data *)app_data;
	HildonPortraitFlags flags;

	/* If we disabled the accelerometer, then we don't want to react on it if
	 * it was turned on by another application.
	 *
	 * E.g. when a dialog is open we want to disable auto rotation and if there
	 * is another application that turned on the accelerometer we still don't want
	 * the input.
	 */
	if (d->accelerometer == FALSE) {
		return;
	}
	g_debug("In function set_orientation");
	if (strcmp(orientation, MCE_ORIENTATION_LANDSCAPE) == 0) {
		flags = HILDON_PORTRAIT_MODE_SUPPORT;
		d->portrait = FALSE;
		gtk_widget_hide_all(GTK_WIDGET(d->v_container));
		gtk_container_remove(GTK_CONTAINER(d->window), GTK_WIDGET(d->v_container));
		gtk_container_remove(GTK_CONTAINER(d->v_container), GTK_WIDGET(d->box_table));
		gtk_container_remove(GTK_CONTAINER(d->v_container), GTK_WIDGET(d->panel));
		gtk_widget_set_size_request(GTK_WIDGET(d->box_table), HEIGHT, -1);
		gtk_widget_set_size_request(GTK_WIDGET(d->panel), WIDTH-HEIGHT-PADDING, -1);
		gtk_container_add(GTK_CONTAINER(d->h_container), GTK_WIDGET(d->box_table));
		gtk_container_add(GTK_CONTAINER(d->h_container), GTK_WIDGET(d->panel));
		gtk_container_add(GTK_CONTAINER(d->window), GTK_WIDGET(d->h_container));
	} else if (strcmp(orientation, MCE_ORIENTATION_PORTRAIT) == 0) {
		flags = HILDON_PORTRAIT_MODE_REQUEST;
		d->portrait = TRUE;
		gtk_widget_hide_all(GTK_WIDGET(d->h_container));
		gtk_container_remove(GTK_CONTAINER(d->window), GTK_WIDGET(d->h_container));
		gtk_container_remove(GTK_CONTAINER(d->h_container), GTK_WIDGET(d->box_table));
		gtk_container_remove(GTK_CONTAINER(d->h_container), GTK_WIDGET(d->panel));
		gtk_widget_set_size_request(GTK_WIDGET(d->box_table), HEIGHT, HEIGHT-PADDING);
		gtk_widget_set_size_request(GTK_WIDGET(d->panel), HEIGHT, -1);
		gtk_container_add(GTK_CONTAINER(d->v_container), GTK_WIDGET(d->panel));
		gtk_container_add(GTK_CONTAINER(d->v_container), GTK_WIDGET(d->box_table));
		gtk_container_add(GTK_CONTAINER(d->window), GTK_WIDGET(d->v_container));
	} else {
		g_printerr("ERROR: Orientation must be 'landscape' or 'portrait', not '%s'.\n", orientation);
		return;
	}

	g_debug("INFO: Setting to orientation: %s\n", orientation);
	gtk_widget_show_all(GTK_WIDGET(d->window));
	hildon_gtk_window_set_portrait_flags(GTK_WINDOW(d->window), flags);
}

static DBusHandlerResult dbus_handle_mce_message(DBusConnection *con, DBusMessage *msg, gpointer data) {
	DBusMessageIter iter;
	gchar *mode = NULL;

	g_debug("In function dbus_handle_mce_message");
	if (dbus_message_is_signal(msg, MCE_SIGNAL_IF, MCE_DEVICE_ORIENTATION_SIG)) {
		if (dbus_message_iter_init(msg, &iter)) {
			dbus_message_iter_get_basic(&iter, &mode);
			set_orientation(mode, data);
			g_debug("Current orientation: %s", mode);
		}
	}
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

gboolean orientation_is_portrait(gpointer app_data) {
	struct game_data* d = (struct game_data *)app_data;
	osso_rpc_t ret;
	gboolean result = FALSE;

	g_debug("In orientation_is_portrait\n");
	osso_return_t outcome = osso_rpc_run_system(d->osso_context, MCE_SERVICE, MCE_REQUEST_PATH, MCE_REQUEST_IF, MCE_DEVICE_ORIENTATION_GET, &ret, DBUS_TYPE_INVALID);
	if (outcome == OSSO_OK) {
		g_debug("Current orientation: %s\n", ret.value.s);
		if (strcmp(ret.value.s, MCE_ORIENTATION_PORTRAIT) == 0) {
			result = TRUE;
		}
		osso_rpc_free_val(&ret);
	} else {
		g_printerr("ERROR: Call to DBus failed: \n");
	}
	return result;
}

void orientation_enable_accelerometer(gpointer app_data) {
	struct game_data* d = (struct game_data *)app_data;

	if (d->accelerometer == TRUE) {
		return;
	}
	if (osso_rpc_run_system(d->osso_context, MCE_SERVICE, MCE_REQUEST_PATH, MCE_REQUEST_IF, MCE_ACCELEROMETER_ENABLE_REQ, NULL, DBUS_TYPE_INVALID) == OSSO_OK) {
		g_printerr("INFO: Accelerometers enabled\n");
		d->accelerometer = TRUE;
	} else {
		g_printerr("WARN: Cannot enable accelerometers\n");
	}
}

void orientation_disable_accelerometer(gpointer app_data) {
	struct game_data* d = (struct game_data *)app_data;
	if (d->accelerometer == FALSE) {
		return;
	}

	if (osso_rpc_run_system(d->osso_context, MCE_SERVICE, MCE_REQUEST_PATH, MCE_REQUEST_IF, MCE_ACCELEROMETER_DISABLE_REQ, NULL, DBUS_TYPE_INVALID) == OSSO_OK) {
		g_debug("INFO: Accelerometers disabled\n");
		d->accelerometer = FALSE;
	} else {
		g_printerr("WARN: Cannot disable accelerometers\n");
	}
}

void orientation_init(gpointer app_data) {
	struct game_data* d = (struct game_data *)app_data;
	DBusError error;

	/* Get the DBus connection */
	DBusConnection *con = osso_get_sys_dbus_connection(d->osso_context);
	if (con) {
		/* Enable accelerometer */
		orientation_enable_accelerometer(app_data);
		/* Get current orientation from DBus and save into app_data */
		d->portrait = orientation_is_portrait(app_data);
		/* Add a DBus matching rule */
		dbus_error_init (&error); 
		dbus_bus_add_match(con, MCE_MATCH_RULE, &error);
		if (dbus_error_is_set(&error)){
			g_debug("WARN: dbus_bus_add_match failed: %s\n", error.message);
			dbus_error_free(&error);
		}
		/* Add the callback, which should be called, once the device is rotated */
		if (!dbus_connection_add_filter(con, (DBusHandleMessageFunction)dbus_handle_mce_message, app_data, NULL)) {
			g_debug("WARN: Error dbus_connection_add_filter failed\n");
		}
	}
}
