/**
 * Copyright (C) 2009 Floriano Scioscia.
 *
 * This file is part of 100 Boxes.
 *
 * 100 Boxes is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * 100 Boxes is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with 100 Boxes. If not, see <http://www.gnu.org/licenses/>.
 */
#include "sound.h"

static void eos_message_received(GstBus *bus, GstMessage *message, GstElement *pipeline);
static void error_message_received(GstBus *bus, GstMessage *message, GstElement *pipeline);
static void free_pipeline(GstElement *pipeline);

static void error_message_received(GstBus *bus, GstMessage *message, GstElement *pipeline) {
	GError *err;
	gchar *debug;
	gst_message_parse_error(message, &err, &debug);
	g_printerr("Error: %s", debug);
	/* stop playback and free pipeline */
	gst_element_set_state(pipeline, GST_STATE_NULL);
	free_pipeline(pipeline);
}

static void eos_message_received(GstBus *bus, GstMessage *message, GstElement *pipeline) {
	/* stop playback and free pipeline */
	gst_element_set_state(pipeline, GST_STATE_NULL);
	g_debug("PLAYING FINISHED");
	free_pipeline(pipeline);
}

static void free_pipeline(GstElement *pipeline) {
	gst_object_unref(GST_OBJECT(pipeline));
	pipeline = NULL;
}

void sound_init(int *argc, char **argv[]) {
	gst_init(argc, argv);
}

int play_sound(const gchar *path) {
	GstElement *pipeline;
	GstElement *filesrc;
	GstBus *bus;
	GError *error = NULL;

	/* setup pipeline and configure elements */
	pipeline = gst_parse_launch ("filesrc name=my_filesrc ! decodebin ! audioconvert ! audioresample ! autoaudiosink", &error);
	if (!pipeline) {
		g_printerr("Parse error: %s\n", error->message);
		return ERR_PIPELINE_SETUP_FAILED;
	}
	filesrc = gst_bin_get_by_name(GST_BIN(pipeline), "my_filesrc");
	if (!filesrc) {
		g_printerr("Parse error: no filesrc");
		free_pipeline(pipeline);
		return ERR_PIPELINE_SETUP_FAILED;
	}
	g_object_set(G_OBJECT(filesrc), "location", path, NULL);

	/* setup message handling */
	bus = gst_pipeline_get_bus(GST_PIPELINE (pipeline));
	gst_bus_add_signal_watch_full(bus, G_PRIORITY_HIGH);
	g_signal_connect(bus, "message::eos", (GCallback)eos_message_received, pipeline);
	g_signal_connect(bus, "message::error", (GCallback)error_message_received, pipeline);
	gst_object_unref(GST_OBJECT(bus));
	
	/* start playback */
	gst_element_set_state(pipeline, GST_STATE_PLAYING);
	g_debug("PLAYING STARTED");
	return ERR_NONE;	
}

