
#include <QFrame>
#include <QKeyEvent>

#include "BlogView.h"
#include "writepageview.h"
#include "WPNetworkEngine.h"
#include "wpconfirmdialog.h"
#include "utils.h"

static const char* KLocalDraft = "Local draft";

WritePageView::WritePageView(BlogView *parent) : QFrame(parent), iParent(parent)
{
    ui.setupUi(this);
    ui.scrollArea->setWidget(ui.layoutWidget);
    ui.layoutWidget->setContentsMargins(5, 5, 5, 5);
    ui.titleEditor->SetExplanation(tr("Page title"));
    ui.textBrowser->setExplanationText(tr("Tap to begin writing"));

    // Prepare Actions
    QAction* leftSoftKey = new QAction(tr("Save"), this);
    connect(leftSoftKey, SIGNAL(triggered()), this, SLOT(SavePage()));
    leftSoftKey->setSoftKeyRole(QAction::PositiveSoftKey);
    QAction* rightSoftKey = new QAction(tr("Cancel"), this);
    connect(rightSoftKey, SIGNAL(triggered()), this, SLOT(Cancel()));
    rightSoftKey->setSoftKeyRole(QAction::NegativeSoftKey);

    connect(ui.addPhotosButton, SIGNAL(clicked(bool)), this, SLOT(AddPhotosClicked(bool)));
    connect(ui.addVideosButton, SIGNAL(clicked(bool)), this, SLOT(AddVideosClicked(bool)));

    ui.scrollArea->DisableScrollAtWidget(ui.statusEditor->view());

    // Finally add the data for WPCustomViewBasicData
    //
    iTitleText = tr("Edit Page");
    iTabText = tr("Edit");
    iHappyAction = leftSoftKey;
    iSadAction = rightSoftKey;
}

WritePageView::~WritePageView()
{
}

void WritePageView::ResetView (WPPage page)
{
    iAddedImages.clear();
    iAddedVideos .clear();

    // basic items
    ui.titleEditor->setText(page->iTitle);
    ui.textBrowser->reset();
    ui.textBrowser->setText(page->iDescription);
    ui.dateTimeEditor->setDateTime(page->GetLocalTime());
    ui.addPhotosButton->setText(tr("Add photos..."));
    ui.addVideosButton->setText(tr("Add videos..."));

    // statuses
    ui.statusEditor->clear();

    QStringList keyList, valueList;
    QMap<QString, QString>::const_iterator i;

    for (i = iBlog->iPageStatuses.constBegin(); i != iBlog->iPageStatuses.constEnd(); i++)
    {
        keyList.append(i.key());
        valueList.append(i.value());
    }

    ui.statusEditor->addItems(valueList);

    if (page->IsLocal())
    {
        ui.statusEditor->addItem(KLocalDraft);
        ui.statusEditor->setCurrentIndex(ui.statusEditor->count() - 1);

    } 
    else 
    {
        for (int i = 0; i < keyList.count(); ++i)
        {
            if (page->iStatus.compare(keyList.at(i)) == 0)
                ui.statusEditor->setCurrentIndex(i);
        }
    }
    ui.titleEditor->setFocus();
    ui.textBrowser->setFocus(); // maemo5 focus workaround
}

void WritePageView::NewPage(WPDataBlog& aBlog)
{
    iBlog = &aBlog;
    iPage = new WPDataPage(aBlog);

    ResetView(iPage);
}

void WritePageView::EditPage(WPDataBlog& aBlog, WPPage aPage)
{
    iBlog = &aBlog;
    iPage = aPage;

    ResetView(iPage);
}

void WritePageView::SavePage()
{
    if (ui.statusEditor->currentText() == QLatin1String(KLocalDraft)) {
        if (iAddedImages.count() > 0)
            emit ShowErrorNote(tr("Images are not saved in local drafts. Please re-add images when publishing the post."));

        // just update the local storage
        iPage->UpdateLocal(
                ui.titleEditor->text(),         // title
                ui.textBrowser->text(),         // description
                ui.dateTimeEditor->dateTime()   // local dateCreated
        );
    }
    else
    {
        // save on network
        iPage->UpdateStatus(
                ui.titleEditor->text(),         // title
                ui.textBrowser->text(),         // description
                ui.dateTimeEditor->dateTime(),  // local dateCreated
                iBlog->iPageStatuses.key(ui.statusEditor->currentText()),   // status
                iAddedImages + iAddedVideos
        );
    }
 
    // forget page
    iPage = NULL;
   
    // close
    // XXX: more generic
    iParent->ShowTabbedListView();
#ifdef Q_WS_MAEMO_5
    close();
#endif
}

void WritePageView::AddPhotosClicked(bool aDummy)
{
    Q_UNUSED(aDummy)
    iParent->ShowImageList(iAddedImages, false);
}

void WritePageView::AddVideosClicked(bool aDummy)
{
    Q_UNUSED(aDummy)
    iParent->ShowVideoList(iAddedImages, false);
}

void WritePageView::ImagesAdded(const QStringList& aFilenames)
{
    iAddedImages = aFilenames;

    QString labelText;
    if (iAddedImages.size() == 0)
        labelText = tr("Add photos") + "...";
    else if (iAddedImages.size() == 1)
        labelText = tr("Adding 1 new photo");
    else
        labelText = tr("Adding %1 new photos").arg(iAddedImages.size());

    ui.addPhotosButton->setText(labelText);
}

void WritePageView::VideosAdded(const QStringList& aFilenames)
{
    iAddedVideos = aFilenames;

    QString labelText;
    if (iAddedVideos.size() == 0)
        labelText = tr("Add videos...");
    else if (iAddedVideos.size() == 1)
        labelText = tr("Adding 1 new video");
    else
        labelText = tr("Adding %1 new videos").arg(iAddedVideos.size());

    ui.addVideosButton->setText(labelText);
}

void WritePageView::setFocus()
{
    ui.scrollArea->ScrollToTop();
    ui.scrollArea->setFocus();
    
    if(Utils::smallScreen()) {
        ui.titleEditor->setFocus();
    }
}

void WritePageView::Cancel()
{
    if(ui.titleEditor->text().length() || ui.textBrowser->text().length())
    {
        WP_CONFIRM_AND_RETURN_ON_FALSE(this);
    }

    iParent->ShowTabbedListView();
}

