#include "wpbloggraphicslistitem.h"
#include "WPData.h"
#include "utils.h"

WPBlogGraphicsListItem::Geometry WPBlogGraphicsListItem::iGeometry;

WPBlogGraphicsListItem::WPBlogGraphicsListItem(const WPDataBlog& blog, BusyIndicator& busyIndicator, ThemeData& themeData)
    : WPGraphicsListItem(busyIndicator, themeData)
    , iBlog(blog)
    , iTitle(Utils::toDisplayHtml(blog.iBlogName))
{
    int draftPosts = 0;
    int unapprovedComments = 0;
    QDateTime latestActivity;
    QDateTime comparisonDate;

    for (QListIterator<WPPost> i = blog.GetPosts(); i.hasNext(); ) {
        const WPPost post = i.next();

        if (post->GetLocalTime() > latestActivity)
            latestActivity = post->GetLocalTime();

        if (post->IsDraft())
            draftPosts++;
    }

    for (QListIterator<WPComment> j = blog.GetComments(); j.hasNext(); ) {
        WPComment comment = j.next();

        if (comment->GetLocalTime() > latestActivity)
            latestActivity = comment->GetLocalTime();

        if (comment->IsUnapproved())
            unapprovedComments++;
    }

    if (unapprovedComments == 1)
        iStatus += tr("1 unapproved comment");
    else if (unapprovedComments > 1)
        iStatus += QString::number(unapprovedComments) + tr(" unapproved comments");

    if (draftPosts > 0)
    {
        if (!iStatus.isEmpty())
            iStatus += ", ";
        if (draftPosts == 1)
            iStatus += tr("1 draft");
        else
            iStatus += QString::number(draftPosts) + tr(" drafts");
    }

    iActivity = tr("Latest activity: ") + WPNetworkEngine::LocalizedTime(latestActivity);;
}

WPBlogGraphicsListItem::~WPBlogGraphicsListItem()
{

}

void WPBlogGraphicsListItem::paintContent(QPainter* painter, const QStyleOptionGraphicsItem* option, QWidget* widget)
{
    Q_UNUSED(option); Q_UNUSED(widget);
    painter->setPen(Qt::black);

    painter->setFont(*iThemeData.iListContentFont);
    painter->drawText(iGeometry.iTitle, iTitle);

    painter->setFont(*iThemeData.iListInfoFont);
    painter->drawText(iGeometry.iStatus, iStatus);
    painter->drawText(iGeometry.iActivity, iActivity);
}

bool WPBlogGraphicsListItem::IsBusy()
{
    return false;
}

bool WPBlogGraphicsListItem::HasError()
{
    return false;
}

void WPBlogGraphicsListItem::handleGeomeryChange()
{
    iGeometry.update(size(), iThemeData);
    
    calculateElidedTexts();

    setMinimumHeight(iGeometry.height());
}

const WPDataBlog& WPBlogGraphicsListItem::getBlog() const
{
    return iBlog;
}

void WPBlogGraphicsListItem::calculateElidedTexts()
{
    QFontMetrics titleFm(*iThemeData.iListContentFont);
    iTitle = titleFm.elidedText(Utils::toDisplayHtml(iBlog.iBlogName),
                                Qt::ElideRight, iGeometry.iTitle.width());
}

void WPBlogGraphicsListItem::Geometry::update (const QSizeF &size, const ThemeData &theme)
{
    if (iCachedSize == size)
        // fresh
        return;

    else
        iCachedSize = size;

    // XXX: recalc boundingRect by hand
    int itemPadding = theme.iListItemInternalPadding;
    int hMargin = theme.iListItemContentHMargin;
    int vMargin = theme.iListItemContentVMargin;
    int width = size.width() - 2 * hMargin;
    
    // Font metrics
    QFontMetrics titleFont (*theme.iListContentFont);
    QFontMetrics infoFont (*theme.iListInfoFont);

    iTitle.setRect(hMargin, vMargin, width, titleFont.height());
    iStatus.setRect(hMargin, iTitle.bottom() + itemPadding, width, infoFont.height());
    iActivity.setRect(hMargin, iStatus.bottom() + 2 * itemPadding, width, infoFont.height());

    iHeight = iActivity.bottom() + 2 * itemPadding + hMargin;
}

