/*
 * statsvisits.cpp
 *
 *  Created on: 19.3.2010
 *      Author: aahv
 */

#include "statsvisits.h"

#include <QList>
#include <QPair>
#include <QDate>
#include <QMap>
#include <QPushButton>
#include <QButtonGroup>
#include <QGridLayout>
#include <QLabel>

#include <QDebug>

StatsVisits::StatsVisits(QWidget *parent) :
        QWidget(parent)
{
    label = new QLabel(this);
    label->setWordWrap(true);

    dayButton = new QPushButton(tr("Days"), this);
    dayButton->setCheckable(true);
    dayButton->setChecked(true);
    weekButton = new QPushButton(tr("Weeks"), this);
    weekButton->setCheckable(true);
    monthButton = new QPushButton(tr("Months"), this);
    monthButton->setCheckable(true);

    viewButtonGroup = new QButtonGroup(this);
    viewButtonGroup->addButton(dayButton, 0);
    viewButtonGroup->addButton(weekButton, 1);
    viewButtonGroup->addButton(monthButton, 2);

    plot = new LinePlot(this);
    plot->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);

    dailyData = new StatsDailyData();
    weeklyData = new StatsWeeklyData();
    monthlyData = new StatsMonthlyData();

    connect(viewButtonGroup, SIGNAL(buttonClicked(int)), this, SLOT(buttonClicked(int)));

    updateLayout();
}

StatsVisits::~StatsVisits()
{
    delete dailyData;
    delete weeklyData;
    delete monthlyData;
}

void StatsVisits::resizeEvent(QResizeEvent *)
{
    updateLayout();
}

void StatsVisits::updateLayout()
{
    if (layout())
        delete layout();

    QGridLayout *layout = new QGridLayout(this);
    layout->setMargin(6);
    if (width() < height()) {
        layout->addWidget(label, 0, 0, 1, 3);
        layout->addWidget(dayButton, 1, 0);
        layout->addWidget(weekButton, 1, 1);
        layout->addWidget(monthButton, 1, 2);
        layout->addWidget(plot, 2, 0, 1, 3);
    } else {
        QVBoxLayout *buttonLayout = new QVBoxLayout(this);
        buttonLayout->addWidget(dayButton);
        buttonLayout->addWidget(weekButton);
        buttonLayout->addWidget(monthButton);
        buttonLayout->addStretch();

        layout->addWidget(label, 0, 0, 1, 2);
        layout->addWidget(plot, 1, 0);
        layout->addLayout(buttonLayout, 1, 1);
    }

    setLayout(layout);
}

void StatsVisits::buttonClicked(int id) {
    switch (id) {
    case 0:
        switchPlotData(dailyData);
        break;

    case 1:
        switchPlotData(weeklyData);
        break;

    case 2:
        switchPlotData(monthlyData);
        break;

    default:
        qWarning() << "Invalid button id:" << id;
    }
}

void StatsVisits::updateStats(const WPDataStats &stats) {
    label->setText(tr("Visits (%1 total visits in last 365 days)").arg(stats.iTotalVisits));

    dailyData->reload(stats.iVisits);
    weeklyData->reload(stats.iVisits);
    monthlyData->reload(stats.iVisits);

    buttonClicked(viewButtonGroup->checkedId());
}

void StatsVisits::switchPlotData(LinePlotData *data) {
    if (data->getDataCount() == 0)
        return;

    plot->setData(data);
    plot->setXAxisMajorTickFrequency((data == dailyData? 2: 1));
    plot->repaint();
}
