#include "LocationButton.h"

#include <QFontMetrics>
#include <QLayout>
#include <QTimer>
#include <QIcon>
#include <QDateTime>

static const int SPINNER_COUNT = 8;

#if defined(HAVE_QTM_LOCATION)
#include <QGeoCoordinate>

using QtMobility::QGeoCoordinate;
using QtMobility::QGeoPositionInfo;
#endif // defined(HAVE_QTM_LOCATION)

LocationButton::LocationButton(QWidget * parent)
    : QPushButton(parent)
    , supported(true)
{
    timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(timerTimeout()));

    icons = new QIcon[SPINNER_COUNT];
    for (int i = 0; i < SPINNER_COUNT; ++i)
        icons[i] = QIcon(QString(":/images/spinner%1.png").arg(i + 1));

    resetPosition();
}

LocationButton::~LocationButton()
{
    delete[] icons;
}

void LocationButton::resetPosition()
{
    supported = true;
#if defined(HAVE_QTM_LOCATION)
    position = QGeoPositionInfo();
#endif // defined(HAVE_QTM_LOCATION)
    updateText();
}

void LocationButton::setBusy(bool busy)
{
    if (busy) {
        iconIndex = 0;
        timer->start(125);
    } else {
        timer->stop();
        setIcon(QIcon());
    }
}

void LocationButton::setNotSupported()
{
    supported = false;
    updateText();
}

void LocationButton::setFormattedAddress(const QString &address)
{
    formattedAddress = address;
    updateText();
}

#if defined(HAVE_QTM_LOCATION)
void LocationButton::setPosition(QtMobility::QGeoPositionInfo const & newPosition)
{
    supported = true;
    position = newPosition;
    updateText();
}
#endif // defined(HAVE_QTM_LOCATION)

void LocationButton::resizeEvent(QResizeEvent* event)
{
    QPushButton::resizeEvent(event);
    updateText();
}

void LocationButton::updateText()
{
    if(!supported) {
        setEnabled(false);
        setText(tr("Adding location not supported"));
        return;
    }
    
#if defined(HAVE_QTM_LOCATION)
    setEnabled(true);
    if(!position.isValid()) {
        setText(tr("Add location..."));
        return;
    }
    
    int maxWidth = getTextWidth();
    QFontMetrics fm = fontMetrics();
    QGeoCoordinate coordinate = position.coordinate();
    QString coordStr = coordinate.toString(QGeoCoordinate::DegreesMinutesSecondsWithHemisphere);
    if(fm.width(coordStr) > maxWidth) {
        coordStr = coordinate.toString(QGeoCoordinate::DegreesMinutesWithHemisphere);
        if(fm.width(coordStr) > maxWidth) {
            coordStr = coordinate.toString(QGeoCoordinate::DegreesWithHemisphere);
        }
    }

    if (!formattedAddress.isEmpty()) {
        coordStr = "Near " + formattedAddress;
        if (fm.width(coordStr) > maxWidth)
            coordStr = fm.elidedText(coordStr, Qt::ElideRight, maxWidth);
    }
    
    /*: %1 is a location coordinate, which has to be on it's own line
     * e.g. 57 18' 22.5" N, 4 27' 32.7" W
     * This REALLY needs to be on it's own line, or the whole view might break!
     */
    setText(tr("Remove location:\n%1\n(%2)").arg(coordStr, formatTimeSince(position.timestamp())));
#endif // defined(HAVE_QTM_LOCATION)
}

int LocationButton::getTextWidth()
{
    /* XXX how the heck do we really find out the available text width?
     * Worst case scenario is that the text overflows making the 
     */
    
    // assume max one em of padding
    QFontMetrics fm = fontMetrics();
    int em = fm.height();
    return contentsRect().width() - 2 * em - iconSize().width();
}

void LocationButton::timerTimeout()
{
    iconIndex = (iconIndex + 1) % SPINNER_COUNT;
    setIcon(icons[iconIndex]);
    if (iconIndex == 0)
        updateText();
}

QString LocationButton::formatTimeSince(const QDateTime &t)
{
    int secs = t.secsTo(QDateTime::currentDateTime());
    if (secs < 60)
        return QString("%1 second%2 ago").arg(secs).arg(secs == 1? "": "s");
    int mins = secs / 60;
    if (mins < 60)
        return QString("%1 minute%2 ago").arg(mins).arg(mins == 1? "": "s");
    int hours = mins / 60;
    if (hours < 24)
        return QString("%1 hour%2 ago").arg(hours).arg(hours == 1? "": "s");
    int days = hours / 24;
    if (days < 7)
        return QString("%1 day%2 ago").arg(days).arg(days == 1? "": "s");
    int weeks = days / 7;
    if (weeks < 5)
        return QString("%1 week%2 ago").arg(weeks).arg(weeks == 1? "": "s");
    int months = days / 30;
    if (months < 12)
        return QString("%1 month%2 ago").arg(months).arg(months == 1? "": "s");
    int years = months / 12;
    return QString("%1 year%2 ago").arg(years).arg(years == 1? "": "s");
}
