/*
 * CheckableListItem.cpp
 *
 *  Created on: 8.1.2010
 *      Author: apoi
 */

#include <QKeyEvent>
#include <QImageReader>
#include <QTimer>

#include "CheckableListItem.h"
#include "ui_CheckableListItem.h"
#include "MDragWidget.h"
#include "ImageThumbnailThread.h"

const char* KChecked = ":/images/item-checked.png";
const char* KUnchecked = ":/images/item-unchecked.png";
const char* KHierarchy = ":/images/hierarchy-indicator.png";
const char* KShowThumbnail = ":/images/thumbnail-show.png";
const char* KWaitThumbnail = ":/images/thumbnail-wait.png";

CheckableListItem::CheckableListItem(const QString& aItemId, bool aChecked,
        ImageThumbnailThread & aThumbnailThread, MDragWidget& aParent)
    : QFrame(&aParent)
    , ui(new Ui::CheckableListItem)
    , iItemId(aItemId)
    , iChecked(aChecked)
    , iParent(aParent)
    , iThumbnailCreated(false)
    , iThumbnailThread(aThumbnailThread)
{
    static const QPixmap showThumbnail(KShowThumbnail);
    ui->setupUi(this);

    setObjectName("imageListItem");
    ui->indentLabel->setEnabled(false);
    ui->indentLabel->hide();
    // Set an arrow that shows the thumbnail
    ui->thumbLabel->setPixmap(showThumbnail);
    
    UpdateCheckState();
}

CheckableListItem::~CheckableListItem()
{
    delete ui;
}

bool CheckableListItem::IsChecked() const
{
    return iChecked;
}

void CheckableListItem::SetCheckState(bool aState)
{
    iChecked = aState;
    UpdateCheckState();
}

void CheckableListItem::UpdateCheckState()
{
    // Style sheets according to the selected state and item type
    if (iChecked) {
        // XXX: this sucks, should be a QSS selector
        setStyleSheet("background-color: #B2D9FB;");

    } else {
        setStyleSheet("");
    }
}

void CheckableListItem::mouseReleaseEvent(QMouseEvent *e)
{
    static const QPixmap waitThumbnail(KWaitThumbnail);

    // XXX: F*** MDragWidget S***
    if (hasFocus() && !iParent.MDragWidgetIsDragging())
    {
        iChecked = !iChecked;
        UpdateCheckState();
    }
    QFrame::mouseReleaseEvent(e);
}

void CheckableListItem::keyPressEvent(QKeyEvent *e)
{
    if (e->key() == Qt::Key_Select)
    {
        iChecked = !iChecked;
        UpdateCheckState();
        e->accept();
    }
    else if (e->key() == Qt::Key_Right && !iThumbnailCreated)
    {
        CreateThumbnail();
        e->accept();
    }
    else
    {
        QFrame::keyPressEvent(e);
    }
}

void CheckableListItem::changeEvent(QEvent *e)
{
    QFrame::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void CheckableListItem::ThumbnailCreated(void * requester, QImage const aImage)
{
    if(requester == this) {
        CreateThumbnail(aImage);
        disconnect(&iThumbnailThread, SIGNAL(ThumbnailCreated(void *, QImage const)),
                   this, SLOT(ThumbnailCreated(void *, QImage const)));
    }
}

const QString& CheckableListItem::ItemId() const
{
    return iItemId;
}

void CheckableListItem::InitImageItem(const QString& aFilename, const QString& aDate, const QString& aSize)
{
    ui->titleLabel->setText(aFilename);
    ui->modifiedLabel->setText(aDate);
    ui->sizeLabel->setText(aSize);
}

void CheckableListItem::InitCategoryItem(const QString& aCategoryName, int aCategoryLevel)
{
    static const QPixmap iconHierarchy(KHierarchy);
    ui->titleLabel->setText(aCategoryName);
    // This can't be defined in .qss
    if (aCategoryLevel == 0) {
        ui->indentLabel->hide();
    } else {
        ui->indentLabel->setIndent(aCategoryLevel * 20);
        ui->indentLabel->setMinimumWidth(40 + (aCategoryLevel - 1) * 20);
        ui->indentLabel->setPixmap(iconHierarchy);
    }
}

void CheckableListItem::CreateThumbnail()
{
    if (iThumbnailCreated)
        return;
    
    connect(&iThumbnailThread, SIGNAL(ThumbnailCreated(void *, QImage const)),
            this, SLOT(ThumbnailCreated(void *, QImage const)),
            Qt::QueuedConnection);
    iThumbnailCreated = true;
    iThumbnailThread.AddImage(iItemId, this);
}

bool CheckableListItem::ThumbnailCreated() const
{
    return iThumbnailCreated;
}

void CheckableListItem::CreateThumbnail(const QImage& aImage)
{
    ui->thumbLabel->setPixmap(QPixmap::fromImage(aImage));
}

