#include <QWidget>
#include <QDateTime>
#include <QKeyEvent>

#include <iostream>

#include "WPNetworkEngine.h"
#include "addpostview.h"
#include "BlogView.h"
#include "WPData.h"
#include "LocationFinder.h"
#include "utils.h"

#include "wpconfirmdialog.h"

static const char* KLocalDraft = "Local draft";

AddPostView::AddPostView(BlogView *parent) : QFrame(0), iParent(parent)
{
    ui.setupUi(this);
    ui.scrollArea->setWidget(ui.layoutWidget);
    ui.layoutWidget->setContentsMargins(5, 5, 5, 5);

    iCategoryListLength = 30; // TODO some better method

    ui.titleEdit->SetExplanation(tr("Post title"));
    ui.passwordEdit->SetExplanation(tr("Password"));
    ui.passwordEdit->SetPasswordMasking(true);
    ui.tagsEdit->SetExplanation(tr("Tags"));
    ui.contentsEdit->setExplanationText(tr("Tap to begin writing"));

    // Prepare softkeys
    QAction* leftSoftKey = new QAction(tr("Save"), this);
    connect(leftSoftKey, SIGNAL(triggered()), this, SLOT(SavePost()));
    leftSoftKey->setSoftKeyRole(QAction::PositiveSoftKey);
//    addAction(leftSoftKey);
    QAction* rightSoftKey = new QAction(tr("Cancel"), this);
    connect(rightSoftKey, SIGNAL(triggered()), this, SLOT(Cancel()));
    rightSoftKey->setSoftKeyRole(QAction::NegativeSoftKey);
//    addAction(rightSoftKey);

    connect(ui.addPhotosButton, SIGNAL(clicked(bool)), this, SLOT(AddPhotosClicked(bool)));
    connect(ui.addVideosButton, SIGNAL(clicked(bool)), this, SLOT(AddVideosClicked(bool)));
    connect(ui.categoriesEdit, SIGNAL(clicked()), this, SLOT(CategoriesClicked()));
    //connect(ui.addLocationDataBox, SIGNAL(stateChanged(int)), this, SLOT(AddLocationDataBoxClicked(int)));
    connect(ui.addLocationButton, SIGNAL(clicked()), this, SLOT(AddLocationButtonPressed()));

    ui.scrollArea->DisableScrollAtWidget(ui.statusEdit->view());

    // Finally add the data for WPCustomViewBasicData
    //
    iTitleText = tr("Edit Post");
    iTabText = tr("Edit");
    iHappyAction = leftSoftKey;
    iSadAction = rightSoftKey;
    
    iDialogController = new DialogController(parent, true);

#if defined(HAVE_QTM_LOCATION)
    connect(&iLocationFinder, SIGNAL(searchComplete(const QtMobility::QGeoPositionInfo&)),
             this, SLOT(LocationSearchComplete(const QtMobility::QGeoPositionInfo&)));
    connect(&iLocationFinder, SIGNAL(timeout()),
             this, SLOT(LocationSearchTimedOut()));    
    connect(&iLocationFinder, SIGNAL(TaskStarted(TWPNetworkEngineCommand)),
             iDialogController, SLOT(TaskStarted(TWPNetworkEngineCommand))
    );

    connect(&iLocationFinder, SIGNAL(TaskFinished(TWPNetworkEngineCommand)), 
             iDialogController, SLOT(TaskFinished(TWPNetworkEngineCommand))
    );
    connect(&iLocationFinder, SIGNAL(TaskFailed(TWPNetworkEngineCommand, QString)), 
             iDialogController, SLOT(TaskFailed(TWPNetworkEngineCommand, QString))
    );
#endif // defined(HAVE_QTM_LOCATION)
    
    connect(iDialogController, SIGNAL(TaskCancelled(TWPNetworkEngineCommand)),
            this, SLOT(LocationSearchCancelled(TWPNetworkEngineCommand)));
}

AddPostView::~AddPostView()
{
}

void AddPostView::NewPost(WPDataBlog *aBlog)
{
    iDataBlog = aBlog;
    iPost = new WPDataPost(*aBlog);

    // XXX: refactor to ResetView(iPost)
    iCategories.clear();
    iAddedImages.clear();
    iAddedVideos.clear();
    ui.titleEdit->setText(QString());
    ui.contentsEdit->reset();
    ui.contentsEdit->setText(QString());
    ui.tagsEdit->setText(QString());
    ui.dateTimeEdit->setDateTime(iPost->GetLocalTime());
    ui.passwordEdit->setText(QString());
    ui.addPhotosButton->setText(tr("Add photos..."));
    ui.addVideosButton->setText(tr("Add videos..."));
    ui.categoriesEdit->setText(tr("Add categories..."));

    // Statuses
    ui.statusEdit->clear();
    QStringList keyList, valueList;
    QMap<QString, QString>::const_iterator i;
    for (i = aBlog->iPostStatuses.constBegin(); i != aBlog->iPostStatuses.constEnd(); i++)
    {
        keyList.append(i.key());
        valueList.append(i.value());
    }
    ui.statusEdit->addItems(valueList);
    ui.statusEdit->addItem(KLocalDraft);

#if defined(HAVE_QTM_LOCATION)
    iLocation = QtMobility::QGeoPositionInfo();
    ui.addLocationButton->resetCoordinate();
#else // defined(HAVE_QTM_LOCATION)
    ui.addLocationButton->setNotSupported();
#endif // defined(HAVE_QTM_LOCATION)
}

void AddPostView::EditPost(WPDataBlog *aBlog, WPPost aPost)
{
    iDataBlog = aBlog;
    iPost = aPost;

    // XXX: refactor to ResetView(iPost)
    iAddedImages.clear();
    iAddedVideos.clear();
    ui.titleEdit->setText(aPost->iTitle);
    ui.contentsEdit->reset();
    ui.contentsEdit->setText(aPost->iDescription);
    ui.tagsEdit->setText(aPost->iMtKeywords);
    ui.dateTimeEdit->setDateTime(aPost->GetLocalTime());
    ui.passwordEdit->setText(aPost->iWpPassword);
    ui.addPhotosButton->setText(tr("Add photos..."));
    ui.addVideosButton->setText(tr("Add videos..."));

    // Collect categories
    iCategories = aPost->iCategories;
    ShowSelectedCategories();

    // Fill the status box, preselect the correct item
    ui.statusEdit->clear();
    QStringList keyList, valueList;
    QMap<QString, QString>::const_iterator j;
    for (j = aBlog->iPostStatuses.constBegin(); j != aBlog->iPostStatuses.constEnd(); j++)
    {
        keyList.append(j.key());
        valueList.append(j.value());
    }
    ui.statusEdit->addItems(valueList);
    if (aPost->IsLocal())
    {
        ui.statusEdit->addItem(KLocalDraft);
        ui.statusEdit->setCurrentIndex(ui.statusEdit->count() - 1);
    }
    // Using regular for loop as we need to get the index
    for (int i = 0; i < keyList.count(); ++i)
    {
        if (aPost->iStatus.compare(keyList.at(i)) == 0)
        {
            ui.statusEdit->setCurrentIndex(i);
            break;
        }
    }

#if defined(HAVE_QTM_LOCATION)
    iLocation = aPost->iLocation;
    if(iLocation.isValid()){
        ui.addLocationButton->setCoordinate(iLocation.coordinate());
    } else {
        ui.addLocationButton->resetCoordinate();
    }
#else // defined(HAVE_QTM_LOCATION)
    ui.addLocationButton->setNotSupported();
#endif // defined(HAVE_QTM_LOCATION)
}

void AddPostView::SavePost()
{
    if (ui.statusEdit->currentText() == QLatin1String(KLocalDraft)) {
        if (iAddedImages.count() > 0)
            emit ShowErrorNote(tr("Images are not saved in local drafts. Please re-add images when publishing the post."));
        
        if (iAddedVideos.count() > 0)
            emit ShowErrorNote(tr("Videos are not saved in local drafts. Please re-add videos when publishing the post."));

        // just update locally
        iPost->UpdateLocal(
            ui.titleEdit->text(),           // title
            ui.contentsEdit->text(),        // description
            ui.tagsEdit->text(),            // tags
            ui.dateTimeEdit->dateTime(),    // dateCreated
            ui.passwordEdit->text(),        // password
            iCategories                    // categories
#if defined(HAVE_QTM_LOCATION)
            , iLocation
#endif // defined(HAVE_QTM_LOCATION)
        );
    
    } else {
        // update on network
        iPost->UpdateStatus(
            ui.titleEdit->text(),           // title
            ui.contentsEdit->text(),        // description
            ui.tagsEdit->text(),            // tags
            ui.dateTimeEdit->dateTime(),    // dateCreated
            ui.passwordEdit->text(),        // password
            iCategories,                    // categories
            iDataBlog->iPostStatuses.key(ui.statusEdit->currentText()),     // status
            iAddedImages + iAddedVideos
#if defined(HAVE_QTM_LOCATION)
	        , iLocation
#endif // defined(HAVE_QTM_LOCATION)
        );
    }

    // forget item
    iPost = NULL;

    // XXX: more generic
    iParent->ShowTabbedListView();
#ifdef Q_WS_MAEMO_5
    close();
#endif
}

void AddPostView::ImagesAdded(const QStringList& aFilenames)
{
    iAddedImages = aFilenames;

    QString labelText;
    if (iAddedImages.size() == 0)
        labelText = tr("Add photos") + "...";
    else if (iAddedImages.size() == 1)
        labelText = tr("Adding 1 new photo");
    else
        labelText = tr("Adding %1 new photos").arg(iAddedImages.size());

    ui.addPhotosButton->setText(labelText);
}

void AddPostView::VideosAdded(const QStringList& aFilenames)
{
    iAddedVideos = aFilenames;

    QString labelText;
    if (iAddedVideos.size() == 0)
        labelText = tr("Add videos...");
    else if (iAddedVideos.size() == 1)
        labelText = tr("Adding 1 new video");
    else
        labelText = tr("Adding %1 new videos").arg(iAddedVideos.size());

    ui.addVideosButton->setText(labelText);
}

void AddPostView::SelectCategories (const QStringList &categories)
{
    iCategories = categories;

    // update
    ShowSelectedCategories();
}

void AddPostView::ShowSelectedCategories()
{
    QString textList;
    foreach (QString str, iCategories)
        textList += str + ", ";
    if (textList.length() > 0)
        textList = textList.mid(0, textList.length() - 2);
    else
        textList = tr("Add categories") + "...";

    if (textList.length() > iCategoryListLength)
        textList = textList.left(iCategoryListLength) + "...";
    ui.categoriesEdit->setText(textList);
}

void AddPostView::CategoriesClicked()
{
    iParent->ShowCategoryView(iCategories);
}

void AddPostView::Cancel()
{
    if(ui.titleEdit->text().length() || ui.contentsEdit->text().length())
    {
        WP_CONFIRM_AND_RETURN_ON_FALSE;
    }
    
#if defined(HAVE_QTM_LOCATION)
    iLocationFinder.cancelSearch();
#endif // defined(HAVE_QTM_LOCATION)

    iParent->ShowTabbedListView();
}

void AddPostView::AddPhotosClicked(bool aDummy)
{
    Q_UNUSED(aDummy)
    iParent->ShowImageList(iAddedImages, true);
}

void AddPostView::AddVideosClicked(bool aDummy)
{
    Q_UNUSED(aDummy)
    iParent->ShowVideoList(iAddedImages, true);
}

void AddPostView::AddLocationButtonPressed(){
#if defined(HAVE_QTM_LOCATION)
    if(!iLocation.isValid()){
	    iLocationFinder.reset();
    	iLocationFinder.startSearchingForLocation(0);
        
        iDialogController->ShowTaskProgress();
    } else {
        iLocation = QtMobility::QGeoPositionInfo();
        ui.addLocationButton->resetCoordinate();
    }
#endif // defined(HAVE_QTM_LOCATION)
}

void AddPostView::AddLocationDataBoxClicked(int newState)
{
}

#if defined(HAVE_QTM_LOCATION)
void AddPostView::LocationSearchComplete(const QtMobility::QGeoPositionInfo& info)
{
    iLocation = QtMobility::QGeoPositionInfo(info);
    ui.addLocationButton->setCoordinate(info.coordinate());
}
#endif // defined(HAVE_QTM_LOCATION)

void AddPostView::setFocus()
{
    ui.scrollArea->ScrollToTop();
    if(Utils::smallScreen()) {
        ui.titleEdit->setFocus();
    }
}

void AddPostView::LocationSearchCancelled (TWPNetworkEngineCommand command)
{
#if defined(HAVE_QTM_LOCATION)
    iLocationFinder.cancelSearch();
#endif // defined(HAVE_QTM_LOCATION)
}

void AddPostView::LocationSearchTimedOut()
{
}
