#include "LocationFinder.h"

#ifdef HAVE_QTM_LOCATION

#include <QTimer>

QGeoPositionInfoSource* LocationFinder::positionSource = 0;
QGeoSatelliteInfoSource* LocationFinder::satelliteInfoSource = 0;

LocationFinder::LocationFinder() : satellitesInView(0), satellitesInUse(0), searchCancelled(false) {
    if(!positionSource)
        positionSource = QGeoPositionInfoSource::createDefaultSource(this);

    if(positionSource){
        connect(positionSource, SIGNAL(positionUpdated(const QGeoPositionInfo&)),
                this, SLOT(positionUpdated(const QGeoPositionInfo&)));
	    connect(positionSource, SIGNAL(updateTimeout()), 
                 this, SLOT(searchTimedOut()));
    } else {
        connect(this, SIGNAL(positionUpdatedNoSource(QGeoPositionInfo)),
                this, SLOT(positionUpdated(QGeoPositionInfo)));
    }

    if(!satelliteInfoSource)
        satelliteInfoSource = QGeoSatelliteInfoSource::createDefaultSource(this);
    
    if(satelliteInfoSource){
        connect(satelliteInfoSource, SIGNAL(satellitesInViewUpdated(const QList<QGeoSatelliteInfo>&)),
		this, SLOT(satellitesInViewUpdated(const QList<QGeoSatelliteInfo>&)));
        connect(satelliteInfoSource, SIGNAL(satellitesInUseUpdated(const QList<QGeoSatelliteInfo>&)),
		this, SLOT(satellitesInUseUpdated(const QList<QGeoSatelliteInfo>&)));
    }
}

LocationFinder::~LocationFinder(){
}

const QtMobility::QGeoPositionInfo& LocationFinder::getCurrentPosition() const {
    return lastPosition;
}

void LocationFinder::positionUpdated(const QGeoPositionInfo &update){
    if(searchCancelled) return;

    lastPosition = QGeoPositionInfo(update);

    qDebug("Position updated");

    emit searchComplete(lastPosition);
    emit TaskFinished(EFetchingLocation);
}

void LocationFinder::fakePositionUpdate(){
    if(searchCancelled) return;
    
    emit searchComplete(QGeoPositionInfo(QtMobility::QGeoCoordinate(60.05, 20.543), QDateTime::currentDateTime()));
    emit TaskFinished(EFetchingLocation);
}

void LocationFinder::reset(){
    cancelSearch();
    lastPosition = QGeoPositionInfo();
    satellitesInView = 0;
    satellitesInUse = 0;
}

void LocationFinder::cancelSearch(){
    if(positionSource){
        positionSource->stopUpdates();
    }
    
    if(satelliteInfoSource){
        satelliteInfoSource->stopUpdates();
    }
    
    searchCancelled = true;
}

void LocationFinder::startSearchingForLocation(int timeout){
   searchCancelled = false;

   emit TaskStarted(EFetchingLocation);
   if(positionSource){
       qDebug("Position source exists, starting search");
       positionSource->requestUpdate(timeout);
   } else {
      emit TaskFailed(EFetchingLocation, tr("Could not connect to location service."));
   }
   
   //QTimer::singleShot(10000, this, SLOT(fakePositionUpdate()));
   //QTimer::singleShot(10000, this, SLOT(searchTimedOut()));
}

void LocationFinder::searchTimedOut(){

    if(searchCancelled) return;
    qDebug("Location search timeout!");

    emit timeout();
    emit TaskFailed(EFetchingLocation, tr("Location search timed out."));
}

void LocationFinder::satellitesInViewUpdated(const QList<QGeoSatelliteInfo>& satellites){
    satellitesInView = satellites.count();
}

void LocationFinder::satellitesInUseUpdated(const QList<QGeoSatelliteInfo>& satellites){
    satellitesInUse = satellites.count();
}

int LocationFinder::getSatellitesInView() const {
    return satellitesInView;
}

int LocationFinder::getSatellitesInUse() const {
    return satellitesInUse;
}

#endif // HAVE_QTM_LOCATION
