/*
 * vim:ts=4:sw=4:et:cindent:cino=(0
 */ 

#include <config.h>
#include <glib.h>
#include <glib/gi18n-lib.h>

#include <gtk/gtk.h>
#include <hildon/hildon-picker-button.h>
#include <hildon/hildon-touch-selector.h>
#include <hildon/hildon-entry.h>
#include <hildon/hildon-defines.h>
#include <hildon/hildon-banner.h>
#include <hildon/hildon-pannable-area.h>

#include <gconf/gconf-client.h>

#include <libprofile.h>

#include "libtweakr-section/tweakr-section.h"
#include "libtweakr-section/tweakr-module.h"


#define TWEAKR_TYPE_PROFILE_SECTION \
        (tweakr_profile_section_type)
#define TWEAKR_PROFILE_SECTION(obj) \
        (G_TYPE_CHECK_INSTANCE_CAST ((obj), \
        TWEAKR_TYPE_PROFILE_SECTION, \
        TweakrProfileSection))
#define TWEAKR_PROFILE_SECTION_CLASS(k) \
        (G_TYPE_CHECK_CLASS_CAST((k), \
        TWEAKR_TYPE_PROFILE_SECTION, \
        TweakrProfileSectionClass))
#define TWEAKR_IS_PROFILE_SECTION(obj) \
        (G_TYPE_CHECK_INSTANCE_TYPE ((obj), \
        TWEAKR_TYPE_PROFILE_SECTION))

#define GCONF_PATH "/system/tweakr"

typedef struct _TweakrProfileSection TweakrProfileSection;
typedef struct _TweakrProfileSectionClass
               TweakrProfileSectionClass;

struct _TweakrProfileSection
{
    TweakrSection parent_instance;

    GtkWidget *preset_button;
    GtkWidget *delete_button;
    GtkWidget *delete_dialog;
    GtkWidget *delete_box;
    GConfClient *gconf;
};

struct _TweakrProfileSectionClass
{
    TweakrSectionClass parent_class;
};


static GType tweakr_profile_section_get_type (GTypeModule *module);
static void tweakr_profile_section_class_init
    (TweakrProfileSectionClass *class);
static void tweakr_profile_section_init
    (TweakrProfileSection *section);
static void tweakr_profile_section_dispose (GObject *obj);

static gboolean _save (TweakrSection *section,
                       gboolean *requires_restart);

static GType tweakr_profile_section_type = 0;
static TweakrSectionClass *
    tweakr_profile_section_parent_class = NULL;


G_MODULE_EXPORT void
tweakr_module_load (TweakrModule *module)
{
    tweakr_profile_section_get_type (G_TYPE_MODULE (module));
}

G_MODULE_EXPORT void
tweakr_module_unload (TweakrModule *module)
{
}

static GType
tweakr_profile_section_get_type (GTypeModule *module)
{
    if (!tweakr_profile_section_type)
    {
        static const GTypeInfo section_info =
        {
            sizeof (TweakrProfileSectionClass),
            (GBaseInitFunc) NULL,
            (GBaseFinalizeFunc) NULL,
            (GClassInitFunc) tweakr_profile_section_class_init,
            NULL,           /* class_finalize */
            NULL,           /* class_data     */
            sizeof (TweakrProfileSection),
            0,              /* n_preallocs    */
            (GInstanceInitFunc) tweakr_profile_section_init
        };

        tweakr_profile_section_type =
            g_type_module_register_type (module, TWEAKR_TYPE_SECTION,
                                         "TweakrProfileSection",
                                         &section_info, 0);
    }

    return tweakr_profile_section_type;
}

static void
tweakr_profile_section_class_init
    (TweakrProfileSectionClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);
    TweakrSectionClass *section_class =
        TWEAKR_SECTION_CLASS (klass);

    tweakr_profile_section_parent_class =
        g_type_class_peek_parent (klass);

    section_class->name   = "_Profile";
    section_class->save = _save;

    object_class->dispose = tweakr_profile_section_dispose;
}

static void
_save_preset_clicked (HildonPickerButton *button,
                      TweakrProfileSection *section)
{
    GtkWindow *parent;
    GtkWidget *dialog, *entry;
    const gchar* text = NULL;

    parent = GTK_WINDOW (gtk_widget_get_ancestor (tweakr_section_get_widget
                                                  (TWEAKR_SECTION (section)),
                                                  GTK_TYPE_WINDOW));
    dialog = gtk_dialog_new_with_buttons
        (_("Preset name"),
         parent,
         GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR,
         GTK_STOCK_OK, GTK_RESPONSE_OK,
         NULL);

    entry = hildon_entry_new (HILDON_SIZE_FINGER_HEIGHT);
    gtk_widget_show (entry);

    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), entry,
                        TRUE, TRUE, 0);
    while (TRUE)
    {
        gint ret;
        gint iter;

        ret = gtk_dialog_run (GTK_DIALOG (dialog));
        if (ret == GTK_RESPONSE_OK)
        {
            GtkWidget *banner;
            profileval_t *values, *values_iter;
            GConfClient *gconf;
            gboolean cont = FALSE;

            text = hildon_entry_get_text (HILDON_ENTRY (entry));
            if (text == NULL || text[0] == '\0')
            {
                GtkWidget *banner;

                banner = hildon_banner_show_information
                    (dialog, NULL, _("Enter the name first."));

                continue;
            }

            for (iter = 0; iter < strlen (text); iter++)
            {
                if (!g_ascii_isalnum (text[iter]) &&
                    text[iter] != '-' &&
                    text[iter] != '_')
                {
                    GtkWidget *banner;

                    banner = hildon_banner_show_information
                        (dialog, NULL,
                         _("Preset name can only contains letters, numbers, - and _."));

                    cont = TRUE;
                }
            }

            if (cont)
                continue;

            /* Save the settings to our own gconf directory. */

            gconf = gconf_client_get_default ();
            gconf_client_set_string (gconf, GCONF_PATH "/current-preset",
                                     _("None"), NULL);
            values = profile_get_values ("general");
            for (values_iter = values;
                 values_iter->pv_key != NULL;
                 values_iter++)
            {
                gchar key[128];

                g_snprintf (key, 128, "%s/%s/%s", GCONF_PATH, text,
                            values_iter->pv_key);
                gconf_client_set_string (gconf, key, values_iter->pv_val,
                                         NULL);
            }
            profile_free_values (values);
            g_object_unref (gconf);

            banner = hildon_banner_show_information
                (GTK_WIDGET (parent), NULL,
                 _("Preset saved. Use the status menu to select it."));
        }
        else
        {
            gtk_widget_destroy (dialog);
            break;
        }
        gtk_widget_destroy (dialog);
    }
}

static GtkWidget *
_build_save_preset_button (TweakrProfileSection *section)
{
    GtkWidget *button;

    button = hildon_button_new (HILDON_SIZE_AUTO |
                                HILDON_SIZE_FINGER_HEIGHT,
                                HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    hildon_button_set_title (HILDON_BUTTON (button),
                             _("Save current General profile to new preset"));
    gtk_button_set_alignment (GTK_BUTTON (button), 0.0f, 0.5f);

    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (_save_preset_clicked), section);

    gtk_widget_show (button);
    return button;
}

static void
_preset_clicked (HildonButton *button, TweakrProfileSection *section)
{
    const gchar *path = g_object_get_data (G_OBJECT (button), "path");
    gchar *basename;
    gchar *current;

    current = gconf_client_get_string (section->gconf,
                                       GCONF_PATH "/current-preset",
                                       NULL);
    basename = g_path_get_basename (path);
    if (g_strcmp0 (current, basename) == 0)
    {
        gconf_client_set_string (section->gconf,
                                 GCONF_PATH "/current-preset", _("None"),
                                 NULL);
    }

    gconf_client_recursive_unset (section->gconf, path, 0, NULL);
    g_free (basename);
    g_free (current);

    gtk_dialog_response (GTK_DIALOG (section->delete_dialog),
                         GTK_RESPONSE_OK);
    hildon_banner_show_information (NULL, NULL,
                                    _("Preset deleted."));
}

static void
_add_preset (gchar *preset, TweakrProfileSection *section)
{
    GtkWidget *button;
    gchar *basename;

    basename = g_path_get_basename (preset);
    button = hildon_button_new_with_text
        (HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_VERTICAL,
         basename, NULL);
    gtk_button_set_alignment (GTK_BUTTON (button), 0.0f, 0.5f);
    gtk_box_pack_start (GTK_BOX (section->delete_box), button, FALSE, FALSE,
                        0);
    g_object_set_data_full (G_OBJECT (button), "path", preset,
                            (GDestroyNotify) g_free);
    g_signal_connect (button, "clicked", G_CALLBACK (_preset_clicked),
                      section);
}

static void
_delete_preset_clicked (HildonButton *b, TweakrProfileSection *section)
{
    GtkWidget *panarea;
    gint ret;
    GSList *presets;
    GtkWidget *parent;

    parent = gtk_widget_get_ancestor (GTK_WIDGET (b), GTK_TYPE_WINDOW);

    section->delete_dialog = gtk_dialog_new ();
    gtk_window_set_modal (GTK_WINDOW (section->delete_dialog), TRUE);
    gtk_window_set_title (GTK_WINDOW (section->delete_dialog),
                          _("Select preset"));

    panarea = hildon_pannable_area_new ();
    section->delete_box = gtk_vbox_new (FALSE, 0);

    hildon_pannable_area_add_with_viewport (HILDON_PANNABLE_AREA (panarea),
                                            section->delete_box);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (section->delete_dialog)->vbox),
                        panarea, TRUE, TRUE, 0);

    if (section->gconf == NULL)
    {
        section->gconf = gconf_client_get_default ();
    }
    presets = gconf_client_all_dirs (section->gconf, GCONF_PATH, NULL);

    g_object_set (G_OBJECT (panarea), "height-request",
                  MIN (350, g_slist_length (presets) * 70), NULL);

    g_slist_foreach (presets, (GFunc) _add_preset, section);

    gtk_widget_show_all (GTK_DIALOG (section->delete_dialog)->vbox);
    ret = gtk_dialog_run (GTK_DIALOG (section->delete_dialog));
    gtk_widget_destroy (section->delete_dialog);

    g_slist_free (presets);
}

static GtkWidget *
_build_delete_preset_button (TweakrProfileSection *section)
{
    GtkWidget *button;

    button = hildon_button_new (HILDON_SIZE_AUTO |
                                HILDON_SIZE_FINGER_HEIGHT,
                                HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    hildon_button_set_title (HILDON_BUTTON (button),
                             _("Delete a profile preset"));
    gtk_button_set_alignment (GTK_BUTTON (button), 0.0f, 0.5f);

    g_signal_connect (G_OBJECT (button), "clicked",
                      G_CALLBACK (_delete_preset_clicked), section);

    gtk_widget_show (button);
    return button;
}

static void
tweakr_profile_section_init (TweakrProfileSection *section)
{
    TweakrSection *iface;

    iface = TWEAKR_SECTION (section);
    iface->name = _("Profile presets");
    iface->widget = gtk_vbox_new (FALSE, 0);

    section->preset_button = _build_save_preset_button (section);
    section->delete_button = _build_delete_preset_button (section);

    gtk_box_pack_start (GTK_BOX (iface->widget), section->preset_button,
                        FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (iface->widget), section->delete_button,
                        FALSE, FALSE, 0);
}

static void
tweakr_profile_section_dispose (GObject *obj)
{
    TweakrProfileSection *section = TWEAKR_PROFILE_SECTION (obj);

    if (section->gconf != NULL)
    {
        g_object_unref (section->gconf);
        section->gconf = NULL;
    }

    G_OBJECT_CLASS (tweakr_profile_section_parent_class)->dispose (obj);
}

static gboolean _save (TweakrSection *section, gboolean *requires_restart)
{
    return TRUE;
}

