#ifndef SONGMODEL_H
#define SONGMODEL_H

#include "songlist.h"
#include <QAbstractListModel>

using namespace QtUbuntuOne;

class ArtworkCache;

class SongRoles : public QObject
{
    Q_OBJECT

    Q_ENUMS(Role)

public:
    enum Role {
        IdRole = Qt::UserRole + 1,
        TitleRole,
        ArtistRole,
        ArtistIdRole,
        AlbumTitleRole,
        AlbumArtistRole,
        AlbumIdRole,
        GenreRole,
        FilePathRole,
        FileSuffixRole,
        MimeTypeRole,
        UrlRole,
        ArtworkUrlRole,
        ArtworkRole,
        StreamUrlRole,
        YearRole,
        DiscNumberRole,
        BitRateRole,
        DurationRole,
        SizeRole,
#ifndef QML_USER_INTERFACE
        ArtworkPressedRole
#endif
    };
};

class SongModel : public QAbstractListModel
{
    Q_OBJECT

    Q_PROPERTY(int count
               READ rowCount
               NOTIFY countChanged)
    Q_PROPERTY(bool loading
               READ loading
               NOTIFY loadingChanged)
    Q_PROPERTY(QList<Song*> songs
               READ songs)

public:
    explicit SongModel(QObject *parent = 0);
    ~SongModel();

#if QT_VERSION >= 0x050000
    QHash<int, QByteArray> roleNames() const;
#endif
    int rowCount(const QModelIndex &parent = QModelIndex()) const;

    QVariant data(const QModelIndex &index, int role) const;
    Q_INVOKABLE QVariant data(int row, int role) const;

    bool setData(const QModelIndex &index, const QVariant &value, int role);
    Q_INVOKABLE bool setData(int row, const QVariant &value, int role);

    QMap<int, QVariant> itemData(const QModelIndex &index) const;
    Q_INVOKABLE QMap<int, QVariant> itemData(int row) const;

    Song* get(const QModelIndex &index) const;
    Q_INVOKABLE Song* get(int row) const;

    bool loading() const;

    QList<Song*> songs() const;

public slots:
    void getSongs();
    void getAlbumSongs(const QString &albumId);
    void getPlaylistSongs(const QString &playlistId);

    void reload();
    void clear();

private:
    void setLoading(bool loading);

    QString albumId() const;
    void setAlbumId(const QString &id);

    QString playlistId() const;
    void setPlaylistId(const QString &id);

    void appendSong(Song *song);
    void appendSongs(QList<Song*> songs);

    void insertSong(int i, Song *song);
    void insertSongs(int i, QList<Song*> songs);

    void removeSong(int i);
    void removeSongs(int i, int count);

private slots:
    void addSongs(SongList *list);
    void onArtworkReady();

signals:
    void countChanged(int count);
    void loadingChanged(bool loading);

private:
    ArtworkCache *m_cache;
#ifndef QML_USER_INTERFACE
    int m_artworkPressedRow;
#endif
    QList<Song*> m_list;

    bool m_loading;

    QString m_albumId;

    QString m_playlistId;

    QHash<int, QByteArray> m_roleNames;
};

#endif // SONGMODEL_H
