#include "songinfowindow.h"
#include "separatorlabel.h"
#include "artworklabel.h"
#include "listview.h"
#include "nowplayingwindow.h"
#include "nowplayingbutton.h"
#include "scrollarea.h"
#include "../shared/utils.h"
#include "../shared/artworkcache.h"
#include "../mediaplayer/mobility/mediaplayer.h"
#include <QMenuBar>
#include <QGridLayout>
#include <QLabel>
#include <QAction>

SongInfoWindow::SongInfoWindow(Song *song, QWidget *parent) :
    StackedWindow(parent),
    m_song(song),
    m_queueAction(this->menuBar()->addAction(tr("Add to playback queue"), this, SLOT(queueSong()))),
    m_titleLabel(new QLabel(QString("<p style='font-size:32px;'>%1</p>").arg(m_song->title()), this)),
    m_artistLabel(new QLabel(QString("%1: %2").arg(tr("Artist")).arg(m_song->artist()), this)),
    m_dateLabel(new QLabel(QString("%1: %2").arg(tr("Year")).arg(m_song->year()), this)),
    m_genreLabel(new QLabel(QString("%1: %2").arg(tr("Genre")).arg(m_song->genre().isEmpty() ? tr("Unknown") : m_song->genre()), this)),
    m_formatLabel(new QLabel(QString("%1: %2").arg(tr("Format")).arg(m_song->fileSuffix()), this)),
    m_sizeLabel(new QLabel(QString("%1: %2").arg(tr("Size")).arg(Utils::fileSizeFromBytes(m_song->size())), this)),
    m_lengthLabel(new QLabel(QString("%1: %2").arg(tr("Length")).arg(Utils::durationFromSecs(m_song->duration())), this)),
    m_cache(new ArtworkCache(this)),
    m_artworkLabel(new ArtworkLabel(this)),
    m_propertiesSeparator(new SeparatorLabel(tr("Properties"), this)),
    m_scrollArea(new ScrollArea(this))
{
    this->setWindowTitle(tr("Song info"));
    this->setCentralWidget(m_scrollArea);

    m_artworkLabel->setFixedSize(124, 124);
    m_titleLabel->setWordWrap(true);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);

    grid->setContentsMargins(0, 0, 0, 0);
    grid->addItem(new QSpacerItem(10, 10, QSizePolicy::Maximum), 0, 0);
    grid->addWidget(m_titleLabel, 0, 1, 1, 2, Qt::AlignTop);
    grid->addWidget(m_artworkLabel, 1, 1, 2, 1, Qt::AlignTop);
    grid->addWidget(m_artistLabel, 1, 2, Qt::AlignTop);
    grid->addWidget(m_dateLabel, 2, 2, Qt::AlignTop);
    grid->addWidget(m_propertiesSeparator, 3, 1, 1, 2);
    grid->addWidget(m_genreLabel, 4, 1, 1, 2);
    grid->addWidget(m_formatLabel, 5, 1, 1, 2);
    grid->addWidget(m_sizeLabel, 6, 1, 1, 2);
    grid->addWidget(m_lengthLabel, 7, 1, 1, 2);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_scrollArea->setMonitorScrolling(true);

    this->connect(m_cache, SIGNAL(artworkReady()), this, SLOT(setArtwork()));
    this->connect(m_artworkLabel, SIGNAL(clicked()), this, SLOT(playSong()));
    this->connect(m_scrollArea, SIGNAL(scrollingStarted()), NowPlayingButton::instance(), SLOT(hide()));
    this->connect(m_scrollArea, SIGNAL(scrollingStopped()), NowPlayingButton::instance(), SLOT(poke()));

    this->setArtwork();
    this->onOrientationChanged();
}

void SongInfoWindow::setLandscapeLayout() {
    m_scrollArea->widget()->setMaximumWidth(800);
}

void SongInfoWindow::setPortraitLayout() {
    m_scrollArea->widget()->setMaximumWidth(480);
}

void SongInfoWindow::setArtwork() {
    QImage artwork = m_cache->artwork(m_song->artworkUrl(), QSize(124, 124));

    if (artwork.isNull()) {
        artwork = QImage("/usr/share/icons/hicolor/124x124/hildon/mediaplayer_default_album.png");
    }

    m_artworkLabel->setImage(artwork);
}

void SongInfoWindow::playSong() {
    MediaPlayer::instance()->play(m_song);
    NowPlayingWindow *window = new NowPlayingWindow(this);
    window->show();
}

void SongInfoWindow::queueSong() {
    MediaPlayer::instance()->addSong(m_song);
}
