#include "scrollarea.h"
#include <QAbstractKineticScroller>
#include <QScrollBar>
#include <QTimer>

ScrollArea::ScrollArea(QWidget *parent) :
    QScrollArea(parent),
    m_kineticScroller(property("kineticScroller").value<QAbstractKineticScroller *>()),
    m_scrollTimer(0),
    m_minimum(0),
    m_maximum(1000000),
    m_monitor(false)
{
    this->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    this->connect(this->verticalScrollBar(), SIGNAL(rangeChanged(int,int)), this, SLOT(setScrollRange(int,int)));
}

void ScrollArea::keyPressEvent(QKeyEvent *event) {
    if (event->key() == Qt::Key_Down) {
        if (event->modifiers() == Qt::ShiftModifier) {
            this->positionAtEnd();
        }
        else {
            m_kineticScroller->scrollTo(QPoint(0, this->height()));
        }

        event->accept();
    }
    else if (event->key() == Qt::Key_Up) {
        if (event->modifiers() == Qt::ShiftModifier) {
            this->positionAtBeginning();
        }
        else {
            m_kineticScroller->scrollTo(QPoint(0, -this->height()));
        }

        event->accept();
    }
    else {
        event->ignore();
    }
}

bool ScrollArea::monitorScrolling() const {
    return m_monitor;
}

void ScrollArea::setMonitorScrolling(bool monitor) {
    if (monitor != this->monitorScrolling()) {
        m_monitor = monitor;

        if (monitor) {
            if (!m_scrollTimer) {
                m_scrollTimer = new QTimer(this);
                m_scrollTimer->setInterval(500);
                m_scrollTimer->setSingleShot(true);
                this->connect(m_scrollTimer, SIGNAL(timeout()), this, SIGNAL(scrollingStopped()));
            }

            this->connect(this->verticalScrollBar(), SIGNAL(valueChanged(int)), this, SLOT(onScrollPositionChanged(int)));
        }
        else {
            this->disconnect(this->verticalScrollBar(), SIGNAL(valueChanged(int)), this, SLOT(onScrollPositionChanged(int)));
        }
    }
}

bool ScrollArea::isScrolling() const {
    switch (m_kineticScroller->state()) {
    case QAbstractKineticScroller::Pushing:
    case QAbstractKineticScroller::AutoScrolling:
        return true;
    default:
        return false;
    }
}

void ScrollArea::setScrollRange(int minimum, int maximum) {
    m_minimum = minimum;
    m_maximum = maximum;
}

void ScrollArea::onScrollPositionChanged(int position) {
    if (!m_scrollTimer->isActive()) {
        emit scrollingStarted();
    }

    if (position == m_maximum) {
        emit atEnd();
    }

    m_scrollTimer->start();
}

void ScrollArea::positionAtBeginning() {
    this->ensureVisible(0, 0);
}

void ScrollArea::positionAtEnd() {
    this->ensureVisible(0, m_maximum);
}
