#include "playlistsongswindow.h"
#include "songlistdelegate.h"
#include "songinfowindow.h"
#include "nowplayingwindow.h"
#include "listview.h"
#include "artworklabel.h"
#include "separatorlabel.h"
#include "nowplayingbutton.h"
#include "scrollarea.h"
#include "../shared/songmodel.h"
#include "../shared/artworkcache.h"
#include "../mediaplayer/mobility/mediaplayer.h"
#include <QAction>
#include <QLabel>
#include <QPushButton>
#include <QMenuBar>
#include <QGridLayout>
#include <QTimer>
#include <QScrollBar>

PlaylistSongsWindow::PlaylistSongsWindow(Playlist *playlist, QWidget *parent) :
    StackedWindow(parent),
    m_playlist(playlist),
    m_view(new ListView(this)),
    m_model(new SongModel(this)),
    m_cache(new ArtworkCache(this)),
    m_contextMenu(new QMenu(this)),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueSong()))),
    m_titleLabel(new QLabel(QString("<p style='font-size:32px;'>%1</p>").arg(m_playlist->name()), this)),
    m_artwork(new ArtworkLabel(this)),
    m_playButton(new QPushButton(tr("Play all"), this)),
    m_separator(new SeparatorLabel(QString("%1 %2").arg(m_playlist->songCount()).arg(tr("songs")), this)),
    m_scrollArea(new ScrollArea(this))
{
    this->setWindowTitle(tr("Playlist"));
    this->setCentralWidget(m_scrollArea);

    SongListDelegate *delegate = new SongListDelegate(m_view);

    m_view->setModel(m_model);
    m_view->setItemDelegate(delegate);
    m_view->setMonitorScrolling(true);
    m_titleLabel->setWordWrap(true);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);

    m_artwork->setInteractive(false);
    m_artwork->setFixedSize(124, 124);
    m_playButton->setFixedWidth(150);

    grid->setContentsMargins(0, 0, 0, 0);
    grid->addItem(new QSpacerItem(10, 10, QSizePolicy::Maximum), 0, 0);
    grid->addWidget(m_titleLabel, 0, 1, 1, 3, Qt::AlignTop);
    grid->addWidget(m_artwork, 1, 1, 2, 1, Qt::AlignTop);
    grid->addWidget(m_playButton, 2, 3, Qt::AlignRight);
    grid->addWidget(m_separator, 3, 0, 1, 4);
    grid->addWidget(m_view, 4, 0, 1, 4);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_scrollArea->setMonitorScrolling(true);

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_cache, SIGNAL(artworkReady()), this, SLOT(setArtwork()));
    this->connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playSong(QModelIndex)));
    this->connect(m_view, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_playButton, SIGNAL(clicked()), this, SLOT(playPlaylist()));
    this->connect(m_scrollArea, SIGNAL(scrollingStarted()), NowPlayingButton::instance(), SLOT(hide()));
    this->connect(m_scrollArea, SIGNAL(scrollingStopped()), NowPlayingButton::instance(), SLOT(poke()));

    m_model->getPlaylistSongs(m_playlist->id());

    this->onOrientationChanged();
}

void PlaylistSongsWindow::showEvent(QShowEvent *event) {
    this->onLoadingChanged(m_model->loading());
    StackedWindow::showEvent(event);
}

void PlaylistSongsWindow::setLandscapeLayout() {
    m_scrollArea->widget()->setMaximumWidth(800);
}

void PlaylistSongsWindow::setPortraitLayout() {
    m_scrollArea->widget()->setMaximumWidth(480);
}

void PlaylistSongsWindow::setArtwork() {
    QImage artwork = m_cache->artwork(m_model->data(0, SongRoles::ArtworkUrlRole).toUrl(), QSize(124, 124));

    if (artwork.isNull()) {
        artwork = QImage("/usr/share/icons/hicolor/124x124/hildon/mediaplayer_default_album.png");
    }

    m_artwork->setImage(artwork);
}

void PlaylistSongsWindow::onLoadingChanged(bool loading) {
    this->toggleBusy(loading);
    m_playButton->setEnabled((!loading) && (m_model->rowCount() > 0));

    if ((!m_model->loading()) && (m_model->rowCount() > 0)) {
        m_view->setFixedHeight(m_model->rowCount() * m_view->sizeHintForRow(0));
        this->setArtwork();
    }
}

void PlaylistSongsWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_addToQueueContextAction);
}

void PlaylistSongsWindow::queueSong() {
    MediaPlayer::instance()->addSong(m_model->get(m_view->currentIndex()));
}

void PlaylistSongsWindow::onItemClicked(const QModelIndex &index) {
    SongInfoWindow *window = new SongInfoWindow(m_model->get(index), this);
    window->show();
}

void PlaylistSongsWindow::playSong(const QModelIndex &index) {
    MediaPlayer::instance()->play(m_model->get(index));
    NowPlayingWindow *window = new NowPlayingWindow(this);
    window->show();
}

void PlaylistSongsWindow::playPlaylist() {
    MediaPlayer::instance()->play(m_model->songs());
    NowPlayingWindow *window = new NowPlayingWindow(this);
    window->show();
}
