// Filename: mainwindow.cpp

#include <QDebug>
#include <QMessageBox>
#include <QPushButton>
#include <QDesktopServices>
#include <QUrl>

#include "mainwindow.h"
#include "mapwidget.h"
#include "settingsdialog.h"
#include "iconloader.h"  
#include "cachewindow.h"
#include "config.h"
#include "tool.h"

#include <QIcon>
#include <QSettings>

#ifndef Q_WS_MAEMO_5
#include <QVBoxLayout>
#else
#include <QMaemo5InformationBox>
#endif

#include "tool_rot13.h"
#include "tool_projection.h"

void MainWindow::launchToolRot13() {
  ToolRot13 *tool = new ToolRot13(this);
  tool->show();
}

void MainWindow::launchToolProjection() {
  ToolProjection *tool = new ToolProjection(m_mapWidget, m_wptDb, this);
  tool->show();
}

void MainWindow::enableToolProjection(bool on) {
  qDebug() << __FUNCTION__ << on;

  if(m_ActionToolProjection)
    m_ActionToolProjection->setEnabled(on);
}

void MainWindow::createMenu() {

  addMenuEntry(tr("&About"), this, SLOT(about()));
  addMenuEntry(tr("&Settings"), this, SLOT(settingsdialog()));

  /* create a submenu for the plugin selection */  
  m_cacheProvider->createMenu(menuBar());

  /* add tools to the tools menu */
  QMenu *toolsMenu = menuBar()->addMenu(tr("&Tools"));
  ToolRot13::registerMenu(toolsMenu, this, SLOT(launchToolRot13()));
  m_ActionToolProjection = ToolProjection::registerMenu(toolsMenu, this, SLOT(launchToolProjection()));
  m_ActionToolProjection->setEnabled(false);
}

void MainWindow::settingsdialog() {
  qDebug() << __FUNCTION__;

  SettingsDialog dialog(m_mapWidget, m_cacheProvider, this);
  dialog.exec(); 
}

void MainWindow::helpButtonClicked() {
  QUrl url("http://www.harbaum.org/till/cacheme/");
  QDesktopServices::openUrl(url);
}

void MainWindow::aboutBox(QWidget *parent, const QString &title, const QString &text) {
  QMessageBox *msgBox = 
    new QMessageBox(title, text, QMessageBox::Information, 0, 0, 0, parent);
  msgBox->setAttribute(Qt::WA_DeleteOnClose);

  msgBox->addButton(QMessageBox::Ok);

#if defined(Q_OS_SYMBIAN) || defined(Q_WS_SIMULATOR)
  QAction *helpAction = new QAction( tr("Help"), this );
  helpAction->setSoftKeyRole( QAction::NegativeSoftKey );
  connect(helpAction, SIGNAL(triggered()), this, SLOT(helpButtonClicked()));
  msgBox->addAction(helpAction );
#else
  QPushButton *helpButton = msgBox->addButton(QMessageBox::Help);
  connect(helpButton, SIGNAL(clicked()), this, SLOT(helpButtonClicked()));
#endif

  QIcon icon = msgBox->windowIcon();
  QSize size = icon.actualSize(QSize(64, 64));
  msgBox->setIconPixmap(icon.pixmap(size));

  msgBox->exec();
}

void MainWindow::about() {
  aboutBox(this, tr("About CacheMe"),
	   tr("CacheMe") + " " + tr("Version") + " " + VERSION + "\n" + 
	   tr("Build date:") + " " + __DATE__ + "\n" +
           tr("(c) 2011 by Till Harbaum <till@harbaum.org>") + "\n\n" +
           tr("CacheMe is an online geocaching application.")
	   );
}


MainWindow::MainWindow(LocationProvider *locationProvider, QWidget *parent) : 
  CustomWindow("MainWindow", parent), m_locationProvider(locationProvider),
  m_lastFlags(0), m_ActionToolProjection(NULL) {

  qDebug() << __FUNCTION__;
  m_cacheProvider = new CacheProvider(this);

  setWindowTitle(tr("CacheMe"));

  IconLoader iconLoader(48);
  QPixmap *icon = iconLoader.load(APPNAME);
  if(icon) setWindowIcon(QIcon(*icon));

  createMenu();

  // search for service providers
  QStringList list = QGeoServiceProvider::availableServiceProviders();
  for(int i=0;i<list.size();i++)
    qDebug() << __FUNCTION__ << "Service provider: " << list[i];

  // there needs to be at least one geo service provider
  Q_ASSERT(list.size() > 0);

  QSettings settings;
  settings.beginGroup("Map");
  QString serviceProviderStr = settings.value("Service", "nokia").toString();
  settings.endGroup();

  // give app id key to service provider
  QMap<QString, QVariant> params;
  params.insert("mapping.app_id", "athGTWaNTNPAgYYwbBLh");
  
  this->m_serviceProvider = new QGeoServiceProvider(serviceProviderStr, params);
  if(!this->m_serviceProvider->mappingManager())
    this->m_serviceProvider = new QGeoServiceProvider("nokia", params);

  this->m_mapWidget = new MapWidget(this->m_serviceProvider->mappingManager());

  m_wptDb = new WaypointDb();
  this->m_mapWidget->setWaypointDb(m_wptDb);

  // make sure cache updates are processed
  QObject::connect(this->m_locationProvider, SIGNAL(positionUpdated(const QGeoPositionInfo &)), 
		   this->m_mapWidget, SLOT(positionUpdated(const QGeoPositionInfo &)));
  QObject::connect(this->m_locationProvider, SIGNAL(positionUpdated(const QCompassReading *)), 
		   this->m_mapWidget, SLOT(positionUpdated(const QCompassReading *)));

  QObject::connect(this->m_cacheProvider, SIGNAL(replyOverview(const CacheList &)), 
		   this, SLOT(updateCaches(const CacheList &)));

  QObject::connect(this->m_cacheProvider, SIGNAL(replyInfo(const Cache &)), 
		   this->m_mapWidget, SLOT(showBubble(const Cache &)));

  QObject::connect(this->m_cacheProvider, SIGNAL(replyDetail(const Cache &)), 
		   this, SLOT(showDetail(const Cache &)));

  QObject::connect(this->m_cacheProvider, SIGNAL(replyError(const QString &)), 
		   this, SLOT(displayError(const QString &)));

  QObject::connect(this->m_cacheProvider, SIGNAL(reload()),
		   this, SLOT(reloadCaches()));

  QObject::connect(this->m_cacheProvider, SIGNAL(notifyBusy(bool)),
		   this, SLOT(setBusy(bool)));

  QObject::connect(this->m_mapWidget, SIGNAL(mapChanged(const int)),
		   this, SLOT(reloadCaches(const int)));

  QObject::connect(this->m_mapWidget, SIGNAL(showMessage(const QString &)),
		   this, SLOT(displayError(const QString &)));

  QObject::connect(this->m_mapWidget, SIGNAL(cacheIsSelected(bool)),
		   this, SLOT(enableToolProjection(bool)));

  QObject::connect(this, SIGNAL(zoomInPressed()),
		   this->m_mapWidget, SLOT(zoomIn()));

  QObject::connect(this, SIGNAL(zoomOutPressed()),
		   this->m_mapWidget, SLOT(zoomOut()));

  QObject::connect(this->m_mapWidget, SIGNAL(fullscreen()),
		   this, SLOT(toggleFullscreen()));

  // connect to map to handle clicks on cache icons ...
  QObject::connect(this->m_mapWidget, SIGNAL(cacheClicked(const QString &)),
		   this, SLOT(selectCache(const QString &)));

  // ... and the detail button
  QObject::connect(this->m_mapWidget, SIGNAL(detailClicked(const QString &)),
		   this, SLOT(detailCache(const QString &)));


  // also tell map about window activation and deactivation
  QObject::connect(this, SIGNAL(activated(bool)),
		   this->m_mapWidget, SLOT(windowActivated(bool)));

  // build a graphics scene based on the mapwidget
  QGraphicsScene *scene = new QGraphicsScene(this);
  m_view = new QGraphicsView( scene );
  m_view->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
  m_view->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
  m_view->setVisible(true);
  m_view->setInteractive(true);
  scene->addItem(m_mapWidget);
  setCentralWidget(m_view);

  m_cacheProvider->start();

#ifdef EXPIRATION_DATE
  // create expiration timer if required
  if(QDate::fromString(EXPIRATION_DATE, "ddMMyyyy") <=
     QDate::currentDate()) {
    statusMessage(tr("This copy of CacheMe has expired and will close in one minute!"));
    QTimer::singleShot(60000, this, SLOT(close()));
  } else {
    statusMessage(tr("This copy of CacheMe will expire in %1 days!").
		  arg(QDate::currentDate().daysTo(
		  QDate::fromString(EXPIRATION_DATE, "ddMMyyyy"))));
  }
#endif  
}

MainWindow::~MainWindow() {
  qDebug() << __FUNCTION__;

  delete m_wptDb;
  delete m_cacheProvider;
  delete m_serviceProvider;
}

void MainWindow::resizeEvent(QResizeEvent*) {
  m_view->setSceneRect(QRect(QPoint(0,0), m_view->contentsRect().size()));
  m_mapWidget->resize(m_view->contentsRect().size());
}

void MainWindow::selectCache(const QString &name) {
  // ask CacheProvider for info about this cache
  this->m_cacheProvider->requestInfo(name);
}

void MainWindow::detailCache(const QString &name) {
  // ask CacheProvider for details about this cache
  this->m_cacheProvider->requestDetail(name);
}

void MainWindow::reloadCaches() {
  qDebug() << __FUNCTION__;

  // don't send a request if the map is already preparing a reload
  if(m_mapWidget->preparingReload())
    return;

  // request new caches for the maps new viewport
  m_cacheProvider->requestOverview(this->m_mapWidget->viewport(), m_lastFlags);
}

void MainWindow::reloadCaches(const int flags) {
  m_lastFlags = flags;

  // don't send a request if the map is already preparing a reload
  if(m_mapWidget->preparingReload())
    return;

  // request new caches for the maps new viewport
  m_cacheProvider->requestOverview(this->m_mapWidget->viewport(), flags);
}

void MainWindow::updateCaches(const CacheList &cacheList) {
  //  m_message = "";
  this->m_mapWidget->updateCaches(cacheList);
}

void MainWindow::statusMessage(const QString &message) {
#ifdef Q_WS_MAEMO_5
  QMaemo5InformationBox::information(this, message);
#else
  this->m_mapWidget->addBanner(message);
#endif
}

void MainWindow::displayError(const QString &message) {
  qDebug() << __FUNCTION__ << message;

  if(m_message != message) {
    statusMessage(message);
    m_message = message;
  }
}

void MainWindow::showDetail(const Cache &cache) {
  qDebug() << __FUNCTION__;

  CacheWindow *cacheWindow = 
    new CacheWindow(cache, m_locationProvider, m_wptDb, m_mapWidget, this);
  cacheWindow->show();
}

void MainWindow::setBusy(bool on) {
  static int busyCount = 0;

  if((!on && busyCount == 1) || ( on && busyCount <= 0)) {
#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, on);
#else
    m_mapWidget->setBusy(on);
#endif
  }

  if(on)                        busyCount++;
  else if(!on && busyCount > 0) busyCount--;
}

bool MainWindow::event(QEvent *event) {
  if(event->type() == QEvent::WindowActivate)
    emit activated(true);

  if(event->type() == QEvent::WindowDeactivate)
    emit activated(false);

  return CustomWindow::event(event);
}  
