'''
Advanced Power Abstract Wrappers
2008-2010(c) Kirill Plyashkevich <ru.spirit@gmail.com>
'''

import dbus
from advpowcommon.util.device import device
from advpowcommon.util.system import system

class AbstractWrapper():

  def __init__(self, bus, check = False):
    self.__bus = bus
    self.__dbus_signals = {}
    self.__ifaces = {}
    self.__available = False
    self.export_signals = {}
    self.export_interfaces = {}
    self.interfaces = {}
    self.interfaces_map = {}
    self.depends = {'os': '', 'packages': []}
    self.friendly_name = ''
    self.pre_init()
    self.__check_available()
    if not check:
      self.__end_init()

  def __end_init(self):
    if self.__available:
      self.__dbus_signals = {}
      #'NameOwnerChanged': {'source': (self.__dbus_name_owner_handler, 'NameOwnerChanged', 'org.freedesktop.DBus', '/org/freedesktop/DBus')}}
      for iface in self.interfaces_map: self.__ifaces[self.interfaces_map[iface][0]] = iface
      self.__init_interfaces()
      self.__init_dbus_signals()
      self.post_init()
      self.get_status()

  def pre_init(self):
    raise NotImplementedError("AbstractWrapper.pre_init() must be implemented")

  def post_init(self):
    raise NotImplementedError("AbstractWrapper.post_init() must be implemented")

  def get_status(self):
    raise NotImplementedError("AbstractWrapper.get_status() must be implemented")

  def __dbus_name_owner_handler(self, name, old, new):
    name = str(name)
    if name in self.__ifaces:
      if str(new):
        self.__init_interface(self.__ifaces[name])
      else:
        self.__destroy_interface(self.__ifaces[name])

  def _cleanup(self):
    self.__destroy_dbus_signals()
    self.__destroy_interfaces()

  def get_dep_packages(self):
    return self.dep_packages

  def get_name(self):
    return self.friendly_name

  def get_signals(self):
    return self.export_signals

  def is_available(self):
    return self.__available

  def __check_available(self):
    self.__available = True
    if self.depends['os']:
      if self.depends['os'] != device.get_property('OSSO_PRODUCT_RELEASE_VERSION'):
        self.__available = False
        return self.__available
    for dp in self.depends['packages']:
      if not system.is_package_version_valid(dp, self.depends['packages'][dp]):
        self.__available = False
        break
    return self.__available

  def custom_available(self):
    return True

  def __init_interfaces(self):
    for x in self.interfaces_map: self.__init_interface(x)

  def __init_interface(self, x):
    try:
      self.interfaces[x] = dbus.Interface(self.__bus.get_object(self.interfaces_map[x][0], self.interfaces_map[x][1], introspect=False, follow_name_owner_changes=True), self.interfaces_map[x][2])
    except:
      self.interfaces[x] = None

  def __destroy_interfaces(self):
    for key in self.interfaces: self.__destroy_interface(key)

  def __destroy_interface(self, x):
    self.interfaces[x] = None

  def __init_dbus_signals(self):
    for key in self.__dbus_signals:
      self.__dbus_signals[key]['receiver_match'] = self.__bus.add_signal_receiver(self.__dbus_signals[key]['source'][0], signal_name=self.__dbus_signals[key]['source'][1], dbus_interface=self.__dbus_signals[key]['source'][2], path=self.__dbus_signals[key]['source'][3])

  def __destroy_dbus_signals(self):
    for key in self.__dbus_signals:
      receiver_match = self.__dbus_signals[key]['receiver_match']
      self.__bus.clean_up_signal_match(receiver_match)
      self.__bus.remove_signal_receiver(receiver_match)
      receiver_match.remove()
