# -*- coding: utf-8 -*-
'''
MediaBox Widget UI Cairo Drawing
2010(c) Kirill Plyashkevich <ru.spirit@gmail.com>
'''

import cairo, gtk

class Cairo():
  pi = 3.1415926
  rgb_norm = 65535.0

  ALIGN_LEFT = 0
  ALIGN_CENTER = 0.5
  ALIGN_RIGHT = 1

  ALIGN_TOP = 0
  ALIGN_MIDDLE = 0.5
  ALIGN_BOTTOM = 1

  @staticmethod
  def draw_background(cr, has_alpha=False):
    if has_alpha:
      cr.set_source_rgba(1.0, 1.0, 1.0, 0.0)
    else:
      cr.set_source_rgb(1.0, 1.0, 1.0)
    cr.set_operator(cairo.OPERATOR_SOURCE)
    cr.paint()
    return cr

  @staticmethod
  def draw_rectangle(cr, size, bgcolor='#000000', has_alpha=False, alpha=0.5, draw_background=False):
    try:
      if draw_background:
        cr = Cairo.draw_background(cr, has_alpha)
      else:
        cr.set_operator(cairo.OPERATOR_SOURCE)
      width, height = size[0], size[1]
      cr.rectangle(0, 0, width, height)
      bg_color = gtk.gdk.color_parse(bgcolor)
      cr.set_source_rgba(bg_color.red / Cairo.rgb_norm, bg_color.green/Cairo.rgb_norm, bg_color.blue/Cairo.rgb_norm, alpha)
      cr.fill_preserve()
    except:
      pass
    return cr

  @staticmethod
  def draw_rounded_rectangle(cr, size, bgcolor='#000000', has_alpha=False, alpha=0.5, radius=10, draw_background=False):
    try:
      bg_color = gtk.gdk.color_parse(bgcolor)
      if draw_background:
        Cairo.draw_background(cr, has_alpha)
      else:
        cr.set_operator(cairo.OPERATOR_SOURCE)
      width, height = size[0], size[1]
      cr.move_to(0, radius)
      cr.arc(radius, radius, radius, Cairo.pi, 1.5 * Cairo.pi)
      cr.line_to(width - radius, 0)
      cr.arc(width - radius, radius, radius, 1.5 * Cairo.pi, 0.0)
      cr.line_to(width, height - radius)
      cr.arc(width - radius, height - radius, radius, 0.0, 0.5 * Cairo.pi)
      cr.line_to(radius, height)
      cr.arc(radius, height - radius, radius, 0.5 * Cairo.pi, Cairo.pi)
      cr.close_path()
      cr.set_source_rgba(bg_color.red / Cairo.rgb_norm, bg_color.green/Cairo.rgb_norm, bg_color.blue/Cairo.rgb_norm, alpha)
      cr.fill_preserve()
    except:
      pass
    return cr

  @staticmethod
  def draw_text(cr, text, rect_size, font_size, align=0.5, valign=0.5, text_color='#ffffff'):
    if cr and text:
      x_bearing, y_bearing, t_width, t_height, x_advance, y_advance = Cairo.get_text_size(cr, text, font_size)
      text_x = (rect_size[0] - t_width)*align
      text_y = (rect_size[1] - font_size)*valign - y_bearing
      Cairo.draw_text_at_pos(cr, text, font_size, text_x, text_y, text_color)
    return cr

  @staticmethod
  def draw_text_at_pos(cr, text, font_size, x=0, y=0, text_color='#ffffff'):
    if cr and text:
      txt_color = gtk.gdk.color_parse(text_color)
      cr.set_operator(cairo.OPERATOR_OVER)
      cr.set_font_size(font_size)
      cr.set_antialias(cairo.ANTIALIAS_GRAY)
      cr.move_to(x, y)
      cr.set_source_rgb(txt_color.red / Cairo.rgb_norm, txt_color.green/Cairo.rgb_norm, txt_color.blue/Cairo.rgb_norm)
      cr.show_text(text)
    return cr

  @staticmethod
  def draw_text_with_icon(cr, icon, text, rect_size, font_size, align=0.5, valign=0.5, text_color='#ffffff', padding=0):
    if cr:
      if icon:
        icon_x, icon_y, t_width, t_height, text_x, text_y = 0, (rect_size[1] - icon.get_height())/2, 0, 0, 0, 0
        if text:
          x_bearing, y_bearing, t_width, t_height, x_advance, y_advance = Cairo.get_text_size(cr, text, font_size)
        icon_x = (rect_size[0] - t_width - icon.get_width() - padding)*align
        icon_y = (rect_size[1] - icon.get_height())*valign
        if text:
          text_x = icon_x + icon.get_width() + padding
          text_y = (rect_size[1] - font_size)*valign - y_bearing
          cr = Cairo.draw_text_at_pos(cr, text, font_size, x=text_x, y=text_y, text_color=text_color)
        else:
          icon_x += padding*align
        cr = Cairo.draw_pixbuf(cr, icon, icon_x, icon_y)
      else:
        cr = Cairo.draw_text(cr, text, rect_size, font_size, align, valign, text_color)
    return cr

  @staticmethod
  def get_text_size(cr, text, font_size):
    cr.set_font_size(font_size)
    return cr.text_extents(text)

  @staticmethod
  def draw_pixbuf(cr, pixbuf, x=0, y=0):
    if pixbuf.get_has_alpha():
      cr.set_operator(cairo.OPERATOR_OVER)
    else:
      cr.set_operator(cairo.OPERATOR_ADD)
    cr.set_source_pixbuf(pixbuf, x, y)
    cr.paint()
    return cr

  @staticmethod
  def draw_frame(cr, x, y, width, height, frame_width=1, frame_color='#ffffff'):
    cr.set_operator(cairo.OPERATOR_ADD)
    cr.set_line_width(frame_width)
    fr_color = gtk.gdk.color_parse(frame_color)
    cr.set_source_rgb(fr_color.red / Cairo.rgb_norm, fr_color.green/Cairo.rgb_norm, fr_color.blue/Cairo.rgb_norm)
    cr.rectangle(x, y, width, height)
    cr.stroke_preserve()
    return cr
