/***************************************************************************
**
** Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (directui@nokia.com)
**
** This file is part of libmeegotouch.
**
** If you have questions regarding the use of this file, please contact
** Nokia at directui@nokia.com.
**
** This library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Lesser General Public
** License version 2.1 as published by the Free Software Foundation
** and appearing in the file LICENSE.LGPL included in the packaging
** of this file.
**
****************************************************************************/

#ifndef MSERVICEFWBASEIF_H
#define MSERVICEFWBASEIF_H

#include "mexport.h"

#include <QObject>
#include <QString>
#include "mservicefwproxy.h"

class QDBusAbstractInterface;
class MServiceFwBaseIfPrivate;

//! \internal

/*!
 * \class MServiceFwBaseIf
 * \author Max Waterman
 * \brief The base class from which all service interface classes are be derived.
 *
 * Service Interface classes are autogenerated and so there is no need
 * for developers to directly reference this class.
 *
 * MServiceFwBaseIf hides the servicefw machanisms from the derived class,
 * including service discovery and underlying IPC methods.
 */

class M_EXPORT MServiceFwBaseIf : public QObject
{
    Q_OBJECT
public:
    /*!
     * \brief Constructs a base interface
     * \param iface Interface to which to connect
     * \param parent Parent object
     */
    explicit MServiceFwBaseIf(const QString &iface, QObject *parent = 0);

    /*!
     * \brief Destructor
     */
    virtual ~MServiceFwBaseIf();

    /*!
     * \brief Returns true if the interface is connected and usable, otherwise false.
     * \return Interface validity
     */
    virtual bool isValid() const;

    /*!
     * \brief Returns the name of the currently used service
     * \return Name of currently used service
     */
    virtual QString serviceName() const;

    /*!
     * \brief Returns the address of the service framework proxy
     * \return Address of the service framework proxy
     */
    virtual MServiceFwProxy *serviceFwProxy();

    /*!
     * \brief Set the service name
     * \param service Name of the desired service
     */
    virtual void setService(const QString &service) = 0;

    /*!
     * \brief Returns the name of the interface
     * \return Name of the interface
     */
    QString interfaceName() const;

Q_SIGNALS:
    /*!
     * \brief A signal that is emitted when a service becomes available.
     * \param service The name of the service that has become available
     */
    void serviceAvailable(const QString &service);

    /*!
     * \brief A signal that is emitted when the last service implementing
     * the interface becomes unavailable.
     * \param service The name of the service that has become unavailable
     */
    void serviceUnavailable(const QString &service);

    /*!
     * \brief A signal that is emitted when the service has changed
     * \param service The name of the new service
     */
    void serviceChanged(const QString &service);

public Q_SLOTS:
    /*!
     * \brief Returns a list of available services
     * \param interface Only list services that implement this interface; empty for all interfaces
     * \return List of services
     */
    QStringList serviceNames(const QString &interface = "") const;

private Q_SLOTS:
    /*!
     * \brief Handles the ServicAvailable signal sent by the ServiceMapper
     * \param service The name of the service that has become available
     * \param interface The name of the interface implemented by the service
     *
     * Responsible for appropriately switching services and emitting signals
     * to allow the app to know what has happened and what has changed as a
     * result of a service becoming available.
     * Emits serviceAvailable signal if there was previously none available.
     * Emits serviceChanged signal if the service has changed.
     */
    void handleServiceAvailable(const QString &service, const QString &interface);

    /*!
     * \brief Handles the ServiceUnavailable signal sent by the ServiceMappper
     * \param service The name of the service that has become unavailable
     *
     * Responsible for appropriately switching services and emitting signals
     * to allow the app to know what has happened and what has changed as a
     * result of a service becoming unavailable.
     * Emits serviceUnavailable signal if there was previously none available.
     * Emits serviceChanged signal if the service has changed.
     */
    void handleServiceUnavailable(const QString &service);

protected:
    /*!
     * \brief Resolves the service name for a specific interface
     * \param ifName Interface name
     * \param preferredService Name of a preferred service
     * \return Name of selected service
     */
    QString resolveServiceName(const QString &ifName, const QString &preferredService);

    /*!
     * \brief Return the D-Bus interface proxy
     * \return D-Bus interface proxy
     */
    QDBusAbstractInterface *interfaceProxy() const;

    /*!
     * \brief Set the internally used D-Bus interface proxy
     * \param newInterfaceProxy new interface proxy
     *
     * If there is an existing D-Bus interface proxy, it will be deleted before
     * assigning this new one. Ownership is transferred to this class; it will
     * delete it in the destructor.
     */
    void setInterfaceProxy(QDBusAbstractInterface *newInterfaceProxy);

    /*!
     * \brief Set the address of the service framework proxy
     * \param newServiceFwProxy Address of the service framework proxy
     *
     * If there is an existing service framework proxy, it will be deleted
     * before assigning this new one. Ownership is transferred to this class;
     * it will delete it in the destructor.
     */
    void setServiceFwProxy(MServiceFwProxy *newServiceFwProxy);

    /*!
     * \brief Set the internally used service name
     * \param service the new service name
     */
    void setServiceName(const QString &service);

    /*!
     * \brief Set the internally used interface name
     * \param ifName new name of the interface
     */
    void setInterfaceName(const QString &ifName);

protected:
    MServiceFwBaseIfPrivate * const d_ptr;

private:
    Q_DECLARE_PRIVATE(MServiceFwBaseIf)
    Q_DISABLE_COPY(MServiceFwBaseIf)

#ifdef UNIT_TEST
    friend class Ut_MServiceFwBaseIf;
#endif // UNIT_TEST
};

//! \internal_end

#endif
