/*
 * This file is part of vncviewer.
 *
 * Copyright(C) 2005-2007, Aaron Levinson
 * Copyright(C) 2006-2007, Detlef Schmicker
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "interface.h"
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <libintl.h>
#include <unistd.h>
#include <string.h>

#include <hildon-1/hildon/hildon-program.h>
#include <hildon-1/hildon/hildon-note.h>
#include <hildon-1/hildon/hildon-get-password-dialog.h>
//#include <hildon-1/hildon/hildon-name-password-dialog.h>
#include <hildon-1/hildon/hildon-banner.h>


/*
// !!!!!!!!!!!!!!!!! Fake code from Tomas Junnonen, nokia
#include <X11/extensions/XTest.h>

#define SELECT_KEY_LONG_PRESS 1000
#define FINGER_PRESS_BUTTON 8

guint timeout_id;

static gboolean
select_key_long_press(GdkWindow * window)
{
  //  GtkWidget *entry = GTK_WIDGET(data);
  gint x=100, y=100, width, height;
  g_message("Fake 1a");

  //  gdk_window_get_origin(window, &x, &y);
  //g_message("Fake 1b");
  //gdk_drawable_get_size(window, &width, &height);
  //g_message("Fake 1c");
  //if (width && height)
  {
  g_message("Fake 1d");
    XTestFakeMotionEvent(GDK_DISPLAY(), -1, 
                         x + (width / 2), y + (height / 2), 0);
  g_message("Fake 2");
    XTestFakeButtonEvent(GDK_DISPLAY(), FINGER_PRESS_BUTTON, TRUE, 0);
  g_message("Fake 3");
    XTestFakeButtonEvent(GDK_DISPLAY(), FINGER_PRESS_BUTTON, FALSE, 0);
  g_message("Fake 4");
  }

  timeout_id = 0;
  return FALSE;
}

//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!111
*/


#include "rfb.h"

#define MAIN_TEXT MAIN_VIEW_NAME

// for custom version of maemo_af_desktop that allows the home button
// to be disabled
#define TOGGLE_HOME_SERVICE "org.maemo.togglehome"
#define TOGGLE_HOME_INTERFACE "org.maemo.togglehome.signal"
#define TOGGLE_HOME_PATH "/org/maemo/togglehome/signal"
#define MESSAGE_ENABLE_HOME_BUTTON "enable_home_button"
#define MESSAGE_DISABLE_HOME_BUTTON "disable_home_button"

extern gchar *arg_hostname;
extern gchar *arg_password;
extern gboolean arg_have_password;
extern gboolean arg_lockdown;
extern gchar *arg_lockdown_password;


/* Privates: */

void create_menu(void *mainvw);
void disable_home_key_lock(void *mainvw);
static void create_toolbar(MainView *mainview);
static void callback_fullscreen(GtkBin *action, gpointer data);
static gboolean callback_keypress(GtkWidget *widget, GdkEventKey *event,
				  gpointer userdata);
static gboolean callback_keyrelease(GtkWidget *widget, GdkEventKey *event,
				  gpointer userdata);

static void scrolledwin_size_allocate(GtkWidget *widget,
				      GtkAllocation *allocation,
				      gpointer userdata);

static void scrolledwin2_size_request(GtkWidget *widget,
				      GtkRequisition *requisition,
				      gpointer userdata);

static gboolean mainview_auto_connect(gpointer userdata);
static gboolean make_connection(const gchar *display, gboolean shared,
				VncViewerDepthType depth,
				VncViewerRotationType rotationType,
				VncViewerEncodingType encodingType,
				MainView *mainview);
static gboolean app_key_snooper(GtkWidget *widget, GdkEventKey *keyevent,
				HildonProgram *app);
static void about_dialog_url_hook_function(GtkAboutDialog *about,
					   const gchar *link,
					   gpointer data);

// Creates and initializes a main_view
MainView* interface_main_view_new(AppData *data)
{
  m_button_pressed=FALSE;
  p_button_pressed=FALSE;
//fremantle
  pm_button_for_toggle_fullscreen=FALSE;

    // Zero memory with g_new0
    MainView* result = g_new0(MainView, 1);
    GtkWidget *main_vbox = gtk_vbox_new(FALSE, 0);

    result->directionKeyPress = 0;
    result->bMultiplePresses = FALSE;

    result->m_bCtrlOn = FALSE;
    result->m_bAltOn = FALSE;
    result->m_bShiftOn = FALSE;

    result->file_item = 0;
    result->fullscreen_item = 0;

    result->fullscreen_tb = 0;

    g_type_init();
    result->gc_client = gconf_client_get_default();

    // get persisted settings

    result->m_hostList = gconf_client_get_list(result->gc_client,
					       MAEMO_VNCVIEWER_GCONF_HOSTS,
					       GCONF_VALUE_STRING, 0);

    // if this one isn't found, it's okay, because we want to start with
    // a default of FALSE
    result->default_opts.m_bShared =
      gconf_client_get_bool(result->gc_client, MAEMO_VNCVIEWER_GCONF_SHARED, 0);
    result->default_opts.m_nDepth =
      (VncViewerDepthType) gconf_client_get_int(result->gc_client,
						MAEMO_VNCVIEWER_GCONF_DEPTH, 0);
    // if not found, 0 is also the default
    result->default_opts.m_nRotationType =
      (VncViewerDepthType) gconf_client_get_int(result->gc_client,
						MAEMO_VNCVIEWER_GCONF_ROTATION, 0);

    // if not found, 0 is also the default
    result->default_opts.m_nEncodingType =
      (VncViewerDepthType) gconf_client_get_int(result->gc_client,
						MAEMO_VNCVIEWER_GCONF_ENCODING, 0);
    // i.e. if not found, use the default
    if (!result->default_opts.m_nDepth)
      result->default_opts.m_nDepth = VNC_VIEWER_DEPTH_DEFAULT;
    else if (result->default_opts.m_nDepth == VNC_VIEWER_DEPTH_24BIT)
      // 24 bit is no longer supported
      result->default_opts.m_nDepth = VNC_VIEWER_DEPTH_16BIT;

    // using ! to allow a missing gconf value to work properly
    // avoids needing to examine a GError
    result->bShowNormalToolbar =
      !gconf_client_get_bool(result->gc_client,
			     MAEMO_VNCVIEWER_GCONF_NORMAL_TOOLBAR, 0);
    if (arg_lockdown)
      result->bShowFullscreenToolbar = FALSE;
    else
      result->bShowFullscreenToolbar =
	!gconf_client_get_bool(result->gc_client,
			       MAEMO_VNCVIEWER_GCONF_FULLSCREEN_TOOLBAR, 0);
//fremantle force FullscreenToolbar
// removed again
//    result->bShowFullscreenToolbar = TRUE;

    result->bShowScrollbars =
      !gconf_client_get_bool(result->gc_client,
			     MAEMO_VNCVIEWER_GCONF_SHOW_SCROLLBARS, 0);

    // the default is FALSE for the fullscreen setting
    if (arg_lockdown)
      result->fullscreen = TRUE;
    else
      result->fullscreen =
	gconf_client_get_bool(result->gc_client,
			      MAEMO_VNCVIEWER_GCONF_FULLSCREEN, 0);

    result->m_bHaveHomeKeyLock = FALSE;
    result->m_nKeySnooperId = 0;

    // disable home button (if maemo_af_desktop has been replaced)
    if (arg_lockdown)
    {
      osso_return_t ret = 
	osso_rpc_run(data->osso, TOGGLE_HOME_SERVICE, TOGGLE_HOME_PATH,
		     TOGGLE_HOME_INTERFACE, MESSAGE_DISABLE_HOME_BUTTON,
		     NULL, DBUS_TYPE_INVALID);
      if (ret == OSSO_OK)
	result->m_bHaveHomeKeyLock = TRUE;
    }

    gtk_about_dialog_set_url_hook(about_dialog_url_hook_function,
				  data->osso, 0);

    // Store handle to app's data to view's data
    result->data = data;
    // Store handle to HildonWindow to app's data
    result->data->main_view = HILDON_WINDOW(hildon_window_new());
    // Create all our necessary parts
    create_toolbar(result);
    create_menu((void *) result);
    result->scrolledwin = gtk_scrolled_window_new(0, 0);
    g_signal_connect(G_OBJECT(result->scrolledwin), "size-allocate",
    		     G_CALLBACK(scrolledwin_size_allocate), result);

    // don't want to see scrollbars unless connected to a VNC server
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(result->scrolledwin),
				   GTK_POLICY_NEVER, GTK_POLICY_NEVER);
    gtk_container_add(GTK_CONTAINER(result->data->main_view),
		      main_vbox);

    // connecting to these signals for some things that need to be
    // handled at this level.  Most key press and release events
    // are handled by the VncViewer widget 
    g_signal_connect(G_OBJECT(result->data->main_view), "key_press_event",
		     G_CALLBACK(callback_keypress), result);
    g_signal_connect(G_OBJECT(result->data->main_view), "key_release_event",
		     G_CALLBACK(callback_keyrelease), result);

    gtk_box_pack_start(GTK_BOX(main_vbox), result->scrolledwin, TRUE, TRUE, 0);
    gtk_widget_show(main_vbox);
    // need to do a show all first, in order to get everything
    // displayed--after doing that, the toolbar may be disabled
    gtk_widget_show_all(GTK_WIDGET(result->data->main_view));

    if (result->fullscreen)
    {
      // switch back so that callback_fullscreen works properly
      result->fullscreen = FALSE;
      callback_fullscreen(0, result);
    }
    else
    {
      if (!result->bShowNormalToolbar)
	gtk_widget_hide_all(GTK_WIDGET(result->toolbar));
    }

    result->vnc = 0;

    // adding an idle function, the purpose of which is to cause the
    // connect dialog to be displayed after the user interface is up
    g_idle_add((GSourceFunc) mainview_auto_connect, result);

    return result;
}

/* clean up the allocated memory */
void interface_main_view_destroy(MainView *mainview)
{
  if (mainview->keys_menu)
    g_object_unref(mainview->keys_menu);

  if (mainview->m_hostList)
  {
    // persist host list
    GSList *slist = mainview->m_hostList;
    gconf_client_set_list(mainview->gc_client, MAEMO_VNCVIEWER_GCONF_HOSTS,
			  GCONF_VALUE_STRING, slist, 0);

    while (slist)
    {
      g_free(slist->data);
      slist = slist->next;
    }
    g_slist_free(mainview->m_hostList);
  }

  // persist other settings
  gconf_client_set_bool(mainview->gc_client, MAEMO_VNCVIEWER_GCONF_SHARED,
			mainview->default_opts.m_bShared, 0);
  gconf_client_set_int(mainview->gc_client, MAEMO_VNCVIEWER_GCONF_DEPTH,
		       (gint) mainview->default_opts.m_nDepth, 0);
  gconf_client_set_int(mainview->gc_client, MAEMO_VNCVIEWER_GCONF_ROTATION,
		       (gint) mainview->default_opts.m_nRotationType, 0);
  gconf_client_set_int(mainview->gc_client, MAEMO_VNCVIEWER_GCONF_ENCODING,
		       (gint) mainview->default_opts.m_nEncodingType, 0);

  // using ! as per comments in interface_main_view_new()
  gconf_client_set_bool(mainview->gc_client,
			MAEMO_VNCVIEWER_GCONF_NORMAL_TOOLBAR,
			!mainview->bShowNormalToolbar, 0);
  gconf_client_set_bool(mainview->gc_client,
			MAEMO_VNCVIEWER_GCONF_FULLSCREEN_TOOLBAR,
			!mainview->bShowFullscreenToolbar, 0);
  gconf_client_set_bool(mainview->gc_client,
			MAEMO_VNCVIEWER_GCONF_SHOW_SCROLLBARS,
			!mainview->bShowScrollbars, 0);

  gconf_client_set_bool(mainview->gc_client, MAEMO_VNCVIEWER_GCONF_FULLSCREEN,
			mainview->fullscreen, 0);

  g_object_unref(mainview->gc_client);

  g_free(mainview);
}

void interface_vnc_passwd(MainView *mainview, gboolean bUserDisabled,
			  const gchar *secType, gchar **user,
			  gchar **passwd)
{
  GtkWidget *dialog =
    hildon_get_password_dialog_new(GTK_WINDOW(mainview->data->main_view), FALSE);
  //gchar *titleStr = g_strdup_printf("VNC Authentication [%s]", secType);
  //gtk_window_set_title(GTK_WINDOW(dialog), titleStr);
  //gtk_window_set_title(GTK_WINDOW(dialog), "VNC Authentication");
  //g_free(titleStr);

  if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK)
  {
    HildonGetPasswordDialog *hgpd = HILDON_GET_PASSWORD_DIALOG(dialog);

    *passwd = g_strdup(hildon_get_password_dialog_get_password(hgpd));

    // since these strings are returned, they need to be duplicated
  }

  gtk_widget_destroy(dialog);
}

/* Privates */
void options_button_clicked(GtkButton *button,
			    gpointer data)
{
  GtkWidget *parent_dialog = GTK_WIDGET(data);
  GtkWidget *dialog, *shared_button;
  GtkWidget *scrolled_window;
  GtkWidget *vbox, *frame, *main_vbox;
  GtkToggleButton *default_radio, *eight_bit_radio, *sixteen_bit_radio;
  GtkToggleButton *toggle = 0;

  dialog = gtk_dialog_new_with_buttons("VNC Viewer: Options",
				       GTK_WINDOW(parent_dialog),
				       (GtkDialogFlags) (GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT),
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
				       0);

  gtk_dialog_set_has_separator(GTK_DIALOG(dialog), TRUE);
  gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_ACCEPT);

  scrolled_window = gtk_scrolled_window_new(0, 0);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
				 GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), scrolled_window);
  main_vbox = gtk_vbox_new(FALSE, 0);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window),
  					main_vbox);
  g_signal_connect(G_OBJECT(scrolled_window), "size-request",
		   G_CALLBACK(scrolledwin2_size_request), main_vbox);

  options_settings *pOpts;
  pOpts = (options_settings *) g_object_get_data(G_OBJECT(parent_dialog),
						 "options");

  shared_button = gtk_check_button_new_with_mnemonic("_Shared connection (do not disconnect other viewers)");
  //gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), shared_button);
  gtk_box_pack_start(GTK_BOX(main_vbox), shared_button, TRUE, TRUE, 5);
  //gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), shared_button,
  //	     TRUE, TRUE, 5);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(shared_button),
			       pOpts->m_bShared);

  frame = gtk_frame_new("Display depth");
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
  //gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), frame);
  gtk_box_pack_start(GTK_BOX(main_vbox), frame, TRUE, TRUE, 5);
  //gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), frame,
  //		     TRUE, TRUE, 5);

  vbox = gtk_vbox_new(TRUE, 2);
  //gtk_button_box_set_layout(GTK_BUTTON_BOX(vbox), GTK_BUTTONBOX_EDGE);
  gtk_container_add(GTK_CONTAINER(frame), vbox);
  //gtk_container_set_border_width(GTK_CONTAINER(vbox), 1);

  default_radio = GTK_TOGGLE_BUTTON(gtk_radio_button_new_with_mnemonic(0, "Server _Default"));
  eight_bit_radio =
    GTK_TOGGLE_BUTTON(gtk_radio_button_new_with_mnemonic_from_widget(GTK_RADIO_BUTTON(default_radio), "_8-bit"));
  sixteen_bit_radio =
    GTK_TOGGLE_BUTTON(gtk_radio_button_new_with_mnemonic_from_widget(GTK_RADIO_BUTTON(eight_bit_radio), "_16-bit"));

  gtk_container_add(GTK_CONTAINER(vbox), GTK_WIDGET(default_radio));
  gtk_container_add(GTK_CONTAINER(vbox), GTK_WIDGET(eight_bit_radio));
  gtk_container_add(GTK_CONTAINER(vbox), GTK_WIDGET(sixteen_bit_radio));

  switch (pOpts->m_nDepth)
  {
  case VNC_VIEWER_DEPTH_DEFAULT:
    toggle = default_radio;
    break;
  
  case VNC_VIEWER_DEPTH_8BIT:
    toggle = eight_bit_radio;
    break;
  
  case VNC_VIEWER_DEPTH_16BIT:
    toggle = sixteen_bit_radio;
    break;
  
  default:
    break;
  }

  gtk_toggle_button_set_active(toggle, TRUE);

  // encoding
  GtkWidget *encoding_frame = gtk_frame_new("Preferred encoding");
  gtk_frame_set_shadow_type(GTK_FRAME(encoding_frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start(GTK_BOX(main_vbox), encoding_frame, TRUE, TRUE, 5);
  //gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), encoding_frame,
  //		     TRUE, TRUE, 5);

  GtkWidget *encoding_vbox = gtk_vbox_new(TRUE, 2);
  gtk_container_add(GTK_CONTAINER(encoding_frame), encoding_vbox);

  GtkToggleButton *auto_encoding_radio =
    GTK_TOGGLE_BUTTON(gtk_radio_button_new_with_mnemonic(0, "Automatically _select"));
  GtkToggleButton *raw_encoding_radio =
    GTK_TOGGLE_BUTTON(gtk_radio_button_new_with_mnemonic_from_widget(GTK_RADIO_BUTTON(auto_encoding_radio), "Ra_w"));
  GtkToggleButton *hextile_encoding_radio =
    GTK_TOGGLE_BUTTON(gtk_radio_button_new_with_mnemonic_from_widget(GTK_RADIO_BUTTON(raw_encoding_radio), "_Hextile"));

  gtk_container_add(GTK_CONTAINER(encoding_vbox),
		    GTK_WIDGET(auto_encoding_radio));
  gtk_container_add(GTK_CONTAINER(encoding_vbox),
		    GTK_WIDGET(raw_encoding_radio));
  gtk_container_add(GTK_CONTAINER(encoding_vbox),
		    GTK_WIDGET(hextile_encoding_radio));

  switch (pOpts->m_nEncodingType)
  {
  case VNC_VIEWER_ENCODING_AUTO:
    toggle = auto_encoding_radio;
    break;
  
  case VNC_VIEWER_ENCODING_RAW:
    toggle = raw_encoding_radio;
    break;
  
  case VNC_VIEWER_ENCODING_HEXTILE:
    toggle = hextile_encoding_radio;
    break;
  
  default:
    break;
  }

  gtk_toggle_button_set_active(toggle, TRUE);


  // rotation
  GtkWidget *rotation_frame = gtk_frame_new("Rotation type");
  gtk_frame_set_shadow_type(GTK_FRAME(rotation_frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start(GTK_BOX(main_vbox), rotation_frame, TRUE, TRUE, 5);
  //gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), rotation_frame,
  //		     TRUE, TRUE, 5);

  GtkWidget *rotation_vbox = gtk_vbox_new(TRUE, 2);
  gtk_container_add(GTK_CONTAINER(rotation_frame), rotation_vbox);

  GtkToggleButton *auto_rotate_radio =
    GTK_TOGGLE_BUTTON(gtk_radio_button_new_with_mnemonic(0, "_Automatically rotate"));
  GtkToggleButton *always_rotate_radio =
    GTK_TOGGLE_BUTTON(gtk_radio_button_new_with_mnemonic_from_widget(GTK_RADIO_BUTTON(auto_rotate_radio), "Always _rotate"));
  GtkToggleButton *never_rotate_radio =
    GTK_TOGGLE_BUTTON(gtk_radio_button_new_with_mnemonic_from_widget(GTK_RADIO_BUTTON(always_rotate_radio), "_Never rotate"));

  gtk_container_add(GTK_CONTAINER(rotation_vbox),
		    GTK_WIDGET(auto_rotate_radio));
  gtk_container_add(GTK_CONTAINER(rotation_vbox),
		    GTK_WIDGET(always_rotate_radio));
  gtk_container_add(GTK_CONTAINER(rotation_vbox),
		    GTK_WIDGET(never_rotate_radio));

  switch (pOpts->m_nRotationType)
  {
  case VNC_VIEWER_ROTATE_AUTO:
    toggle = auto_rotate_radio;
    break;
  
  case VNC_VIEWER_ROTATE_ALWAYS:
    toggle = always_rotate_radio;
    break;
  
  case VNC_VIEWER_ROTATE_NEVER:
    toggle = never_rotate_radio;
    break;
  
  default:
    break;
  }

  gtk_toggle_button_set_active(toggle, TRUE);

  gtk_widget_show_all(dialog);

  if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT)
  {
    pOpts->m_bShared =
      gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(shared_button));

    if (gtk_toggle_button_get_active(default_radio))
      pOpts->m_nDepth = VNC_VIEWER_DEPTH_DEFAULT;
    else if (gtk_toggle_button_get_active(eight_bit_radio))
      pOpts->m_nDepth = VNC_VIEWER_DEPTH_8BIT;
    else if (gtk_toggle_button_get_active(sixteen_bit_radio))
      pOpts->m_nDepth = VNC_VIEWER_DEPTH_16BIT;

    if (gtk_toggle_button_get_active(auto_rotate_radio))
      pOpts->m_nRotationType = VNC_VIEWER_ROTATE_AUTO;
    else if (gtk_toggle_button_get_active(always_rotate_radio))
      pOpts->m_nRotationType = VNC_VIEWER_ROTATE_ALWAYS;
    else if (gtk_toggle_button_get_active(never_rotate_radio))
      pOpts->m_nRotationType = VNC_VIEWER_ROTATE_NEVER;

    if (gtk_toggle_button_get_active(auto_encoding_radio))
      pOpts->m_nEncodingType = VNC_VIEWER_ENCODING_AUTO;
    else if (gtk_toggle_button_get_active(raw_encoding_radio))
      pOpts->m_nEncodingType = VNC_VIEWER_ENCODING_RAW;
    else if (gtk_toggle_button_get_active(hextile_encoding_radio))
      pOpts->m_nEncodingType = VNC_VIEWER_ENCODING_HEXTILE;
  }

  gtk_widget_destroy(dialog);
}

void host_entry_changed(GtkEditable *editable,
			gpointer     data)
{
  GtkDialog *dialog = GTK_DIALOG(data);
  GtkEntry *entry = GTK_ENTRY(editable);
  const gchar *entry_text = gtk_entry_get_text(entry);
  // enable/disable the OK button based on whether or not this field
  // is empty.  i.e., the user must specify a host name
  if (entry_text && (entry_text[0] != '\0'))
    gtk_dialog_set_response_sensitive(dialog, GTK_RESPONSE_ACCEPT, TRUE);
  else
    gtk_dialog_set_response_sensitive(dialog, GTK_RESPONSE_ACCEPT, FALSE);
}

static void vnc_connection_closed(MainView *mainview)
{
  gtk_widget_destroy(GTK_WIDGET(mainview->vnc));
  mainview->vnc = 0;

  // don't want scrollbars to show up anymore, not connected anymore
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(mainview->scrolledwin),
				 GTK_POLICY_NEVER, GTK_POLICY_NEVER);

  //gtk_window_set_title(GTK_WINDOW(mainview->data->main_view), "");

  if (mainview->m_bCtrlOn)
  {
    gtk_check_menu_item_set_active(mainview->ctrl_item, FALSE);
    mainview->m_bCtrlOn = FALSE;
  }

  if (mainview->m_bAltOn)
  {
    gtk_check_menu_item_set_active(mainview->alt_item, FALSE);
    mainview->m_bAltOn = FALSE;
  }

  if (mainview->m_bShiftOn)
  {
    gtk_check_menu_item_set_active(mainview->shift_item, FALSE);
    mainview->m_bShiftOn = FALSE;
  }

  if (!arg_lockdown)
    gtk_widget_set_sensitive(mainview->disconnect_item, FALSE);

  gtk_widget_set_sensitive(GTK_WIDGET(mainview->disconnect_tb), FALSE);
  gtk_widget_set_sensitive(GTK_WIDGET(mainview->im_tb), FALSE);
  gtk_widget_set_sensitive(GTK_WIDGET(mainview->keys_tb), FALSE);

  // this is a kludge to prevent the scrollbar from remaining in full screen
  // mode--the scrollbar really isn't there.  Somehow, the widget isn't being
  // invalidated properly.
  gtk_widget_hide_all(mainview->scrolledwin);
}

static void vnc_connection_died(VncViewer *vnc, gpointer data) {
  MainView *mainview = (MainView *) data;
  GtkWidget *msgDialog;

  if (arg_lockdown)
    msgDialog = hildon_note_new_information(GTK_WINDOW(mainview->data->main_view), "Connection to server lost.  Contact the administrator or click OK to attempt to connect again.");
  else
    msgDialog =
      hildon_note_new_information(GTK_WINDOW(mainview->data->main_view),
				    "Connection to server lost");
  gtk_dialog_run(GTK_DIALOG(msgDialog));
  gtk_widget_destroy(msgDialog);

  vnc_connection_closed(mainview);

  // try to reconnect if we have a password and hostname and the previous
  // hostname is equal to the inputted hostname
  if (arg_hostname && arg_have_password &&
      (strcmp(arg_hostname,
	      (gchar *) g_slist_nth_data(mainview->m_hostList, 0)) == 0))
  {
    while (1)
    {
      if (make_connection((const gchar *) arg_hostname,
			  mainview->default_opts.m_bShared,
			  mainview->default_opts.m_nDepth,
			  mainview->default_opts.m_nRotationType,
			  mainview->default_opts.m_nEncodingType,
			  mainview) ||
	  !arg_lockdown)
	break;
    }
  }
}

static gboolean make_connection(const gchar *display, gboolean shared,
				VncViewerDepthType depth,
				VncViewerRotationType rotationType,
				VncViewerEncodingType encodingType,
				MainView *mainview)
{
  gint fd;
  gint32 res, ret;
  gchar *msg = 0;
  gchar *tmp;
  //GdkGeometry geometry;
  //GdkWindowHints geom_mask;
  GtkWidget *msgDialog;
  GtkAdjustment *hadj, *vadj;

  // Display an information, as connecting may take long, if the wlan is not yet connected
  GtkWidget * banner =hildon_banner_show_animation(GTK_WIDGET(mainview->data->main_view),
				 "Connecting...","Connecting...");
  // Wait a while for the xwindows system having handled the banner window
  int tmpi=0;
  while (tmpi++<800)
    gtk_main_iteration_do(FALSE);

  fd = connectToServer(display);

  //connection ready (or not) connecting banner can disapear
  gtk_widget_destroy(GTK_WIDGET(banner));

  if (fd < 0 || (res = doVersionHandshake(fd)) < 0)
  {
    if (arg_lockdown)
    {
      msgDialog = hildon_note_new_information(GTK_WINDOW(mainview->data->main_view), "Unable to connect to server.  Contact the administrator or click OK to attempt to connect again.");
      GtkWidget *adminButton = gtk_dialog_add_button(GTK_DIALOG(msgDialog),
						     GTK_STOCK_PREFERENCES, 1);
      gtk_button_set_label(GTK_BUTTON(adminButton), "Admin...");
    }
    else
      msgDialog = hildon_note_new_information(GTK_WINDOW(mainview->data->main_view),
					      "Could not connect to server");
    gint nResponse = gtk_dialog_run(GTK_DIALOG(msgDialog));
    gtk_widget_destroy(msgDialog);

    if (fd >= 0)
      close(fd);

    if (arg_lockdown && (nResponse == 1))
    {
	if (arg_lockdown_password && strlen(arg_lockdown_password))
	{
	  GtkWidget *dialog =
	    hildon_get_password_dialog_new(GTK_WINDOW(mainview->data->main_view),
					   FALSE);
	  gtk_widget_show(dialog);
	  gint nResponse = gtk_dialog_run(GTK_DIALOG(dialog));
	  if (nResponse == GTK_RESPONSE_OK)
	  {
	    HildonGetPasswordDialog *passDialog =
	      HILDON_GET_PASSWORD_DIALOG(dialog);
	    if (strcmp(arg_lockdown_password,
		       hildon_get_password_dialog_get_password(passDialog)) == 0)
	    {
	      arg_lockdown = FALSE;
	      create_menu(mainview);
	      disable_home_key_lock(mainview);
	    }
	    else
	    {
	      msgDialog =
		hildon_note_new_information(GTK_WINDOW(mainview->data->main_view),
					    "Password incorrect");
	      gtk_dialog_run(GTK_DIALOG(msgDialog));
	      gtk_widget_destroy(msgDialog);
	    }
	  }

	  gtk_widget_destroy(dialog);
	}
	else
	{
	  arg_lockdown = FALSE;
	  create_menu(mainview);
	  disable_home_key_lock(mainview);
	}
    }

    return FALSE;
  }

  switch (res)
  {
  case rfbConnFailed:
    tmp = getFailReason(fd);
    msg = g_strconcat("Connection failed: ", tmp, NULL);
    g_free(tmp);

    msgDialog = hildon_note_new_information(GTK_WINDOW(mainview->data->main_view),
					    msg);
    gtk_dialog_run(GTK_DIALOG(msgDialog));
    gtk_widget_destroy(msgDialog);

    g_free(msg);
    close(fd);
    return FALSE;

  case rfbNoAuth:
    break;

  case rfbVncAuth:
    if (arg_hostname && arg_have_password &&
	(strcmp(arg_hostname, display) == 0))
      ret = performAuth(fd, arg_password ? arg_password : "");
    else
    {
      interface_vnc_passwd(mainview, TRUE, "VncAuth", 0, &msg);
      if (!msg)
      {
	close(fd);
	return FALSE;
      }

      ret = performAuth(fd, msg);
      g_free(msg);
    }

    switch (ret)
    {
    case rfbVncAuthOK:
      break;

    case rfbVncAuthFailed:
      msgDialog =
	hildon_note_new_information(GTK_WINDOW(mainview->data->main_view),
				    "Authentication to VNC server failed");
      gtk_dialog_run(GTK_DIALOG(msgDialog));
      gtk_widget_destroy(msgDialog);

      close(fd);

      return FALSE;

    case rfbVncAuthTooMany:
      msgDialog =
	hildon_note_new_information(GTK_WINDOW(mainview->data->main_view),
				    "Too many authentication failures occurred");
      gtk_dialog_run(GTK_DIALOG(msgDialog));
      gtk_widget_destroy(msgDialog);

      close(fd);

      return FALSE;
    }

    break;

  default:
    msgDialog = hildon_note_new_information(GTK_WINDOW(mainview->data->main_view),
					    "Unknown error");
    gtk_dialog_run(GTK_DIALOG(msgDialog));
    gtk_widget_destroy(msgDialog);

    //close(fd);

    return FALSE;
  }

  // create VncViewer widget
  mainview->vnc = VNC_VIEWER(vnc_viewer_new(fd, shared, depth, encodingType));
  mainview->vnc->mainview = (void *) mainview;

  switch (rotationType)
  {
  case VNC_VIEWER_ROTATE_AUTO:
    if (mainview->vnc->height > mainview->vnc->width)
      mainview->vnc->m_bRotateDisplay = TRUE;
    else
      mainview->vnc->m_bRotateDisplay = FALSE;
    break;

  case VNC_VIEWER_ROTATE_ALWAYS:
    mainview->vnc->m_bRotateDisplay = TRUE;
    break;

  case VNC_VIEWER_ROTATE_NEVER:
    mainview->vnc->m_bRotateDisplay = FALSE;
    break;

  default:
    // shouldn't get here
    mainview->vnc->m_bRotateDisplay = FALSE;
    break;
  }

  vnc_viewer_set_cursor(mainview->vnc, VNC_VIEWER_CURSOR_DOT);
  g_signal_connect(G_OBJECT(mainview->vnc), "connection_died",
		   G_CALLBACK(vnc_connection_died), mainview);

  // make the size of the VncViewer widget the size returned by the server
  if (!mainview->vnc->m_bRotateDisplay)
  {
    gtk_widget_set_size_request(GTK_WIDGET(mainview->vnc),
				mainview->vnc->width,
				mainview->vnc->height);
  }
  else
  {
    gtk_widget_set_size_request(GTK_WIDGET(mainview->vnc),
				mainview->vnc->height,
				mainview->vnc->width);
  }

  // set up adjustments for the scrolled window
  hadj = gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(mainview->scrolledwin));
  if (!mainview->vnc->m_bRotateDisplay)
  {
    hadj->lower = 0;
    hadj->upper = mainview->vnc->width;
  }
  else
  {
    hadj->lower = mainview->vnc->height;
    hadj->upper = 0;
  }
  hadj->value = 0;
  hadj->step_increment = 20;
  if (!mainview->vnc->m_bRotateDisplay)
    hadj->page_increment = GTK_WIDGET(mainview->scrolledwin)->allocation.width;
  else
    hadj->page_increment = GTK_WIDGET(mainview->scrolledwin)->allocation.height;
  hadj->page_size = hadj->page_increment;
  gtk_adjustment_changed(hadj);
  
  vadj = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(mainview->scrolledwin));
  vadj->lower = 0;
  if (!mainview->vnc->m_bRotateDisplay)
    vadj->upper = mainview->vnc->height;
  else
    vadj->upper = mainview->vnc->width;
  vadj->value = 0;
  vadj->step_increment = 20;
  if (!mainview->vnc->m_bRotateDisplay)
    vadj->page_increment = GTK_WIDGET(mainview->scrolledwin)->allocation.height;
  else
    vadj->page_increment = GTK_WIDGET(mainview->scrolledwin)->allocation.width;
  vadj->page_size = vadj->page_increment;
  gtk_adjustment_changed(vadj);
  
  // use a viewport with the scrolled window and VncViewer widget
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(mainview->scrolledwin),
					GTK_WIDGET(mainview->vnc));
  gtk_widget_show_all(mainview->scrolledwin);
  //gtk_widget_show(GTK_WIDGET(mainview->vnc));
  gtk_widget_grab_focus(GTK_WIDGET(mainview->vnc));

  if (mainview->bShowScrollbars)
    // cause the scrollbars to be displayed, if necessary
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(mainview->scrolledwin),
				   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  // no good way to update title after disconnecting from server
  // so no longer putting the host name in the title
  //gtk_window_set_title(GTK_WINDOW(mainview->data->main_view), mainview->vnc->desk_name);

  //g_free(msg);

  if (!arg_lockdown)
    gtk_widget_set_sensitive(mainview->disconnect_item, TRUE);

  gtk_widget_set_sensitive(GTK_WIDGET(mainview->disconnect_tb), TRUE);
  gtk_widget_set_sensitive(GTK_WIDGET(mainview->im_tb), TRUE);
  gtk_widget_set_sensitive(GTK_WIDGET(mainview->keys_tb), TRUE);

  //gtk_window_set_resizable(GTK_WINDOW(mainview->data->main_view), FALSE);

  return TRUE;
}

static void callback_connect(GtkBin *action, gpointer data)
{
  MainView *mainview = (MainView *) data;
  GtkWidget *dialog, *hbox, *host_label, *options_button;
  GtkComboBox *host_entry_combo;
  GtkEntry *host_entry;

  dialog = gtk_dialog_new_with_buttons("VNC Viewer: Connection Details",
				       GTK_WINDOW(mainview->data->main_view),
				       (GtkDialogFlags) (GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT),
				       0);
  options_settings opts;
  opts.m_bShared = mainview->default_opts.m_bShared;
  opts.m_nDepth = mainview->default_opts.m_nDepth;
  opts.m_nRotationType = mainview->default_opts.m_nRotationType;
  opts.m_nEncodingType = mainview->default_opts.m_nEncodingType;
  // the "options" data is used by the Options dialog, it allows the opts
  // struct to be shared between the two dialogs
  g_object_set_data(G_OBJECT(dialog), "options", &opts);

  options_button = gtk_button_new_with_mnemonic("O_ptions...");
  gtk_button_set_relief(GTK_BUTTON(options_button), GTK_RELIEF_HALF);
  g_signal_connect(G_OBJECT(options_button), "clicked",
		   G_CALLBACK(options_button_clicked), dialog);
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->action_area),
			      options_button);

  gtk_dialog_add_buttons(GTK_DIALOG(dialog), GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
			 GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, 0);

  //gtk_window_set_resizable(GTK_WINDOW(dialog), TRUE);
  gtk_dialog_set_has_separator(GTK_DIALOG(dialog), FALSE);
  gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog), GTK_RESPONSE_ACCEPT,
				    FALSE);
  gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_ACCEPT);

  hbox = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, TRUE, TRUE, 10);
  //gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), hbox);

  host_label = gtk_label_new("VNC server:");
  host_entry_combo = GTK_COMBO_BOX(gtk_combo_box_entry_new_text());
  host_entry = GTK_ENTRY(GTK_BIN(host_entry_combo)->child);
  g_object_set(G_OBJECT(host_entry), "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_FULL, 0);
  g_signal_connect(G_OBJECT(host_entry), "changed",
		   G_CALLBACK(host_entry_changed),
		   dialog);
  gtk_entry_set_width_chars(host_entry, 25);
  gtk_entry_set_activates_default(host_entry, TRUE);
  //gtk_box_pack_start(GTK_BOX(hbox), host_label, TRUE, TRUE, 10);
  //gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(host_entry_combo), TRUE,
  //		     TRUE, 10);
  gtk_container_add(GTK_CONTAINER(hbox), host_label);
  gtk_container_add(GTK_CONTAINER(hbox), GTK_WIDGET(host_entry_combo));

  if (mainview->m_hostList)
  {
    guint nHosts = 0;
    GSList *slist = mainview->m_hostList;
    while (slist && (nHosts <= 8))
    {
      gtk_combo_box_append_text(host_entry_combo, (gchar *) slist->data);
      slist = slist->next;
      nHosts++;
    }

    // automatically choose the last host if there is one
    gtk_combo_box_set_active(host_entry_combo, 0);
    gtk_editable_set_position(GTK_EDITABLE(host_entry), -1);
  }

  gtk_widget_show_all(dialog);

  if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT)
  {
    const gchar *display = gtk_entry_get_text(host_entry);
    gint nActive = -1;
    gtk_widget_hide_all(dialog);

    if (mainview->vnc)
      vnc_connection_closed(mainview);

    make_connection(display, opts.m_bShared, opts.m_nDepth,
		    opts.m_nRotationType, opts.m_nEncodingType,
		    mainview);

    // preserve settings, even if the connection failed
    mainview->default_opts.m_bShared = opts.m_bShared;
    mainview->default_opts.m_nDepth = opts.m_nDepth;
    mainview->default_opts.m_nRotationType = opts.m_nRotationType;
    mainview->default_opts.m_nEncodingType = opts.m_nEncodingType;

    nActive = gtk_combo_box_get_active(host_entry_combo);
    if (nActive != -1)
    {
      if (nActive != 0)
      {
	// then the combo box element was used and it isn't the first
	// one--move it to the beginning of m_hostList
	GSList *nthNode = g_slist_nth(mainview->m_hostList, nActive);
	mainview->m_hostList =
	  g_slist_prepend(mainview->m_hostList, nthNode->data);
	g_slist_delete_link(mainview->m_hostList, nthNode);
      }
    }
    else
    {
      // then this is a new entry, add it to the beginning of the list
      mainview->m_hostList =
	g_slist_prepend(mainview->m_hostList, g_strdup(display));
    }
  }

  gtk_widget_destroy(dialog);
}

static void callback_disconnect(GtkBin *action, gpointer data)
{
  MainView *mainview = (MainView *) data;
  vnc_connection_closed(mainview);
}

static void callback_fullscreen(GtkBin *action, gpointer data)
{
  MainView *mainview = (MainView *) data;

  // because calling gtk_toggle_tool_button_set_active and
  // gtk_check_menu_item_set_active will result in another call to
  // callback_fullscreen, changes are delayed till both the
  // states of the fullscreen items in the menu and toolbar have been
  // updated

  if (mainview->fullscreen_tb)
  {
    GtkToggleToolButton *pFullScreenTTB =
      GTK_TOGGLE_TOOL_BUTTON(mainview->fullscreen_tb);
    if (gtk_toggle_tool_button_get_active(pFullScreenTTB))
    {
      if (mainview->fullscreen)
      {
	gtk_toggle_tool_button_set_active(pFullScreenTTB, FALSE);
	return;
      }
    }
    else
    {
      if (!mainview->fullscreen)
      {
	gtk_toggle_tool_button_set_active(pFullScreenTTB, TRUE);
	return;
      }
    }
  }

  if (mainview->fullscreen_item)
  {
    GtkCheckMenuItem *pFullScreenCMI =
      GTK_CHECK_MENU_ITEM(mainview->fullscreen_item);
    if (gtk_check_menu_item_get_active(pFullScreenCMI))
    {
      if (mainview->fullscreen)
      {
	gtk_check_menu_item_set_active(pFullScreenCMI, FALSE);
	return;
      }
    }
    else
    {
      if (!mainview->fullscreen)
      {
	gtk_check_menu_item_set_active(pFullScreenCMI, TRUE);
	return;
      }
    }
  }

  mainview->fullscreen = !mainview->fullscreen;

  gboolean bShowToolbar = TRUE;
  if (mainview->fullscreen)
    gtk_window_fullscreen(GTK_WINDOW(mainview->data->main_view));
  else
    gtk_window_unfullscreen(GTK_WINDOW(mainview->data->main_view));

  // handle toolbar
  if ((mainview->fullscreen) && (!mainview->bShowFullscreenToolbar))
    bShowToolbar = FALSE;
  else if ((!mainview->fullscreen) && (!mainview->bShowNormalToolbar))
    bShowToolbar = FALSE;

  if (bShowToolbar)
    gtk_widget_show_all(mainview->toolbar);
  else
    gtk_widget_hide_all(mainview->toolbar);
}

void callback_im(GtkBin *action, gpointer data)
{
  MainView *mainview = (MainView *) data;

  if (mainview->vnc)
  {
    // send enter key press and release to VNC widget
    //GdkEvent *pEvent = gdk_event_new(GDK_KEY_PRESS);
    //if (pEvent)
    //{
      GdkEventKey eventKey;
      GdkKeymapKey *keys;
      gint n_keys;
      gdk_keymap_get_entries_for_keyval(gdk_keymap_get_default(), GDK_Return,
					&keys, &n_keys);
      //GdkEventKey *pKeyEvent = (GdkEventKey *) pEvent;
      //pKeyEvent->type = GDK_KEY_PRESS;
      //pKeyEvent->window = (GTK_WIDGET(mainview->vnc))->window;
      //pKeyEvent->keyval = GDK_Return;
      /*
	{type = GDK_KEY_PRESS, window = 0x8f0f428, send_event = 0 '\0',
	time = 2264591236, state = 0, keyval = 65293, length = 1,
	string = 0x8f01e50 "\r", hardware_keycode = 36, group = 0 '\0'}
      */
      eventKey.type = GDK_KEY_PRESS;
      eventKey.hardware_keycode = keys[0].keycode;
      eventKey.keyval = GDK_Return;
      eventKey.state = 0;
      eventKey.window = (GTK_WIDGET(mainview->vnc))->window;
      eventKey.length = 1;
      eventKey.string = "\r";
      eventKey.send_event = 0;
      eventKey.group = 0;

      gtk_widget_grab_focus(GTK_WIDGET(mainview->vnc));

      vnc_viewer_key_press_event(GTK_WIDGET(mainview->vnc), &eventKey);
      //gtk_widget_event(GTK_WIDGET(mainview->vnc), pEvent);

      //pKeyEvent->type = GDK_KEY_RELEASE;
      //gtk_widget_event(GTK_WIDGET(mainview->vnc), pEvent);
      eventKey.type = GDK_KEY_RELEASE;
      vnc_viewer_key_release_event(GTK_WIDGET(mainview->vnc), &eventKey);
      //gdk_event_free(pEvent);
      //}
  }
}

void callback_keys_toggled(GtkToggleToolButton *toggle_tool_button,
			   gpointer data)
{
  MainView *mainview = (MainView *) data;

  if (mainview->vnc)
  {
    if (gtk_toggle_tool_button_get_active(toggle_tool_button) == TRUE)
      gtk_menu_popup(GTK_MENU(mainview->keys_menu), 0, 0, 0, 0, 0,
		     gtk_get_current_event_time());
  }
}

void callback_mButton(GtkCheckMenuItem *checkmenuitem, gpointer data)
{
  MainView *mainview = (MainView *) data;
  if (gtk_check_menu_item_get_active(mainview->mButton.item_Rmouse))
    mainview->mButton.state=PM_Rmouse;  
  if (gtk_check_menu_item_get_active(mainview->mButton.item_Mmouse))
    mainview->mButton.state=PM_Mmouse;  
  if (gtk_check_menu_item_get_active(mainview->mButton.item_Dclick))
    mainview->mButton.state=PM_Dclick;  
  g_message("state : %d",mainview->mButton.state);
}

void callback_normal_toggled(GtkCheckMenuItem *checkmenuitem, gpointer data)
{
  MainView *mainview = (MainView *) data;

  mainview->bShowNormalToolbar = !mainview->bShowNormalToolbar;
  
  if (!mainview->fullscreen)
  {
    if (mainview->bShowNormalToolbar)
      gtk_widget_show_all(mainview->toolbar);
    else
      gtk_widget_hide_all(mainview->toolbar);
  }
}

static void callback_fullscreen_toggled(GtkCheckMenuItem *checkmenuitem, gpointer data)
{
  MainView *mainview = (MainView *) data;

  mainview->bShowFullscreenToolbar = !mainview->bShowFullscreenToolbar;
//fremantle force Fullscreen Toolbar
//removed again
//  mainview->bShowFullscreenToolbar = TRUE;

  if (mainview->fullscreen)
  {
    if (mainview->bShowFullscreenToolbar)
      gtk_widget_show_all(mainview->toolbar);
    else
      gtk_widget_hide_all(mainview->toolbar);
  }
}

static void callback_show_scrollbars_toggled(GtkCheckMenuItem *checkmenuitem,
					     gpointer data)
{
  MainView *mainview = (MainView *) data;
  mainview->bShowScrollbars = !mainview->bShowScrollbars;

  if (mainview->vnc)
  {
    if (mainview->bShowScrollbars)
    {
      // note:  the following is a kludge--when GTK_POLICY_AUTOMATIC was used
      // here instead of GTK_POLICY_ALWAYS, the scroll bars never came back
      // if they should have come back
      // in this case, the policy is later set properly to GTK_POLICY_AUTOMATIC
      // in scrolledwin_size_allocate()
      gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(mainview->scrolledwin),
				     GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);
    }
    else
      gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(mainview->scrolledwin),
				     GTK_POLICY_NEVER, GTK_POLICY_NEVER);
  }
}

void callback_about(GtkBin *action, gpointer data)
{
  MainView *mainview = (MainView *) data;
  const gchar *authors[3] = { "Aaron Levinson <alevinsn@aracnet.com>",
			      "Detlef Schmicker <d.schmicker@physik.de>", 0 };
  //const gchar *documenters[] = { 0 };
  gtk_show_about_dialog(GTK_WINDOW(mainview->data->main_view),
			"name",_("VNC Viewer"),
			"version", VERSION,
			"copyright", "Copyright(C) 2006-2007 Aaron Levinson, Detlef Schmicker",
			//"comments", _("A VNC viewer for maemo"),
			"authors", authors,
			//"documenters", documenters,
			//"translator-credits", 0,
			"website", "http://vncviewer.garage.maemo.org/",
			"website-label", _("Home page"),
			"logo-icon-name", "vncviewer",
			0);
}

void callback_ctrl_alt_del(GtkBin *action, gpointer data)
{
  MainView *mainview = (MainView *) data;
  if (mainview->vnc)
  {
    // send Ctrl-Alt-Del key press and release to VNC widget

    // hold Ctrl, Alt, and Del down
    vnc_viewer_key_press(mainview->vnc, GDK_Control_L);
    vnc_viewer_key_press(mainview->vnc, GDK_Alt_L);
    vnc_viewer_key_press(mainview->vnc, GDK_Delete);

    // release keys
    vnc_viewer_key_release(mainview->vnc, GDK_Control_L);
    vnc_viewer_key_release(mainview->vnc, GDK_Alt_L);
    vnc_viewer_key_release(mainview->vnc, GDK_Delete);
  }
}

void callback_special_key(GtkMenuItem *action, gpointer data)
{
  MainView *mainview = (MainView *) data;
  if (mainview->vnc)
  {
    vnc_viewer_key_press(mainview->vnc,
			 GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(action), "keyval")));
  }
}

void callback_ctrl_toggled(GtkCheckMenuItem *checkmenuitem, gpointer data)
{
  MainView *mainview = (MainView *) data;

  if (!mainview->vnc)
    return;

  mainview->m_bCtrlOn = !mainview->m_bCtrlOn;

  if (mainview->m_bCtrlOn)
    vnc_viewer_key_press(mainview->vnc, GDK_Control_L);
  else
    vnc_viewer_key_release(mainview->vnc, GDK_Control_L);
}

void callback_alt_toggled(GtkCheckMenuItem *checkmenuitem, gpointer data)
{
  MainView *mainview = (MainView *) data;

  if (!mainview->vnc)
    return;

  mainview->m_bAltOn = !mainview->m_bAltOn;

  if (mainview->m_bAltOn)
    vnc_viewer_key_press(mainview->vnc, GDK_Alt_L);
  else
    vnc_viewer_key_release(mainview->vnc, GDK_Alt_L);
}

void callback_shift_toggled(GtkCheckMenuItem *checkmenuitem, gpointer data)
{
  MainView *mainview = (MainView *) data;

  if (!mainview->vnc)
    return;

  mainview->m_bShiftOn = !mainview->m_bShiftOn;

  if (mainview->m_bShiftOn)
    vnc_viewer_key_press(mainview->vnc, GDK_Shift_L);
  else
    vnc_viewer_key_release(mainview->vnc, GDK_Shift_L);
}

void callback_keys_menu_deactivate(GtkMenuShell *menushell, gpointer userdata)
{
  MainView *mainview = (MainView *) userdata;
  gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON(mainview->keys_tb),
				    FALSE);
}

gboolean was_scrolling = FALSE;
// if scrolling is changing from local to remote window, 
// let the user have a better chance to stop scrolling
gint wait_scrolling = 3;
guint wait_release=0;
static gboolean p_m_button_release()
{
  if (wait_release++<5)
    return TRUE;
  p_button_pressed=FALSE;
  m_button_pressed=FALSE;
  wait_release=0;
  return FALSE;
}

gboolean callback_keypress(GtkWidget *widget, GdkEventKey *event,
			   gpointer userdata)
{
  MainView *mainview = (MainView *) userdata;

  //g_message("callback_keypress entered");
  wait_release=0;

  if (arg_lockdown)
  {
    if (event->keyval == GDK_Escape)
      // prevent default behavior for GDK_Escape in hildon_app from
      // happening.  The default behavior in hildon_app for a long
      // back key key press is to exit the application.
      return TRUE;

    return FALSE;
  }

  // note GDK_F4, i.e. the menu, is handled for us by maemo
  switch (event->keyval)
  {
  case GDK_F7:
    if (!p_button_pressed)
    	pm_button_for_toggle_fullscreen=TRUE;
    p_button_pressed=TRUE;
    g_message("p button press");
    return TRUE;
  case GDK_F8:
    if (!m_button_pressed)
    	pm_button_for_toggle_fullscreen=TRUE;
    m_button_pressed=TRUE;
    g_message("m button press");
    return TRUE;
  case GDK_F6:
    callback_fullscreen(0, mainview);
    return TRUE;

  case GDK_Up:
  case GDK_Down:
  case GDK_Left:
  case GDK_Right:
    was_scrolling = FALSE;
    if (mainview->vnc &&
	(gtk_widget_is_focus(GTK_WIDGET(mainview->vnc)) == TRUE))
    {
      // do hardware movement key scrolling
      // only do this if the last movement key pressed (and not released)
      // is the same as the one that was just pressed
      if (mainview->directionKeyPress == event->keyval)
      {
	mainview->bMultiplePresses = TRUE;
	// scroll in the proper direction
	if ((event->keyval == GDK_Up) || event->keyval == GDK_Down)
	{
	  GtkAdjustment *vadj = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(mainview->scrolledwin));

	  if (event->keyval == GDK_Up)
	  {
	    //Detlef Testing
	    if (vadj->value==0) 
	    {
	      if (wait_scrolling<=0) 
	      {
		was_scrolling=TRUE;
		vnc_viewer_key_press_event(GTK_WIDGET(mainview->vnc), event);
		vnc_viewer_key_release_event(GTK_WIDGET(mainview->vnc), event);
	      }
	      else
		wait_scrolling--;
	    }
	    vadj->value = vadj->value - vadj->step_increment;
	    if (vadj->value < vadj->lower)
	      vadj->value = vadj->lower;
	  }
	  else
	  {
	    if (vadj->value+vadj->page_size>=vadj->upper) 
	    {
	      if (wait_scrolling<=0) 
	      {
		was_scrolling=TRUE;
		vnc_viewer_key_press_event(GTK_WIDGET(mainview->vnc), event);
		vnc_viewer_key_release_event(GTK_WIDGET(mainview->vnc), event);
	      }
	      else
		wait_scrolling--;
	    }
	    vadj->value = vadj->value + vadj->step_increment;
	    if (vadj->value > vadj->upper - vadj->page_size)
	      vadj->value = vadj->upper - vadj->page_size;
	  }

	  gtk_adjustment_value_changed(vadj);
	}
	else
	{
	  GtkAdjustment *hadj = gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(mainview->scrolledwin));
	  if (event->keyval == GDK_Left)
	  {
	    if (hadj->value==0) 
	    {
	      if (wait_scrolling<=0) 
	      {
		was_scrolling=TRUE;
		vnc_viewer_key_press_event(GTK_WIDGET(mainview->vnc), event);
		vnc_viewer_key_release_event(GTK_WIDGET(mainview->vnc), event);
	      }
	      else
		wait_scrolling--;
	    }
	    hadj->value = hadj->value - hadj->step_increment;
	    if (hadj->value < hadj->lower)
	      hadj->value = hadj->lower;
	  }
	  else
	  {
	    if (hadj->value+hadj->page_size>=hadj->upper) 
	    {
	      if (wait_scrolling<=0) 
	      {
		was_scrolling=TRUE;
		vnc_viewer_key_press_event(GTK_WIDGET(mainview->vnc), event);
		vnc_viewer_key_release_event(GTK_WIDGET(mainview->vnc), event);
	      }
	      else
		wait_scrolling--;
	    }
	    hadj->value = hadj->value + hadj->step_increment;
	    if (hadj->value > hadj->upper - hadj->page_size)
	      hadj->value = hadj->upper - hadj->page_size;
	  }

	  gtk_adjustment_value_changed(hadj);
	}
      }
      else
	// indicate that a direction key has been pressed
	// if a release event for this direction key occurs
	// next, then the key press and release will be propogated
	// to the VncViewer widget if it is available
	mainview->directionKeyPress = event->keyval;
    }
    return TRUE;

  default:
    break;
  }

  /*
  if (event->keyval == HILDON_HARDKEY_SELECT){
    g_message("HILDON_HARDKEY press");
    if (timeout_id == 0)
      timeout_id = g_timeout_add(SELECT_KEY_LONG_PRESS,
				 select_key_long_press,
				 GTK_WIDGET(userdata)->window);

    return FALSE;
  }
  */

  return FALSE;
}


gboolean callback_keyrelease(GtkWidget *widget, GdkEventKey *event,
			     gpointer userdata)
{
  MainView *mainview = (MainView *) userdata;
  wait_scrolling=3;

  if (arg_lockdown)
    return FALSE;

  switch (event->keyval)
  {
  case GDK_F7:
  case GDK_F8:
    g_timeout_add(100, (GtkFunction) p_m_button_release, NULL);
    if (pm_button_for_toggle_fullscreen)
    	callback_fullscreen(0, mainview);
    pm_button_for_toggle_fullscreen=FALSE;
    g_message("pm button release");
    return TRUE;
  }
  if (mainview->directionKeyPress &&
      (mainview->directionKeyPress == event->keyval) &&
      !mainview->bMultiplePresses)
  {
    // determine if the window with the focus is the VNC widget
    if (mainview->vnc &&
	(gtk_widget_is_focus(GTK_WIDGET(mainview->vnc)) == TRUE))
    {
      // then deliver key press and key down events to the VNC widget
      //event->type = GDK_KEY_PRESS;
      //event->window = (GTK_WIDGET(mainview->vnc))->window;
      //gtk_widget_event(GTK_WIDGET(mainview->vnc), (GdkEvent *) event);
      //gtk_main_do_event((GdkEvent *) event);
      if (!was_scrolling) 
      {
	vnc_viewer_key_press_event(GTK_WIDGET(mainview->vnc), event);
	//event->type = GDK_KEY_RELEASE;
	//gtk_widget_event(GTK_WIDGET(mainview->vnc), (GdkEvent *) event);
	//gtk_main_do_event((GdkEvent *) event);
	vnc_viewer_key_release_event(GTK_WIDGET(mainview->vnc), event);
      }

      mainview->directionKeyPress = 0;
      return TRUE;
    }
  }

  mainview->directionKeyPress = 0;
  mainview->bMultiplePresses = 0;

  /*
  if (event->keyval == HILDON_HARDKEY_SELECT){
    g_message("HILDON_HARDKEY release");
    if (timeout_id)
      {
	g_source_remove(timeout_id);
	timeout_id = 0;
      }

    return FALSE;
  }
  */

  return FALSE;
}

void scrolledwin_size_allocate(GtkWidget *widget,
			       GtkAllocation *allocation,
			       gpointer userdata)
{
  MainView *mainview = (MainView *) userdata;
  GtkPolicyType hpolicy;
  if (!mainview->vnc)
    return;

  gtk_scrolled_window_get_policy(GTK_SCROLLED_WINDOW(mainview->scrolledwin),
				 &hpolicy, 0);

  if ((hpolicy == GTK_POLICY_AUTOMATIC) &&
      (!mainview->vnc->m_bRotateDisplay ?
      ((allocation->width >= mainview->vnc->width) &&
       (allocation->height >= mainview->vnc->height)) :
      ((allocation->width >= mainview->vnc->height) &&
       (allocation->height >= mainview->vnc->width))))
  {
    // then the server desktop size will fit in the window--cause the scroll
    // bars to be turned off

    GtkAdjustment *hadj = gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(mainview->scrolledwin));
    GtkAdjustment *vadj = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(mainview->scrolledwin));

    if (hadj->page_size != allocation->width)
    {
      hadj->page_increment = allocation->width;
      hadj->page_size = hadj->page_increment;
      gtk_adjustment_changed(hadj);
    }
  
    if (vadj->page_size != allocation->height)
    {
      vadj->page_increment = allocation->height;
      vadj->page_size = vadj->page_increment;
      gtk_adjustment_changed(vadj);
    }
  }
  else
  {
    // note:  the following is a kludge--refer to comments in
    // callback_show_scrollbars_toggled()
    if (mainview->bShowScrollbars)
    {
      // get the policy types
      if (hpolicy == GTK_POLICY_ALWAYS)
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(mainview->scrolledwin),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
    }
  }
}

void scrolledwin2_size_request(GtkWidget *widget,
			       GtkRequisition *requisition,
			       gpointer userdata)
{
  // the height of the scrolled window isn't set properly by default,
  // need to base it on size of the main vbox, which isn't calculated
  // properly until we get to this point
  GtkWidget *main_vbox = (GtkWidget *) userdata;
  GtkRequisition requisition2;
  gtk_widget_size_request(main_vbox, &requisition2);
  // indicate to parent that we want to change the height
  requisition->height = requisition2.height;
}

gint CompareHostsInList(const gchar *a, const gchar *b)
{
  return strcmp(a, b);
}

static gboolean mainview_auto_connect(gpointer userdata)
{
  MainView *mainview = (MainView *) userdata;
  // the following test seems to work--we want to wait till
  // the entire screen is finished before auto-displaying the dialog
  // basing things on scrolledwin because it would be displayed after its
  // parents
  if (GTK_WIDGET_DRAWABLE(mainview->scrolledwin))
  {
    if (arg_hostname)
    {
      while (1)
      {
	if (make_connection((const gchar *) arg_hostname,
			    mainview->default_opts.m_bShared,
			    mainview->default_opts.m_nDepth,
			    mainview->default_opts.m_nRotationType,
			    mainview->default_opts.m_nEncodingType,
			    mainview))
	{
	  // move it to the beginning of m_hostList if the node already exists
	  GSList *hostNode = g_slist_find_custom(mainview->m_hostList,
						 arg_hostname,
						 (GCompareFunc) CompareHostsInList);
	  if (hostNode)
	  {
	    // determine position of node in list
	    if (g_slist_position(mainview->m_hostList, hostNode))
	    {
	      // then prepend, not the first node in the list
	      mainview->m_hostList =
		g_slist_prepend(mainview->m_hostList, hostNode->data);
	      g_slist_delete_link(mainview->m_hostList, hostNode);
	    }
	  }
	  else
	  {
	    // then this is a new entry, add it to the beginning of the list
	    mainview->m_hostList =
	      g_slist_prepend(mainview->m_hostList, g_strdup(arg_hostname));
	  }
	}
	else if (arg_lockdown)
	  continue;

	break;
      }
    }
    else
      callback_connect(0, userdata);

    if (arg_lockdown)
    {
      // usurp the Hildon App key snooper
      // doing this here instead of earlier because it is necessary
      // to install the key snooper after the Hildon App key snooper has
      // been installed.  This is because the gtk_key_snooper_install()
      // implementation prepends new key snoopers to the list it maintains,
      // and when key snoopers are called, it calls them in the list's order.
      // if a key snooper returns TRUE for a particular key event, then any
      // additional key snoopers in the list are not invoked for the key
      // event.  The Hildon App key snooper is installed when the Hildon App
      // is realized, which should have occurred by the time we get here.
      mainview->m_nKeySnooperId = 
	gtk_key_snooper_install((GtkKeySnoopFunc) app_key_snooper,
				mainview->data->main_view);
    }

    return FALSE;
  }

  return TRUE;
}

static void create_menu_helper(MainView *mainview)
{
  // Create needed handles
  GtkWidget *file_menu, *view_menu, *toolbar_menu, *help_menu;
  GtkWidget *toolbar_item;
  GtkWidget *connect_item;
  //GtkWidget *take_screenshot_item;
  GtkWidget *fullscreen_item;
  GtkWidget *normal_toolbar_item;
  GtkWidget *full_toolbar_item;
  GtkWidget *show_scrollbars_item;
  GtkWidget *about_item;
//  GtkWidget *mButton_menu;
  
  // Create new menus for submenus in our drop down menu
  file_menu = gtk_menu_new();
  view_menu = gtk_menu_new();
  toolbar_menu = gtk_menu_new();
  help_menu = gtk_menu_new();
//  mButton_menu = gtk_menu_new();
//  mainview->mButton.item_Rmouse=gtk_radio_menu_item_new_with_label(NULL,_("Rmouse"));
//  mainview->mButton.item_Mmouse=
//    gtk_radio_menu_item_new_with_label_from_widget(
//	GTK_RADIO_MENU_ITEM(mainview->mButton.item_Rmouse),_("Mmouse"));
//  mainview->mButton.item_Dclick=
//    gtk_radio_menu_item_new_with_label_from_widget(
//	GTK_RADIO_MENU_ITEM(mainview->mButton.item_Rmouse),_("Dclick"));

//  mainview->mButton_item = gtk_menu_item_new_with_label(_("mButton"));

  // Create the menu items
  mainview->file_item = gtk_menu_item_new_with_label(_("File"));
  connect_item = gtk_menu_item_new_with_label(_("Connect to a VNC server"));
  mainview->disconnect_item = gtk_menu_item_new_with_label(_("Disconnect current VNC session"));
  gtk_widget_set_sensitive(mainview->disconnect_item, FALSE);

  mainview->view_item = gtk_menu_item_new_with_label(_("View"));
  fullscreen_item = gtk_menu_item_new_with_label(_("Full screen"));
  toolbar_item = gtk_menu_item_new_with_label(_("Show toolbar"));

  normal_toolbar_item =
    gtk_check_menu_item_new_with_label(_("Normal screen"));
  full_toolbar_item =
    gtk_check_menu_item_new_with_label(_("Full screen"));
  show_scrollbars_item =
    gtk_check_menu_item_new_with_label(_("Show scrollbars"));
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(normal_toolbar_item),
				 mainview->bShowNormalToolbar);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(full_toolbar_item),
				 mainview->bShowFullscreenToolbar);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(show_scrollbars_item),
				 mainview->bShowScrollbars);

  mainview->keyboard_item = gtk_menu_item_new_with_label(_("Keyboard"));

  mainview->help_item = gtk_menu_item_new_with_label(_("Help"));
  about_item = gtk_menu_item_new_with_label(_("About"));

  mainview->quit_item = gtk_menu_item_new_with_label(_("Quit"));

  // Add menu items to right menus
  gtk_menu_append(file_menu, connect_item);
  gtk_menu_append(file_menu, mainview->disconnect_item);
  gtk_menu_append(view_menu, fullscreen_item);
  gtk_menu_append(view_menu, toolbar_item);
  gtk_menu_append(toolbar_menu, normal_toolbar_item);
  gtk_menu_append(toolbar_menu, full_toolbar_item);
  gtk_menu_append(view_menu, show_scrollbars_item);
  gtk_menu_append(help_menu, about_item);

//  gtk_menu_append(mButton_menu, mainview->mButton.item_Rmouse);
//  gtk_menu_append(mButton_menu, mainview->mButton.item_Mmouse);
//  gtk_menu_append(mButton_menu, mainview->mButton.item_Dclick);


  // Add submenus to the right items
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(mainview->file_item), file_menu);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(mainview->view_item), view_menu);
//  gtk_menu_item_set_submenu(GTK_MENU_ITEM(mainview->mButton_item), mButton_menu);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(toolbar_item), toolbar_menu);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(mainview->keyboard_item),
			    GTK_WIDGET(mainview->keys_menu));
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(mainview->help_item), help_menu);

  // Attach the callback functions to the activate signal
  g_signal_connect(G_OBJECT(connect_item), "activate", 
		   G_CALLBACK(callback_connect), mainview);
  g_signal_connect(G_OBJECT(mainview->disconnect_item), "activate", 
		   G_CALLBACK(callback_disconnect), mainview);
  g_signal_connect(G_OBJECT(fullscreen_item), "activate", 
		   G_CALLBACK(callback_fullscreen), mainview);
  g_signal_connect(G_OBJECT(normal_toolbar_item), "toggled", 
		   G_CALLBACK(callback_normal_toggled), mainview);
  g_signal_connect(G_OBJECT(full_toolbar_item), "toggled", 
		   G_CALLBACK(callback_fullscreen_toggled), mainview);
  g_signal_connect(G_OBJECT(show_scrollbars_item), "toggled", 
		   G_CALLBACK(callback_show_scrollbars_toggled), mainview);
  g_signal_connect(G_OBJECT(about_item), "activate",
		   G_CALLBACK(callback_about), mainview);
  g_signal_connect(G_OBJECT(mainview->quit_item), "activate",
		   gtk_main_quit, 0);

  
  g_signal_connect(G_OBJECT(mainview->mButton.item_Rmouse), "toggled", 
		   G_CALLBACK(callback_mButton), mainview);
  g_signal_connect(G_OBJECT(mainview->mButton.item_Mmouse), "toggled", 
		   G_CALLBACK(callback_mButton), mainview);
  g_signal_connect(G_OBJECT(mainview->mButton.item_Dclick), "toggled", 
		   G_CALLBACK(callback_mButton), mainview);
  

}

// Create the menu items needed for the drop down menu
void create_menu(void *mainvw)
{
  MainView *mainview = (MainView *) mainvw;
  GtkMenu *main_menu;
  GtkWidget *separator = NULL;

  if (!mainview->file_item)
    create_menu_helper(mainview);

  if (arg_lockdown)
    return;

  main_menu = GTK_MENU(gtk_menu_new());
  hildon_program_set_common_menu(mainview->data->app, main_menu);

  separator = gtk_separator_menu_item_new();

  // Add menu items to right menus
  gtk_menu_append(main_menu, mainview->file_item);
  gtk_menu_append(main_menu, mainview->view_item);
  gtk_menu_append(main_menu, mainview->mButton_item);
  gtk_menu_append(main_menu, mainview->keyboard_item);
  gtk_menu_append(main_menu, mainview->help_item);
  gtk_menu_append(main_menu, separator);
  gtk_menu_append(main_menu, mainview->quit_item);

  // We need to show menu items
  gtk_widget_show_all(GTK_WIDGET(main_menu));
}


// Create toolbar to mainview
static void create_toolbar(MainView *mainview)
{
    GtkToolItem *separator_tb1;
    GtkToolItem *fullscreen_tb;
    GtkToolItem *separator_tb2;
    GtkToolItem *quit_tb;

    GtkWidget *ctrl_alt_del_item;
    GtkWidget *keyb_icon;

    // function keys stuff
    GtkMenuShell *function_keys_menu;
    GtkWidget *function_keys_item;
    GtkMenuShell *function_keys_more_menu;
    GtkWidget *function_keys_more_item;

    GtkWidget *f1_item;
    GtkWidget *f2_item;
    GtkWidget *f3_item;
    GtkWidget *f4_item;
    GtkWidget *f5_item;
    GtkWidget *f6_item;
    GtkWidget *f7_item;
    GtkWidget *f8_item;
    GtkWidget *f9_item;
    GtkWidget *f10_item;
    GtkWidget *f11_item;
    GtkWidget *f12_item;

    // other keys stuff
    GtkMenuShell *other_keys_menu;
    GtkWidget *other_keys_item;

    GtkWidget *home_item;
    GtkWidget *end_item;
    GtkWidget *pg_up_item;
    GtkWidget *pg_down_item;
    GtkWidget *insert_item;
    GtkWidget *del_item;
    GtkWidget *esc_item;

    // Create new GTK toolbar
    mainview->toolbar = gtk_toolbar_new();

    // Set toolbar properties
    gtk_toolbar_set_orientation(GTK_TOOLBAR(mainview->toolbar), GTK_ORIENTATION_HORIZONTAL);
    gtk_toolbar_set_style(GTK_TOOLBAR(mainview->toolbar), GTK_TOOLBAR_BOTH_HORIZ);

    // Make menus and buttons to toolbar:
    // Create toolitems using defined items from stock
    mainview->connect_tb = gtk_tool_button_new_from_stock(GTK_STOCK_CONNECT);
    mainview->disconnect_tb = gtk_tool_button_new_from_stock(GTK_STOCK_DISCONNECT);
    gtk_widget_set_sensitive(GTK_WIDGET(mainview->disconnect_tb), FALSE);
    separator_tb1 = gtk_separator_tool_item_new();
    // the following stock item only in GTK 2.8
    //fullscreen_tb = gtk_tool_button_new_from_stock(GTK_STOCK_FULLSCREEN);
    fullscreen_tb = gtk_tool_button_new_from_stock(GTK_STOCK_ZOOM_FIT);
    mainview->im_tb = gtk_tool_button_new_from_stock(GTK_STOCK_EDIT);
    gtk_widget_set_sensitive(GTK_WIDGET(mainview->im_tb), FALSE);
    mainview->keys_tb = gtk_toggle_tool_button_new();
    keyb_icon = gtk_image_new_from_icon_name("qgn_inpu_common_vkb",
					     GTK_ICON_SIZE_BUTTON);
    gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(mainview->keys_tb),
				    keyb_icon);
    gtk_widget_set_sensitive(GTK_WIDGET(mainview->keys_tb), FALSE);
    
    separator_tb2 = gtk_separator_tool_item_new();
    quit_tb = gtk_tool_button_new_from_stock(GTK_STOCK_QUIT);

    // Insert items to toolbar
    gtk_toolbar_insert(GTK_TOOLBAR(mainview->toolbar), mainview->connect_tb, -1);
    gtk_toolbar_insert(GTK_TOOLBAR(mainview->toolbar), mainview->disconnect_tb, -1);
    gtk_toolbar_insert(GTK_TOOLBAR(mainview->toolbar), separator_tb1, -1);
    gtk_toolbar_insert(GTK_TOOLBAR(mainview->toolbar), fullscreen_tb, -1);
    gtk_toolbar_insert(GTK_TOOLBAR(mainview->toolbar), mainview->im_tb, -1);
    gtk_toolbar_insert(GTK_TOOLBAR(mainview->toolbar), mainview->keys_tb, -1);
    gtk_toolbar_insert(GTK_TOOLBAR(mainview->toolbar), separator_tb2, -1);
    gtk_toolbar_insert(GTK_TOOLBAR(mainview->toolbar), quit_tb, -1);

    // Connect signals to buttons
    g_signal_connect(G_OBJECT(mainview->connect_tb), "clicked",
                     G_CALLBACK(callback_connect), mainview);
    g_signal_connect(G_OBJECT(mainview->disconnect_tb), "clicked",
                     G_CALLBACK(callback_disconnect), mainview);
    g_signal_connect(G_OBJECT(fullscreen_tb), "clicked",
                     G_CALLBACK(callback_fullscreen), mainview);
    g_signal_connect(G_OBJECT(mainview->im_tb), "clicked",
                     G_CALLBACK(callback_im), mainview);
    g_signal_connect(G_OBJECT(mainview->keys_tb), "toggled",
                     G_CALLBACK(callback_keys_toggled), mainview);
    g_signal_connect(G_OBJECT(quit_tb), "clicked", gtk_main_quit, 0);
	
    // handle input method menu
    mainview->keys_menu = GTK_MENU_SHELL(gtk_menu_new());
    g_object_ref(G_OBJECT(mainview->keys_menu));
    gtk_object_sink(GTK_OBJECT(mainview->keys_menu));
    g_signal_connect(G_OBJECT(mainview->keys_menu), "deactivate",
    		     G_CALLBACK(callback_keys_menu_deactivate), mainview);
    mainview->ctrl_item =
      GTK_CHECK_MENU_ITEM(gtk_check_menu_item_new_with_label(_("Ctrl")));
    mainview->alt_item =
      GTK_CHECK_MENU_ITEM(gtk_check_menu_item_new_with_label(_("Alt")));
    mainview->shift_item =
      GTK_CHECK_MENU_ITEM(gtk_check_menu_item_new_with_label(_("Shift")));
    gtk_check_menu_item_set_active(mainview->ctrl_item, FALSE);
    gtk_check_menu_item_set_active(mainview->alt_item, FALSE);
    gtk_check_menu_item_set_active(mainview->shift_item, FALSE);
    ctrl_alt_del_item = gtk_menu_item_new_with_label(_("Send Ctrl-Alt-Del"));
    gtk_menu_shell_append(mainview->keys_menu, GTK_WIDGET(mainview->ctrl_item));
    gtk_menu_shell_append(mainview->keys_menu, GTK_WIDGET(mainview->alt_item));
    gtk_menu_shell_append(mainview->keys_menu, GTK_WIDGET(mainview->shift_item));
    function_keys_menu = GTK_MENU_SHELL(gtk_menu_new());
    function_keys_item = gtk_menu_item_new_with_label(_("Send function keys"));
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(function_keys_item),
			      GTK_WIDGET(function_keys_menu));
    gtk_menu_shell_append(mainview->keys_menu, function_keys_item);

    other_keys_menu = GTK_MENU_SHELL(gtk_menu_new());
    other_keys_item = gtk_menu_item_new_with_label(_("Send other keys"));
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(other_keys_item),
			      GTK_WIDGET(other_keys_menu));
    gtk_menu_shell_append(mainview->keys_menu, other_keys_item);

    gtk_menu_shell_append(mainview->keys_menu, GTK_WIDGET(ctrl_alt_del_item));

    g_signal_connect(G_OBJECT(mainview->ctrl_item), "toggled", 
		     G_CALLBACK(callback_ctrl_toggled), mainview);
    g_signal_connect(G_OBJECT(mainview->alt_item), "toggled", 
		     G_CALLBACK(callback_alt_toggled), mainview);
    g_signal_connect(G_OBJECT(mainview->shift_item), "toggled", 
		     G_CALLBACK(callback_shift_toggled), mainview);
    g_signal_connect(G_OBJECT(ctrl_alt_del_item), "activate", 
		     G_CALLBACK(callback_ctrl_alt_del), mainview);

    // setup function keys submenus
    function_keys_more_menu = GTK_MENU_SHELL(gtk_menu_new());
    function_keys_more_item = gtk_menu_item_new_with_label(_("More"));
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(function_keys_more_item),
			      GTK_WIDGET(function_keys_more_menu));

    // function keys
    f1_item = gtk_menu_item_new_with_label("F1");
    f2_item = gtk_menu_item_new_with_label("F2");
    f3_item = gtk_menu_item_new_with_label("F3");
    f4_item = gtk_menu_item_new_with_label("F4");
    f5_item = gtk_menu_item_new_with_label("F5");
    f6_item = gtk_menu_item_new_with_label("F6");
    f7_item = gtk_menu_item_new_with_label("F7");
    f8_item = gtk_menu_item_new_with_label("F8");
    f9_item = gtk_menu_item_new_with_label("F9");
    f10_item = gtk_menu_item_new_with_label("F10");
    f11_item = gtk_menu_item_new_with_label("F11");
    f12_item = gtk_menu_item_new_with_label("F12");

    // associate keyvalue data with each item
    g_object_set_data(G_OBJECT(f1_item), "keyval", GUINT_TO_POINTER(GDK_F1));
    g_object_set_data(G_OBJECT(f2_item), "keyval", GUINT_TO_POINTER(GDK_F2));
    g_object_set_data(G_OBJECT(f3_item), "keyval", GUINT_TO_POINTER(GDK_F3));
    g_object_set_data(G_OBJECT(f4_item), "keyval", GUINT_TO_POINTER(GDK_F4));
    g_object_set_data(G_OBJECT(f5_item), "keyval", GUINT_TO_POINTER(GDK_F5));
    g_object_set_data(G_OBJECT(f6_item), "keyval", GUINT_TO_POINTER(GDK_F6));
    g_object_set_data(G_OBJECT(f7_item), "keyval", GUINT_TO_POINTER(GDK_F7));
    g_object_set_data(G_OBJECT(f8_item), "keyval", GUINT_TO_POINTER(GDK_F8));
    g_object_set_data(G_OBJECT(f9_item), "keyval", GUINT_TO_POINTER(GDK_F9));
    g_object_set_data(G_OBJECT(f10_item), "keyval", GUINT_TO_POINTER(GDK_F10));
    g_object_set_data(G_OBJECT(f11_item), "keyval", GUINT_TO_POINTER(GDK_F11));
    g_object_set_data(G_OBJECT(f12_item), "keyval", GUINT_TO_POINTER(GDK_F12));

    // add F1-F6 plus More into the first submenu
    gtk_menu_shell_append(function_keys_menu, f1_item);
    gtk_menu_shell_append(function_keys_menu, f2_item);
    gtk_menu_shell_append(function_keys_menu, f3_item);
    gtk_menu_shell_append(function_keys_menu, f4_item);
    gtk_menu_shell_append(function_keys_menu, f5_item);
    gtk_menu_shell_append(function_keys_menu, f6_item);
    gtk_menu_shell_append(function_keys_menu, function_keys_more_item);

    // add the rest into the More submenu
    gtk_menu_shell_append(function_keys_more_menu, f7_item);
    gtk_menu_shell_append(function_keys_more_menu, f8_item);
    gtk_menu_shell_append(function_keys_more_menu, f9_item);
    gtk_menu_shell_append(function_keys_more_menu, f10_item);
    gtk_menu_shell_append(function_keys_more_menu, f11_item);
    gtk_menu_shell_append(function_keys_more_menu, f12_item);

    // connect signals for each function key item
    g_signal_connect(G_OBJECT(f1_item), "activate", 
		     G_CALLBACK(callback_special_key), mainview);
    g_signal_connect(G_OBJECT(f2_item), "activate", 
		     G_CALLBACK(callback_special_key), mainview);
    g_signal_connect(G_OBJECT(f3_item), "activate", 
		     G_CALLBACK(callback_special_key), mainview);
    g_signal_connect(G_OBJECT(f4_item), "activate", 
		     G_CALLBACK(callback_special_key), mainview);
    g_signal_connect(G_OBJECT(f5_item), "activate", 
		     G_CALLBACK(callback_special_key), mainview);
    g_signal_connect(G_OBJECT(f6_item), "activate", 
		     G_CALLBACK(callback_special_key), mainview);
    g_signal_connect(G_OBJECT(f7_item), "activate", 
		     G_CALLBACK(callback_special_key), mainview);
    g_signal_connect(G_OBJECT(f8_item), "activate", 
		     G_CALLBACK(callback_special_key), mainview);
    g_signal_connect(G_OBJECT(f9_item), "activate", 
		     G_CALLBACK(callback_special_key), mainview);
    g_signal_connect(G_OBJECT(f10_item), "activate", 
		     G_CALLBACK(callback_special_key), mainview);
    g_signal_connect(G_OBJECT(f11_item), "activate", 
		     G_CALLBACK(callback_special_key), mainview);
    g_signal_connect(G_OBJECT(f12_item), "activate", 
		     G_CALLBACK(callback_special_key), mainview);

    // and also setup the other keys menu
    home_item = gtk_menu_item_new_with_label(_("Home"));
    end_item = gtk_menu_item_new_with_label(_("End"));
    pg_up_item = gtk_menu_item_new_with_label(_("Page Up"));
    pg_down_item = gtk_menu_item_new_with_label(_("Page Down"));
    insert_item = gtk_menu_item_new_with_label(_("Insert"));
    del_item = gtk_menu_item_new_with_label(_("Delete"));
    esc_item = gtk_menu_item_new_with_label(_("Esc"));

    // associate keyvalue data with each item
    g_object_set_data(G_OBJECT(home_item), "keyval",
		      GUINT_TO_POINTER(GDK_Home));
    g_object_set_data(G_OBJECT(end_item), "keyval",
		      GUINT_TO_POINTER(GDK_End));
    g_object_set_data(G_OBJECT(pg_up_item), "keyval",
		      GUINT_TO_POINTER(GDK_Page_Up));
    g_object_set_data(G_OBJECT(pg_down_item), "keyval",
		      GUINT_TO_POINTER(GDK_Page_Down));
    g_object_set_data(G_OBJECT(insert_item), "keyval",
		      GUINT_TO_POINTER(GDK_Insert));
    g_object_set_data(G_OBJECT(del_item), "keyval",
		      GUINT_TO_POINTER(GDK_Delete));
    g_object_set_data(G_OBJECT(esc_item), "keyval",
		      GUINT_TO_POINTER(GDK_Escape));

    // add keys to other submenu
    gtk_menu_shell_append(other_keys_menu, home_item);
    gtk_menu_shell_append(other_keys_menu, end_item);
    gtk_menu_shell_append(other_keys_menu, pg_up_item);
    gtk_menu_shell_append(other_keys_menu, pg_down_item);
    gtk_menu_shell_append(other_keys_menu, insert_item);
    gtk_menu_shell_append(other_keys_menu, del_item);
    gtk_menu_shell_append(other_keys_menu, esc_item);

    // connect signals for each key item
    g_signal_connect(G_OBJECT(home_item), "activate", 
		     G_CALLBACK(callback_special_key), mainview);
    g_signal_connect(G_OBJECT(end_item), "activate", 
		     G_CALLBACK(callback_special_key), mainview);
    g_signal_connect(G_OBJECT(pg_up_item), "activate", 
		     G_CALLBACK(callback_special_key), mainview);
    g_signal_connect(G_OBJECT(pg_down_item), "activate", 
		     G_CALLBACK(callback_special_key), mainview);
    g_signal_connect(G_OBJECT(insert_item), "activate", 
		     G_CALLBACK(callback_special_key), mainview);
    g_signal_connect(G_OBJECT(del_item), "activate", 
		     G_CALLBACK(callback_special_key), mainview);
    g_signal_connect(G_OBJECT(esc_item), "activate", 
		     G_CALLBACK(callback_special_key), mainview);

    // finally, allow all the menu and menu items to be shown in the future
    gtk_widget_show_all(GTK_WIDGET(mainview->keys_menu));

    // Set common toolbar for program
    hildon_window_add_toolbar(mainview->data->main_view, GTK_TOOLBAR(mainview->toolbar));

    // Show toolbar
    //if (mainview->bShowNormalToolbar)
    //  gtk_widget_show_all(GTK_WIDGET(mainview->toolbar));
}

void disable_home_key_lock(void *mainvw)
{
  MainView *mainview = (MainView *) mainvw;

  if (mainview->m_bHaveHomeKeyLock)
  {
    osso_rpc_run(mainview->data->osso, TOGGLE_HOME_SERVICE, TOGGLE_HOME_PATH,
		 TOGGLE_HOME_INTERFACE, MESSAGE_ENABLE_HOME_BUTTON,
		 NULL, DBUS_TYPE_INVALID);
    mainview->m_bHaveHomeKeyLock = FALSE;
  }

  if (mainview->m_nKeySnooperId)
  {
    gtk_key_snooper_remove(mainview->m_nKeySnooperId);
    mainview->m_nKeySnooperId = 0;
  }
}

static gboolean app_key_snooper(GtkWidget *widget, GdkEventKey *keyevent,
				HildonProgram *app)
{
  if (keyevent->keyval == GDK_F5)
    return TRUE;

  return FALSE;
}

static void about_dialog_url_hook_function(GtkAboutDialog *about,
					   const gchar *link,
					   gpointer data)
{
  if (!data)
    return;

  osso_context_t *osso = (osso_context_t *) data;
  osso_rpc_run_with_defaults(osso, "osso_browser", "open_new_window", 0,
			     DBUS_TYPE_STRING, link, DBUS_TYPE_INVALID);
}
