/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "user.h"

User::User(Account *account, QGraphicsWidget *parent)
    : TwitterApplicationItem(parent),
    _defaultAvatar(QImage(":images/resources/avatar.png")),
    _bgColor(QColor(246,253,255))
{
    setMinimumSize(310,74);

    // Draw default avatar if the user is new
    _avatar = new QGraphicsPixmapItem(this);
    _avatar->setPos(13, 13);
    _avatar->setZValue(2.0);

    _avatar->setPixmap(QPixmap::fromImage(_defaultAvatar));
    if(account == NULL) {
        _username = "No account. Register a new account to proceed.";
    } else {
        _username = account->getUsername();

        if(!account->avatar.isNull()) {
            _avatar->setPixmap(QPixmap::fromImage(account->avatar));
        }
    }

    // Create text object to draw on the screen
    _name = new QGraphicsTextItem(this);

    // Create Rects
    _avatarRect = new QGraphicsRectItem(this);
    _avatarRect->setBrush(QBrush(QColor(246, 253, 255)));
    _avatarRect->setRect(13, 13, 48, 48);
    _avatarRect->setPen(Qt::NoPen);
    _avatarRect->setZValue(1.0);

    // Draw username on the screen
    drawName(false);

    // Set Orientation
    _changeOrientation();
    connect(QApplication::desktop(), SIGNAL(resized(int)), SLOT(_changeOrientation()));
}

void User::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget)
{
    Q_UNUSED(option);
    Q_UNUSED(widget);

    painter->setBrush(_bgColor);
    painter->setPen(Qt::NoPen);
    painter->setRenderHint(QPainter::Antialiasing);
    painter->drawRoundedRect(0, 0 , rect().width() - 10, rect().height(), 2, 2);
}

void User::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    Q_UNUSED(event);
}

void User::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{
    if(!contains(event->pos()))
        return;
    select();
}

void User::unselect(void)
{
    _bgColor = QColor(246,253,255);
    drawName(false);
    update();
}

void User::drawName(bool isSelected)
{
    QString colorHex = "#ffffff";;
    if(!isSelected)
        colorHex = "#0f95da";

    QString messageHtml = "<p style=\" font-size:18px; padding-bottom:0px; margin-bottom:0px; font-family:'Nokia Sans';\"> ";
    messageHtml += "<span style=\" color:" + colorHex + ";\">" + _username + "</span></p>";
    _name->setHtml(messageHtml);
}

void User::select(void)
{
    _bgColor = QColor(0,173,239);
    drawName(true);
    update();

    emit selected(this);
}

QString User::getUsername(void)
{
    return _username;
}

void User::_changeOrientation(void)
{
    QRect screenGeometry = QApplication::desktop()->screenGeometry();

    setMinimumSize(-1, -1);
    setMaximumSize(-1, -1);

    if(screenGeometry.width() > screenGeometry.height()) {
        _name->setTextWidth(240);
        setMinimumSize(310, 74);
        setMaximumSize(310, 74);
    } else if (screenGeometry.width() < screenGeometry.height()) {
        _name->setTextWidth(400);
        setMinimumSize(475, 74);
        setMaximumSize(475, 74);
    }

    _name->setPos(74, (rect().height() - _name->boundingRect().height())/2);
}
