/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "loader.h"

#include <QtGui/QDesktopWidget>
#include <QtGui/QApplication>
#include <QtGui/QPen>

/*!
 * \brief Loader constructor.
 */
Loader::Loader(QGraphicsWidget *parent)
        : QGraphicsWidget(parent)
{
    _bg = new QGraphicsRectItem(this);
    _bg->setBrush(Qt::black);
    _bg->setOpacity(0.3);
    _bg->setZValue(1.0);
    _bg->setPen(QPen(Qt::NoPen));

    _image = new QGraphicsWidget(this);
    _image->setZValue(1.0);
    /* This is necessary to set loader image center used by rotate function */
    _image->setTransformOriginPoint(QPointF(24, 24));

    QGraphicsPixmapItem *img = new QGraphicsPixmapItem(_image);
    img->setPixmap(QPixmap(":images/resources/preloader.png"));
    img->setCacheMode(DeviceCoordinateCache);
    img->setTransformationMode(Qt::SmoothTransformation);

    _image->setGeometry(img->boundingRect());

    _entryAnimation = 0;
    _exitAnimation = 0;

    // Change orientation
    _changeOrientation();
    connect(QApplication::desktop(), SIGNAL(resized(int)), SLOT(_changeOrientation()));

    setLoaderPosition(QPointF((size().width() - _image->size().width())/2, (size().height() - _image->size().height())/2));
}

/*!
 * \brief Loader destructor.
 */
Loader::~Loader()
{
    if (_entryAnimation) {
        delete _entryAnimation;
    }
    if (_exitAnimation) {
        delete _exitAnimation;
    }
    if (_image) {
        delete _image;
    }
}

void Loader::setLoaderPosition(QPointF pos)
{
    _changeOrientation();
    _image->setPos(pos);
}

/*!
 * \brief Starts the Entry animation. Loader image
 * is shown going UP.
 */
void Loader::startEntryAnimation(void)
{
    setEntryAnimation();
    _entryAnimation->start();
}

/*!
 * \brief Creates the Entry animation group.
 */
void Loader::setEntryAnimation(void)
{
    /* Rotation */
    _animRotate = new QPropertyAnimation();
    _animRotate->setTargetObject(_image);
    _animRotate->setPropertyName("rotation");
    _animRotate->setStartValue(0.0);
    _animRotate->setEndValue(360.0);
    _animRotate->setDuration(400);
    _animRotate->setEasingCurve(QEasingCurve::Linear);
    _animRotate->setLoopCount(-1);  // Loop until stopped

    /* Opacity */
    _animOpacity = new QPropertyAnimation();
    _animOpacity->setTargetObject(_image);
    _animOpacity->setPropertyName("opacity");
    _animOpacity->setStartValue(0.0);
    _animOpacity->setEndValue(1.0);
    _animOpacity->setDuration(500);
    _animOpacity->setEasingCurve(QEasingCurve::Linear);

    /* Adding the animations to the group */
    _entryAnimation = new QParallelAnimationGroup(this);
    _entryAnimation->addAnimation(_animRotate);
    _entryAnimation->addAnimation(_animOpacity);
}

/*!
 * \brief Starts the Exit animation. Loader image
 * is shown going DOWN.
 */
void Loader::startExitAnimation(void)
{
    setExitAnimation();

    /* When the Exit animation stops,
     * the Entry animation also stops. */
    connect(_exitAnimation, SIGNAL(finished()), _entryAnimation, SLOT(stop()));

    /* When the entry animation stops,
     * a finished() signal is emitted. */
    connect(_entryAnimation, SIGNAL(finished()), SIGNAL(finished()));

    _exitAnimation->start();
}

/*!
 * \brief Creates the Exit animation group.
 */
void Loader::setExitAnimation(void)
{
    _animOpacity->setTargetObject(_image);
    _animOpacity->setPropertyName("opacity");
    _animOpacity->setStartValue(1.0);
    _animOpacity->setEndValue(0.0);
    _animOpacity->setDuration(500);
    _animOpacity->setEasingCurve(QEasingCurve::Linear);

    _exitAnimation = new QParallelAnimationGroup(this);
    _exitAnimation->addAnimation(_animOpacity);
}

QPointF Loader::getLoaderPos(void)
{
    return _image->pos();
}

void Loader::_changeOrientation(void)
{
    setGeometry(parentWidget()->rect());
    _bg->setRect(parentWidget()->rect());
}
