/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "urlshortener.h"

/*! \brief UrlShortener constructor.
 */
UrlShortener::UrlShortener()
{
    connection = new QNetworkAccessManager( this );

    connect( connection, SIGNAL(finished(QNetworkReply*)), this, SLOT(replyFinished(QNetworkReply*)) );
    connect(this, SIGNAL(shortened(QString)), this, SLOT(appendToList(QString)));
}

/*!
 * \brief UrlShortener destructor.
 */
UrlShortener::~UrlShortener()
{
}

/*!
 * \brief Sends a request to http://is.gd to shorten all urls on the list.
 * \param url List of Urls
 */
void UrlShortener::shorten( const QStringList &url )
{
    for(int i = 0; i < url.count(); i++) {
        if( QRegExp( "http://is.gd/" ).indexIn( url.at(i) ) == -1 )
            connection->get( QNetworkRequest( QUrl( "http://is.gd/api.php?longurl=" + url.at(i) ) ) );
    }
}

/*!
 * \brief This function reads the response and emits a signal with the shortened
 * url if successful or an error signal otherwise.
 * \param reply response object.
 */
void UrlShortener::replyFinished( QNetworkReply * reply )
{
    QString response = reply->readLine();
    QString message;

    switch( replyStatus( reply ) ) {
        case 200:
    	    emit shortened( response );
            message = "Shortening sucessed";
            break;
  	case 401:
            message = "The url shortening service couldn't authorize you. Please check your username and password.";
            urlList.clear();
	    break;
	case 500: 
            message = "Error: Could not shorten your URL.";
            urlList.clear();
      	    break;
    	default:
            message = "An unknown error occurred when shortening your URL.";
            urlList.clear();
            break;
    }

    QMaemo5InformationBox::information(NULL, message, QMaemo5InformationBox::DefaultTimeout);
}

/*!
 * \brief Converts the http status code to int.
 * \param reply Response object.
 * \return status code.
 */
int UrlShortener::replyStatus( QNetworkReply *reply ) 
{
    return reply->attribute( QNetworkRequest::HttpStatusCodeAttribute ).toInt();
}

/*!
 * \brief This function parses the message, stores all urls in a list
 * and calls sends a request to shorten the urls.
 */
void UrlShortener::parseUrl(QString message)
{
    QMaemo5InformationBox::information(NULL, tr("Shortening Url(s) in tweet"), QMaemo5InformationBox::DefaultTimeout);

    list = message.split(" ");

    QListIterator<QString> iteratorUrl(list);

    while (iteratorUrl.hasNext()) {
        QString nextWord = iteratorUrl.next();

        if ( nextWord.contains( QRegExp("http://*") ) )
             urlList.append(nextWord);

    }
    this->shorten(urlList);
}

/*!
 * \brief This function adds the shortened url to the original message and
 * emits a signal with the new message.
 */
void UrlShortener::appendToList(QString shortUrl)
{
    temp.append(shortUrl);

    if(temp.count() == urlList.count()) {
        QListIterator<QString> iteratorUrl(list);
        int cont = 0;
        int i = 0;

        while (iteratorUrl.hasNext()) {
            QString nextWord = iteratorUrl.next();

            if( nextWord.contains( QRegExp("http://*") ) ) {
                list.replace(i, temp.at(cont));
                cont++;
            }
            i++;
        }
        QString ret;
        for(int i = 0; i < list.count(); i++)
            ret.append(list.at(i) + " ");
	
        emit newString(ret);
        ret.clear();
        temp.clear();
        urlList.clear();
        list.clear();
    }
}
