/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "timelineview.h"

/*!
 * \brief TimelineView constructor.
 */
TimelineView::TimelineView(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _exitAnim = 0;
    _urlShortener = new UrlShortener;
    _coordinates = new IpLocation;
}

/*!
 * \brief TimelineView destructor.
 */
TimelineView::~TimelineView()
{
    if(this->_topBar)
        delete this->_topBar;
    if(this->_timeline)
        delete this->_timeline;
}

/*!
 * \brief Creates the TimelineView objects and animations.
 */
void TimelineView::createUi(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    startLoader();

    _createEntry();
    _createTopBar();
    _createLabels();
    _createButtons();

    /* Animations */
    _entryAnim = _createEntryAnimation();
    _entryAnim->start();
    _exitAnim = _createExitAnimation();

    connect(qobject_cast<QObject *>(_exitAnim), SIGNAL(finished()), this,
            SLOT(_exitAnimationFinished()));
    connect(_updateButton, SIGNAL(released()), this, SLOT(_emitPostUpdate()));
    connect(_entryUpdate, SIGNAL(sizeText(int)), this,SLOT(_changeCounter(int)));

    connect(_urlShortener, SIGNAL(newString(QString)), this, SLOT(_handleNewMessage(QString)));
    connect(_urlShortener, SIGNAL(errorMessage(QString)), this, SLOT(urlShorteningError(QString)));
    connect(_refreshButton, SIGNAL(released()), this, SLOT(_emitRefresh()));
    connect(_logoutButton, SIGNAL(released()), this, SIGNAL(logout()));
    connect(_entryUpdate, SIGNAL(enterPressed()), this, SLOT(_emitPostUpdate()));
    connect(_backButton, SIGNAL(released()), this, SIGNAL(back()));
    connect(_exposeButton, SIGNAL(released()), this, SLOT(_emitExposeSignal()));
    connect(_moreTweetsButton,SIGNAL(released()),this,SLOT(_emitMoreTweets()));
    connect(_topBar,SIGNAL(finished()),this,SLOT(createTimeline()));
    connect(_topBar, SIGNAL(changeAvatar(QString)), this, SLOT(_emitChangeAvatar(QString)));
    connect(_locationCheckBox, SIGNAL(pressed()), this, SLOT(getCoordinates()));
}

/*!
 * \brief Emits a signal to timelinectrl to download more tweets.
 */
void TimelineView::_emitMoreTweets(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    if(!_loader)
        startLoader();

    emit moreTweets();
}

/*!
 * \brief Creates the timeline.
 */
void TimelineView::createTimeline()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _timeline = new TwitterTimeline(NULL);

    map = new GMaps(NULL);

    connect(_timeline, SIGNAL(finished()), this, SLOT(endLoader()));
    connect(_timeline, SIGNAL(createTweetInfo()), this, SLOT(_createTweetInfo()));
    connect(_timeline, SIGNAL(createTweetInfo()), this, SIGNAL(moveTimeline()));

    emit showTimeline();
}

/*!
 * \brief This slot is called when the a transition back to the login screen is required.
 * This method calls all initial animations backwards.
 */
void TimelineView::_exitAnimationFinished(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    emit done();
}

/*!
 * \brief This method creates the entry animation for all objects on the screen.
 * \return anim Entry animation.
 */
QAbstractAnimation *TimelineView::_createEntryAnimation(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    QParallelAnimationGroup *anim = new QParallelAnimationGroup(this);

    QEasingCurve easingCurve;
    easingCurve.setType(QEasingCurve::OutBounce);

    /* Buttons */
    _addOpacityAnim(anim, _updateButton, 0);
    _addPosAnim(anim, _updateButton, QPointF(800 - _updateButton->boundingRect().width() ,480 - (_updateButton->boundingRect().height()) + 1 - 50),
                QPointF(800 - _updateButton->boundingRect().width() ,480 - (_updateButton->boundingRect().height()) + 1),
                easingCurve, 0);
    _addOpacityAnim(anim, _refreshButton, 0);
    _addPosAnim(anim, _refreshButton, QPointF(_exposeButton->boundingRect().width() + 4, 0 + 50),
                QPointF(_exposeButton->boundingRect().width() + 4, 0),
                easingCurve, 0);
    _addOpacityAnim(anim, _exposeButton, 0);
    _addPosAnim(anim, _exposeButton, QPointF(0, 0 + 50),
                QPointF(0, 0),
                easingCurve, 0);
    _addOpacityAnim(anim, _logoutButton, 0);
    _addPosAnim(anim, _logoutButton, QPointF(800 - _refreshButton->boundingRect().width(), 0 + 50),
                QPointF(800 - _refreshButton->boundingRect().width(), 0),
                easingCurve, 0);
    _addOpacityAnim(anim, _moreTweetsButton, 0);
    _addPosAnim(anim, _moreTweetsButton, QPoint(_exposeButton->boundingRect().width() + _refreshButton->boundingRect().width() + 8, 0 + 50),
                QPointF(_exposeButton->boundingRect().width() + _refreshButton->boundingRect().width() + 8, 0),
                easingCurve, 0);

    /* TopBar */
    _addOpacityAnim(anim, _topBar, 0);
    _addPosAnim(anim, _topBar, QPointF(800 - _refreshButton->boundingRect().width() - _topBar->boundingRect().width() -8, 0 + 50),
                QPointF(800 - _refreshButton->boundingRect().width() - _topBar->boundingRect().width() -8, 0),
                easingCurve, 0);

     /* Update Entry */
    _addOpacityAnim(anim, _entryUpdate, 0);
    _addPosAnim(anim, _entryUpdate, QPointF(0, 481 - _entryUpdate->boundingRect().height() - 50),
                QPointF(0, 481 - _entryUpdate->boundingRect().height()),easingCurve, 0);

    /* Location CheckBox */
    _addOpacityAnim(anim, _locationCheckBox, 0);
    _addPosAnim(anim, _locationCheckBox, QPointF(_entryUpdate->boundingRect().width() ,480 - (_updateButton->boundingRect().height()) + 1 - 50),
                QPointF(_entryUpdate->boundingRect().width() ,480 - (_updateButton->boundingRect().height()) + 1),
                easingCurve, 0);

    return anim;
}

/*!
 * \brief This method creates the exit animation for all objects on the screen.
 * \return anim Exit animation.
 * This method gets the entry animation and sets it backwards as the exit animation.
 */
QAbstractAnimation *TimelineView::_createExitAnimation(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    QParallelAnimationGroup *anim = qobject_cast<QParallelAnimationGroup *>(_createEntryAnimation());

    anim->setDirection(QAbstractAnimation::Backward);
    return anim;
}

/*!
 * \brief Creates the top bar containing user information.
 */
void TimelineView::_createTopBar()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    _topBar = new TwitterTopBar(this);
}

/*!
 * \brief Creates the labels on the screen.
 * There are eight labels: username, #tweets, #followers, #following
 * also, 140-counter, and tweets, followers and following labels.
 */
void TimelineView::_createLabels()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    QFont font;
    font.setFamily("Nokia Sans");
    font.setStyle(QFont::StyleNormal);
    font.setStyleStrategy(QFont::PreferAntialias);
    font.setPixelSize(16);
    font.setBold(true);

    _labelUpdateCounter = new QGraphicsTextItem(this);
    _labelUpdateCounter->setPlainText(tr("140"));
    _labelUpdateCounter->setDefaultTextColor(Qt::white);
    _labelUpdateCounter->setFont(font);
    _labelUpdateCounter->setPos(3,440);
}

/*!
 * \brief Creates the buttons on the screen.
 */
void TimelineView::_createButtons()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _refreshButton = new TwitterButton(TwitterButton::Refresh, NULL, this);
    _refreshButton->setOpacity(0);

    _updateButton = new TwitterButton(TwitterButton::Update,"Tweet",this);
    _updateButton->setOpacity(0);

    _exposeButton = new TwitterButton(TwitterButton::Expose, NULL, this);
    _exposeButton->setOpacity(0);

    _logoutButton = new TwitterButton(TwitterButton::Logout, "Logout", this);
    _logoutButton->setOpacity(0);

    _moreTweetsButton =  new TwitterButton(TwitterButton::More, "More",this);
    _moreTweetsButton->setOpacity(0);

    _backButton = new TwitterButton(TwitterButton::Back, NULL, this);
    _backButton->setOpacity(1.0);
    _backButton->setPos(QPointF(810, 0));

    _locationCheckBox = new TwitterCheckBox(TwitterCheckBox::Location,
    QPointF(_entryUpdate->boundingRect().width(), 480 - _updateButton->boundingRect().height() + 1), false, this);
    _locationCheckBox->setOpacity(1);
}

/*!
 * \brief Creates the entry to write a new status message.
 */
void TimelineView::_createEntry()
{
     qDebug() << "+ " << __PRETTY_FUNCTION__;
    /* Add the entries to the scene */
    // ID entry
    _entryUpdate = new LineEdit(LineEdit::Long,this);
    _entryUpdate->setOpacity(0);
    _entryUpdate->setPos(QPointF(100,435));
}

/*!
 * \brief Creates tweetinfo object.
 */
void TimelineView::_createTweetInfo()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _tweetInfo = new TwitterTweetInfo(_timeline, _topBar, this);
    _tweetInfo->setZValue(-1.0);

    QRectF r = QRectF(8, 62, 792, 422);
    _tweetInfo->setGeometry(r);

    connect(_tweetInfo, SIGNAL(retweet()), this, SIGNAL(retweet()));
    connect(_tweetInfo, SIGNAL(reply()), this, SIGNAL(reply()));
    connect(_tweetInfo, SIGNAL(unfollow()), this, SLOT(_emitUnfollow()));
    connect(_tweetInfo, SIGNAL(del()), this, SLOT(_emitDelete()));
    connect(_tweetInfo, SIGNAL(favorite(QString)), this, SLOT(_emitFavorite(QString)));
}

/*!
 * \brief Adds opacity animation.
 * \param item the item that will be animated.
 * \param anim the item's animation.
 * \param delay to start each item's animation.
 */
void TimelineView::_addOpacityAnim(QParallelAnimationGroup *anim, TwitterApplicationItem *item, int delay)
{
    QPropertyAnimation *animation = new QPropertyAnimation();

    animation->setTargetObject(item);
    animation->setPropertyName("opacity");
    animation->setStartValue(0.0);
    animation->setEndValue(1.0);
    animation->setDuration(500);
    animation->setEasingCurve(QEasingCurve::Linear);

    if (delay > 0) {
        QSequentialAnimationGroup *delayedAnimation;
        delayedAnimation = new QSequentialAnimationGroup();
        delayedAnimation->addPause(delay);
        delayedAnimation->addAnimation(animation);
        anim->addAnimation(delayedAnimation);
    } else {
        anim->addAnimation(animation);
    }
}

/*!
 * \brief Adds the position animation.
 * \param item the item that will be animated
 * \param anim the item's animation
 * \param delay to start each item's animation
 * \param startValue the start position of the animation
 * \param endValue the end position of the animation
 * \param easing the easing curve for the movimentation animation
 */
void TimelineView::_addPosAnim(QParallelAnimationGroup *anim,
                            TwitterApplicationItem *item,
                            QPointF startValue, QPointF endValue,
                            QEasingCurve easing, int delay)
{
    QPropertyAnimation *animation = new QPropertyAnimation();

    animation->setTargetObject(item);
    animation->setPropertyName("pos");
    animation->setStartValue(startValue);
    animation->setEndValue(endValue);
    animation->setDuration(500);
    animation->setEasingCurve(easing);

    if (delay > 0) {
        QSequentialAnimationGroup *delayedAnimation;
        delayedAnimation = new QSequentialAnimationGroup();
        delayedAnimation->addPause(delay);
        delayedAnimation->addAnimation(animation);
        anim->addAnimation(delayedAnimation);
    } else {
        anim->addAnimation(animation);
    }
}


/*! \brief Gets the message typed on message edit box.
 *  \return message Message typed.
 */
QString TimelineView::getUpdateMessage(void)
{
    QWidget *widget;

    widget = _entryUpdate->getEntry()->widget();
    return (dynamic_cast<QLineEdit*>(widget))->text();
}

/*! \brief Sets the message in the message edit box.
 *  \param[in] message The user's message.
 */
void TimelineView::setUpdateMessage(const QString &message)
{
    QWidget *widget;

    widget = _entryUpdate->getEntry()->widget();
    (dynamic_cast<QLineEdit*>(widget))->setText(message);

    this->_changeCounter(_entryUpdate->getTextLength());
}

/*!
 * \brief This slot is called when the user pushes the "update" button.
 * This method starts the loader then sends a signal to the controller to try to login.
 */
void TimelineView::_emitPostUpdate()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    if(number.toInt() < 0) {
        if( this->containsUrl(this->getUpdateMessage()) > 0 ) {
            startLoader();
            _urlShortener->parseUrl(this->getUpdateMessage());
        }
    }
    else {
        startLoader();
        emit postUpdate();
    }
}

/*!
 * \brief Sets the new message on the textbox after the url was shortened.
 * \param newMessage message
 */
void TimelineView::_handleNewMessage(QString newMessage)
{
    this->setUpdateMessage(newMessage);
    this->endLoader();
    _entryUpdate->setCursorPosition(0);
    this->_changeCounter(_entryUpdate->getTextLength());
    _entryUpdate->setTextLength(_entryUpdate->getTextLength());
}

/*!
 * \brief Updates message's length counter.
 */
void TimelineView::updateLabels()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _labelUpdateCounter->setPlainText(tr("140"));
}

/*! \brief This function shows how many characters are left to type.
 */
void TimelineView::_changeCounter(int size)
{
    number = QString::number(140 - size);

    if(number.toInt() < 0)
    {
        _updateButton->enable(false);
        if(this->containsUrl(this->getUpdateMessage()) > 0) {
            _updateButton->enable(true);
            if( QRegExp( "http://is.gd/" ).indexIn( this->getUpdateMessage() ) != -1 ) {
                _updateButton->enable(false);
            }
        }
    }
    else
    {
        _labelUpdateCounter->setDefaultTextColor(Qt::white);
        _updateButton->enable(true);
    }
    _labelUpdateCounter->setPlainText(number);
}

/*! \brief Calls the loader when user clicks on update.
 */
void TimelineView::startLoader()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    _loader = new Loader(this);
    connect(_loader, SIGNAL(finished()), this, SLOT(_deleteLoader()));

    _loader->setZValue(1.0);
    QGraphicsRectItem *bg = _loader->getBackground();
    bg->setOpacity(0.6);

    setEnabled(false);
    _loader->startEntryAnimation();
}

/*! \brief Deletes loader object when loader finishes.
 */
void TimelineView::_deleteLoader()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    delete _loader;
    _loader = NULL;
    this->setEnabled(true);
}

/*!
 * \brief This slot is called when there is any problem with the connection.
 * This slot starts the animation for the transition to the login screen.
 */
void TimelineView::connectionError(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _loader->startExitAnimation();
    if (_exitAnim) {
        _exitAnim->start();
    } else {
        emit done();
    }
}

/*!
 * \brief Calls generic error on timelinectrl when an error while shortening the url occurs.
 * \param errorMessage Error message
 */
void TimelineView::urlShorteningError(QString errorMessage)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    emit shorteningError(errorMessage);
}

/*! \brief This slot is called to start loader's exit animation.
 */
void TimelineView::endLoader()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    _loader->startExitAnimation();
}

/*!
 * \brief Updates the topbar with new information about the user.
 */
void TimelineView::updateTopBar(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    _topBar->updateTopBar(false);
    _topBar->setPos(QPointF(800 - _refreshButton->boundingRect().width() - _topBar->boundingRect().width() -8, 0));
}

/*!
 * \brief this slot is called when the app has the timeline
 * This slot enable the buttons on the screen so the user can request more tweets
 * or post new status
 */
void TimelineView::_hasTimeline(void)
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;
    _updateButton->enable(true);
}

/*!
 * \brief Updates Parses the message checking if it has any url.
 * \param message message
 * \return cont Number of urls.
 */
int TimelineView::containsUrl(QString message)
{
    QList<QString> temp = message.split(" ");

        QListIterator<QString> iteratorUrl(temp);
    int cont = 0;
    while (iteratorUrl.hasNext()) {
        QString nextWord = iteratorUrl.next();

        if ( nextWord.contains( QRegExp("http://*") ) ){
            cont++;
        }
    }
    return cont;
}

/*!
 * \brief Starts exit animation to go back to login screen.
 */
void TimelineView::_backToLogin()
{
    if (_exitAnim) {
        _exitAnim->start();
    } else {
        emit done();
    }
}

/*!
 * \brief Emits a DBus signal to Hildon Desktop who manages the app window with.
 * the task swticher when the expose button is pressed.
 */
void TimelineView::_emitExposeSignal()
{
    QDBusConnection bus = QDBusConnection::sessionBus();
    QDBusMessage message = QDBusMessage::createSignal("/", "com.nokia.hildon_desktop", "exit_app_view");
    bus.send(message);
}

/*!
 * \brief Emits an Unfollow signal.
 */
void TimelineView::_emitUnfollow()
{
    startLoader();
    emit unfollow();
}

/*!
 * \brief Emits a delete signal.
 */
void TimelineView::_emitDelete()
{
    startLoader();
    emit deleteTweet();
}

/*!
 * \brief Emits a favorite signal.
 * \param isFavorite true if status is favorite and false otherwise.
 */
void TimelineView::_emitFavorite(QString isFavorite)
{
    startLoader();
    emit favorite(isFavorite);
}

/*!
 * \brief Emits a refresh signal.
 */
void TimelineView::_emitRefresh()
{
    startLoader();
    emit refresh();
}

/*!
 * \brief Get geographical goordinates.
 */
void TimelineView::getCoordinates()
{
    if(!_locationCheckBox->isChecked())
        _coordinates->getLocation();
}

/*!
 * \brief Emits a signal to change user's avatar.
 * \param image New image to upload.
 */
void TimelineView::_emitChangeAvatar(QString image)
{
    startLoader();
    emit changeAvatar(image);
}
