/*
 *
 *  Copyright (c) 2010 Zagaia (INdT - Instituto Nokia de Tecnologia/
 *       FUCAPI - Fundação Centro de Análise, Pesquisa e Inovação Tecnológica)
 *
 *  This file is part of TweeGo.
 *
 *  TweeGo is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TweeGo is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TweeGo. If not, see <http://www.gnu.org/licenses/>
 *
 */

#include "twittertweetinfo.h"

/*! \brief TwitterTweetInfo constructor.
 *  \param m_username Twitter username
 *  \param m_text Text of respective username
 *  \param m_imageUrl Profile image of respective username
 */
TwitterTweetInfo::TwitterTweetInfo(TwitterTimeline *timeline, TwitterTopBar *topBar, QGraphicsWidget *parent)
                : TwitterApplicationItem(parent)
{
    font.setFamily("Nokia Sans");
    font.setStyle(QFont::StyleNormal);
    font.setStyleStrategy(QFont::PreferAntialias);
    font.setWeight(QFont::Normal);
    font.setPixelSize(18);

    font2.setFamily("Nokia Sans");
    font2.setStyle(QFont::StyleNormal);
    font2.setStyleStrategy(QFont::PreferAntialias);
    font2.setWeight(QFont::Normal);
    font2.setPixelSize(14);

//    setAttribute(Qt::WA_OpaquePaintEvent, true);
    _tweetName = timeline->getTweetName();
    _tweetText = timeline->getTweetText();
    _tweetUrl = timeline->getTweetUrl();
    _tweetId = timeline->getTweetId();
    _tweetSource = timeline->getTweetSource();
    _tweetCreatedAt = timeline->getTweetCreatedAt();
    _tweetUserImage = timeline->getUserImage();
    _username = topBar->getUsername();

    _favorite = (timeline->getTweetFavorite()=="true")?true:false;
    _listUrls = _createUrls();

    _createButtons();
}

/*!
 * \brief TimelineView destructor.
 */
TwitterTweetInfo::~TwitterTweetInfo()
{
    qDebug() << "~  " << __PRETTY_FUNCTION__;
}

/*!
 * \brief Overridden.
 * \param painter Painter object.
 * \param option Styles to drawing.
 */
void TwitterTweetInfo::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *)
{
    Q_UNUSED(option);

    /* Draw the white rect */
    painter->setPen(Qt::NoPen);
    painter->setBrush(Qt::white);
    painter->drawRect(QRect(8,62,784,361));

    painter->drawImage(QRectF(20,72,48,48),_tweetUserImage);

    painter->setFont(font);
    painter->setPen(QColor(15, 149, 218));

    QRectF timelineRect = QRect(80, 64, 704, 65);
    QRectF linkRect = QRect(20, 240, 750, 165);

    /* Writes the Tweet*/
    painter->setPen(QColor(31, 31, 31));
    painter->drawText(timelineRect, Qt::TextWordWrap, _tweetName + " " + _tweetText);
    painter->setPen(QColor(255, 255, 255));
    painter->drawText(timelineRect, Qt::AlignJustify, _tweetName);
    painter->setPen(QColor(15, 149, 218));
    painter->drawText(timelineRect, Qt::AlignJustify, _tweetName);

    /* Writes the time info*/
    painter->setFont(font2);
    painter->setPen(QColor(117, 117, 117));
    painter->drawText(timelineRect, Qt::AlignRight | Qt::AlignBottom, _tweetCreatedAt + " via " + _tweetSource);

    /* Draws lines */
    painter->setPen(QColor(216, 216, 216));
    painter->drawLine(10, 134 , 788 , 134);
    painter->drawLine(10, 209 , 788 , 209);

    /* Writes link info */
    painter->setPen(Qt::black);
    painter->setFont(font);
    if(_listUrls.size() > 0)
        painter->drawText(20, 218, 788,201,Qt::AlignJustify, "Links found on this tweet:");
    else
        painter->drawText(20, 218, 788,201,Qt::AlignJustify, "No links available on this tweet");

    QFontMetrics fm(font);
    painter->setPen(QColor(15, 149, 218));

    for(int i = 0; i< _listUrls.size(); i++) {
        if(((QString)_listUrls[i]).length()>= 30)
             painter->drawText(20, 210 + (36)*(i+1), fm.width(_listUrls[i])+5, fm.height(), Qt::AlignJustify, (QString(_listUrls[i])).left(15)+"..."+(QString(_listUrls[i])).right(15));
        else
            painter->drawText(20, 210 + (36)*(i+1), fm.width(_listUrls[i])+5, fm.height(), Qt::AlignJustify, _listUrls[i]);
    }
}

/*!
 * \brief This function parses the message and puts all URLs that were found in a list.
 * \return URL list.
 */
QStringList TwitterTweetInfo::_createUrls(){

    QStringList auxUrls;
    QStringList listUrls;
    auxUrls = _tweetText.split(" ");

    QListIterator<QString> iteratorUrl(auxUrls);
    while (iteratorUrl.hasNext()) {
            QString nextWord = iteratorUrl.next();

            if( nextWord.contains( QRegExp("http://*") ) )
                listUrls.append(nextWord);
    }

    return listUrls;
}

/*!
 * \brief This function creates all buttons on the detailed tweet screen.
 */
void TwitterTweetInfo::_createButtons()
{
    qDebug() << "+ " << __PRETTY_FUNCTION__;

    if(_favorite) {
        _favoriteButton = new TwitterButton(TwitterButton::Favorite, NULL, this);
    }
    else
        _favoriteButton = new TwitterButton(TwitterButton::NoFavorite, NULL, this);

    _favoriteButton->setPos(784-_favoriteButton->boundingRect().width(), 144);
    _favoriteButton->setOpacity(1.0);

    connect(_favoriteButton, SIGNAL(released()), this, SLOT(favoriteInfo()));

    int width = _favoriteButton ->boundingRect().width();

    if(_tweetName.compare(_username)) {
        _unfollowButton = new TwitterButton(TwitterButton::Generic, "Unfollow", this);
        _unfollowButton->setPos(20, 144);
        connect(_unfollowButton, SIGNAL(released()), this, SIGNAL(unfollow()));
    }

    if(!_tweetName.compare(_username)) {
        _deleteButton = new TwitterButton(TwitterButton::Delete, NULL , this);
        _deleteButton->setPos(788 - ( (width + 5) * 4), 144);
        connect(_deleteButton,SIGNAL(released()), this, SIGNAL(del()));
    }

    _retweetButton = new TwitterButton(TwitterButton::Retweet, NULL, this);
    _retweetButton->setPos(788 - ( (width + 5) * 3), 144);

    _replyButton = new TwitterButton(TwitterButton::Reply  , NULL, this);
    _replyButton->setPos(788 - ( (width + 5) * 2), 144);

    QList<TwitterButton*> listButtons;

    QFontMetrics fm(font);

    int ant, widthUrl = 0;
    
    for(int i = 0; i< _listUrls.size(); i++) {
        if(((QString)_listUrls[i]).length()>= 30)
            ant = fm.width(QString((QString(_listUrls[i])).left(15)+"..."+(QString(_listUrls[i])).right(15)));
        else
            ant = fm.width(_listUrls[i]);

        if (ant > widthUrl)
            widthUrl = ant;
     }

    for(int i = 0; i< _listUrls.size(); i++) {
        _urlButton = new TwitterButton(TwitterButton::GoUrl , NULL, _listUrls[i] , this);
        _urlButton->setPos(30 + widthUrl, 205 + (35 * (i+1)));
        listButtons << _urlButton;
        connect(listButtons.at(i), SIGNAL(released(QString)), this, SLOT(goToUrl(QString)));
    }

    connect(_retweetButton, SIGNAL(released()), this, SIGNAL(retweet()));
    connect(_replyButton, SIGNAL(released()), this, SIGNAL(reply()));
}

/*!
 * \brief This function opens the browser and loads the URL.
 * \param url URL.
 */
void TwitterTweetInfo::goToUrl(QString url)
{
    QDesktopServices::openUrl(url);
}


void TwitterTweetInfo::favoriteInfo(void){


    QString favoriteinfo = (_favorite)?"true":"false";
    _favorite = !_favorite;
    emit favorite(favoriteinfo);

}
