/*
 * vim:ts=4:sw=4:et:cindent:cino=(0
 */ 

#include <config.h>

#include <string.h>
#include <unistd.h>

#include <gmodule.h>

#include "tweakr-types.h"
#include "libtweakr-section/tweakr-module.h"
#include "tweakr-module-manager.h"


enum
{
    PROP_0,
    PROP_MODULE_PATH
};


static GObject *
tweakr_module_manager_constructor (GType type,
                                         guint n_params,
                                         GObjectConstructParam *params);
static void
tweakr_module_manager_finalize (GObject *object);

static void
tweakr_module_manager_get_property (GObject *object,
                                          guint param_id,
                                          GValue *value,
                                          GParamSpec *pspec);

static void
tweakr_module_manager_set_property (GObject *object,
                                          guint param_id,
                                          const GValue *value,
                                          GParamSpec *pspec);

static gboolean
tweakr_module_manager_query_modules
    (TweakrModuleManager *manager);


G_DEFINE_TYPE (TweakrModuleManager, tweakr_module_manager,
               G_TYPE_OBJECT);


static void
tweakr_module_manager_class_init (TweakrModuleManagerClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    object_class->constructor  = tweakr_module_manager_constructor;
    object_class->finalize     = tweakr_module_manager_finalize;
    object_class->get_property = tweakr_module_manager_get_property;
    object_class->set_property = tweakr_module_manager_set_property;

    g_object_class_install_property
        (object_class, PROP_MODULE_PATH,
         g_param_spec_string ("module-path",
                              "Module Path",
                              "The path where to look for modules",
                              NULL,
                              G_PARAM_READWRITE |
                              G_PARAM_CONSTRUCT_ONLY));
}

static void
tweakr_module_manager_init (TweakrModuleManager *manager)
{
    manager->module_path = NULL;
    manager->modules     = NULL;
}

static GObject *
tweakr_module_manager_constructor (GType type,
                                         guint n_params,
                                         GObjectConstructParam *params)
{
    GObject *object;
    TweakrModuleManager *manager;

    object = G_OBJECT_CLASS
        (tweakr_module_manager_parent_class)->constructor (type,
                                                                 n_params,
                                                                 params);

    manager = TWEAKR_MODULE_MANAGER (object);

    if (manager->module_path)
        tweakr_module_manager_query_modules (manager);

    return object;
}

static void
tweakr_module_manager_finalize (GObject *object)
{
    TweakrModuleManager *manager = TWEAKR_MODULE_MANAGER (object);

    g_free (manager->module_path);

    /* GTypeModules most not be finalized, don't unref them */
    g_list_free (manager->modules);

    G_OBJECT_CLASS (tweakr_module_manager_parent_class)->
        finalize (object);
}

static void
tweakr_module_manager_get_property (GObject *object,
                                          guint param_id,
                                          GValue *value,
                                          GParamSpec *pspec)
{
    TweakrModuleManager *manager = TWEAKR_MODULE_MANAGER (object);

    switch (param_id)
    {
        case PROP_MODULE_PATH:
            g_value_set_string (value, manager->module_path);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static void
tweakr_module_manager_set_property (GObject *object,
                                          guint param_id,
                                          const GValue *value,
                                          GParamSpec *pspec)
{
    TweakrModuleManager *manager = TWEAKR_MODULE_MANAGER (object);

    switch (param_id)
    {
        case PROP_MODULE_PATH:
            manager->module_path = g_value_dup_string (value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
            break;
    }
}

static gboolean
tweakr_module_manager_is_valid_module_name (const gchar *name)
{
    return g_str_has_suffix (name, ".so");
}

static gboolean
tweakr_module_manager_query_modules (TweakrModuleManager *manager)
{
    const gchar *name;
    GDir        *dir;
    GError      *error = NULL;

    dir = g_dir_open (manager->module_path, 0, &error);

    if (!dir)
    {
        g_printerr ("Error while opening module dir: %s\n", error->message);
        g_clear_error (&error);
        return FALSE;
    }

    while ((name = g_dir_read_name (dir)))
    {
        if (tweakr_module_manager_is_valid_module_name (name))
        {
            TweakrModule *module;
            gchar     *path;

            path = g_build_filename (manager->module_path, name, NULL);
            module = tweakr_module_new (path);

            if (! g_type_module_use (G_TYPE_MODULE (module)))
            {
                g_printerr ("Failed to load module: %s\n", path);
                g_object_unref (module);
                g_free (path);
                continue;
            }

            g_free (path);

            g_type_module_unuse (G_TYPE_MODULE (module));

            manager->modules = g_list_prepend (manager->modules, module);
        }
    }

    g_dir_close (dir);

    return TRUE;
}

