/*
 * vim:ts=4:sw=4:et:cindent:cino=(0
 */ 

#include <config.h>
#include <glib/gi18n-lib.h>

#include <gtk/gtk.h>
#include <hildon/hildon-picker-button.h>
#include <hildon/hildon-touch-selector.h>
#include <hildon/hildon-note.h>
#include <hildon/hildon-check-button.h>

#include "libtweakr-section/tweakr-section.h"
#include "libtweakr-section/tweakr-module.h"


#define TWEAKR_TYPE_MCE_SECTION \
        (tweakr_mce_section_type)
#define TWEAKR_MCE_SECTION(obj) \
        (G_TYPE_CHECK_INSTANCE_CAST ((obj), \
        TWEAKR_TYPE_MCE_SECTION, \
        TweakrMceSection))
#define TWEAKR_MCE_SECTION_CLASS(k) \
        (G_TYPE_CHECK_CLASS_CAST((k), \
        TWEAKR_TYPE_MCE_SECTION, \
        TweakrMceSectionClass))
#define TWEAKR_IS_MCE_SECTION(obj) \
        (G_TYPE_CHECK_INSTANCE_TYPE ((obj), \
        TWEAKR_TYPE_MCE_SECTION))

#define DISABLE_LABEL _("Do nothing")
#define SHOW_MENU_LABEL _("Show menu")
#define POWER_OFF_LABEL _("Power off")
#define LOCK_LABEL _("Lock device")

enum {
    SHORT_POWER_KEY_DISABLED,
    SHORT_POWER_KEY_MENU,
    SHORT_POWER_KEY_OFF,
    SHORT_POWER_KEY_N
};

enum {
    LONG_POWER_KEY_DISABLED,
    LONG_POWER_KEY_MENU,
    LONG_POWER_KEY_OFF,
    LONG_POWER_KEY_N
};

enum {
    DOUBLE_POWER_KEY_DISABLED,
    DOUBLE_POWER_KEY_MENU,
    DOUBLE_POWER_KEY_OFF,
    DOUBLE_POWER_KEY_LOCK,
    DOUBLE_POWER_KEY_N
};


typedef struct _picker_t
{
    guint index;
    const gchar *value;
    const gchar *label;
} picker_t;

static picker_t spk [SHORT_POWER_KEY_N];
static picker_t lpk [LONG_POWER_KEY_N];
static picker_t dpk [DOUBLE_POWER_KEY_N];

typedef struct _TweakrMceSection TweakrMceSection;
typedef struct _TweakrMceSectionClass
               TweakrMceSectionClass;

struct _TweakrMceSection
{
    TweakrSection parent_instance;

    GKeyFile *ini;
    GtkWidget *short_power_key;
    GtkWidget *long_power_key;
    GtkWidget *double_power_key;
    GtkWidget *lens_cover;

    guint value_changed : 1;
};

struct _TweakrMceSectionClass
{
    TweakrSectionClass parent_class;
};


static GType tweakr_mce_section_get_type (GTypeModule *module);
static void tweakr_mce_section_class_init
    (TweakrMceSectionClass *class);
static void tweakr_mce_section_init
    (TweakrMceSection *section);
static void tweakr_mce_section_dispose (GObject *obj);

static gboolean _save (TweakrSection *section,
                       gboolean *requires_restart);

static GType tweakr_mce_section_type = 0;
static TweakrSectionClass *
    tweakr_mce_section_parent_class = NULL;


G_MODULE_EXPORT void
tweakr_module_load (TweakrModule *module)
{
    tweakr_mce_section_get_type (G_TYPE_MODULE (module));
}

G_MODULE_EXPORT void
tweakr_module_unload (TweakrModule *module)
{
}

static GType
tweakr_mce_section_get_type (GTypeModule *module)
{
    if (!tweakr_mce_section_type)
    {
        static const GTypeInfo section_info =
        {
            sizeof (TweakrMceSectionClass),
            (GBaseInitFunc) NULL,
            (GBaseFinalizeFunc) NULL,
            (GClassInitFunc) tweakr_mce_section_class_init,
            NULL,           /* class_finalize */
            NULL,           /* class_data     */
            sizeof (TweakrMceSection),
            0,              /* n_preallocs    */
            (GInstanceInitFunc) tweakr_mce_section_init
        };

        tweakr_mce_section_type =
            g_type_module_register_type (module, TWEAKR_TYPE_SECTION,
                                         "TweakrMceSection",
                                         &section_info, 0);
    }

    return tweakr_mce_section_type;
}

static void
tweakr_mce_section_class_init
    (TweakrMceSectionClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);
    TweakrSectionClass *section_class =
        TWEAKR_SECTION_CLASS (klass);

    tweakr_mce_section_parent_class =
        g_type_class_peek_parent (klass);

    section_class->name   = "_Mce";
    section_class->save = _save;

    object_class->dispose = tweakr_mce_section_dispose;
}

static void _value_changed (HildonPickerButton *b, gpointer user_data)
{
    TweakrMceSection *section = TWEAKR_MCE_SECTION (user_data);

    section->value_changed = TRUE;
}

static void _lens_cover_changed (HildonCheckButton *b, gpointer user_data)
{
    TweakrMceSection *section = TWEAKR_MCE_SECTION (user_data);

    section->value_changed = TRUE;
}

GtkWidget * _build_short_power_key (TweakrMceSection *section)
{
    gint i;
    GtkWidget *button, *selector;

    spk[0].index = SHORT_POWER_KEY_DISABLED;
    spk[0].value = "disabled";
    spk[0].label = DISABLE_LABEL;

    spk[1].index = SHORT_POWER_KEY_MENU;
    spk[1].value = "menu";
    spk[1].label = SHOW_MENU_LABEL;

    spk[2].index = SHORT_POWER_KEY_OFF;
    spk[2].value = "poweroff";
    spk[2].label = POWER_OFF_LABEL;

    selector = hildon_touch_selector_new_text ();
    for (i = 0; i < SHORT_POWER_KEY_N; i++)
    {
        hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector),
                                           spk[i].label);
    }

    button = hildon_picker_button_new (HILDON_SIZE_AUTO,
                                       HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    hildon_gtk_widget_set_theme_size (button, HILDON_SIZE_FINGER_HEIGHT);

    hildon_button_set_title (HILDON_BUTTON (button),
                             _("Power key short press"));
    gtk_button_set_alignment (GTK_BUTTON (button), 0.0f, 0.5f);
    hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
                                       HILDON_TOUCH_SELECTOR (selector));

    g_signal_connect (G_OBJECT (button), "value-changed",
                      G_CALLBACK (_value_changed), section);

    gtk_widget_show (button);
    return button;
}

GtkWidget * _build_long_power_key (TweakrMceSection *section)
{
    gint i;
    GtkWidget *button, *selector;
    
    lpk[0].index = LONG_POWER_KEY_DISABLED;
    lpk[0].value = "disabled";
    lpk[0].label = DISABLE_LABEL;

    lpk[1].index = LONG_POWER_KEY_MENU;
    lpk[1].value = "menu";
    lpk[1].label = SHOW_MENU_LABEL;

    lpk[2].index = LONG_POWER_KEY_OFF;
    lpk[2].value = "poweroff";
    lpk[2].label = POWER_OFF_LABEL;

    selector = hildon_touch_selector_new_text ();
    for (i = 0; i < LONG_POWER_KEY_N; i++)
    {
        hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector),
                                           lpk[i].label);
    }

    button = hildon_picker_button_new (HILDON_SIZE_AUTO,
                                       HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    hildon_gtk_widget_set_theme_size (button, HILDON_SIZE_FINGER_HEIGHT);

    hildon_button_set_title (HILDON_BUTTON (button),
                             _("Power key long press"));
    gtk_button_set_alignment (GTK_BUTTON (button), 0.0f, 0.5f);
    hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
                                       HILDON_TOUCH_SELECTOR (selector));

    g_signal_connect (G_OBJECT (button), "value-changed",
                      G_CALLBACK (_value_changed), section);

    gtk_widget_show (button);
    return button;
}

GtkWidget * _build_double_power_key (TweakrMceSection *section)
{
    gint i;
    GtkWidget *button, *selector;

    dpk[0].index = DOUBLE_POWER_KEY_DISABLED;
    dpk[0].value = "disabled";
    dpk[0].label = DISABLE_LABEL;

    dpk[1].index = DOUBLE_POWER_KEY_MENU;
    dpk[1].value = "menu";
    dpk[1].label = SHOW_MENU_LABEL;

    dpk[2].index = DOUBLE_POWER_KEY_OFF;
    dpk[2].value = "poweroff";
    dpk[2].label = POWER_OFF_LABEL;

    dpk[3].index = DOUBLE_POWER_KEY_LOCK;
    dpk[3].value = "tklock";
    dpk[3].label = LOCK_LABEL;

    selector = hildon_touch_selector_new_text ();
    for (i = 0; i < DOUBLE_POWER_KEY_N; i++)
    {
        hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector),
                                           dpk[i].label);
    }

    button = hildon_picker_button_new (HILDON_SIZE_AUTO,
                                       HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    hildon_gtk_widget_set_theme_size (button, HILDON_SIZE_FINGER_HEIGHT);

    hildon_button_set_title (HILDON_BUTTON (button),
                             _("Power key double press"));
    gtk_button_set_alignment (GTK_BUTTON (button), 0.0f, 0.5f);
    hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
                                       HILDON_TOUCH_SELECTOR (selector));

    g_signal_connect (G_OBJECT (button), "value-changed",
                      G_CALLBACK (_value_changed), section);

    gtk_widget_show (button);
    return button;
}

GtkWidget *_build_lens_cover (TweakrMceSection *section)
{
    GtkWidget *button;

    button = hildon_check_button_new (HILDON_SIZE_AUTO);
    hildon_gtk_widget_set_theme_size (button, HILDON_SIZE_FINGER_HEIGHT);
    gtk_button_set_label (GTK_BUTTON (button),
                          _("Opening camera lens cover unlocks device"));

    g_signal_connect (G_OBJECT (button), "toggled",
                      G_CALLBACK (_lens_cover_changed), section);

    gtk_widget_show (button);
    return button;
}

static void
tweakr_mce_section_init (TweakrMceSection *section)
{
    TweakrSection *iface;
    gchar *short_power_key_value;
    gchar *long_power_key_value;
    gchar *double_power_key_value;
    gint i;
    gboolean lens_cover_value;

    section->short_power_key = _build_short_power_key (section);
    section->long_power_key = _build_long_power_key (section);
    section->double_power_key = _build_double_power_key (section);
    section->lens_cover = _build_lens_cover (section);

    section->ini = g_key_file_new ();

    if (!g_key_file_load_from_file (section->ini, MCE,
                                    G_KEY_FILE_NONE, NULL))
    {
        g_warning ("%s: failed to load %s", G_STRFUNC, MCE);
        return;
    }

    short_power_key_value = g_key_file_get_string (section->ini, "PowerKey",
                                                   "PowerKeyShortAction",
                                                   NULL);
    long_power_key_value = g_key_file_get_string (section->ini, "PowerKey",
                                                  "PowerKeyLongAction",
                                                  NULL);
    double_power_key_value = g_key_file_get_string (section->ini, "PowerKey",
                                                    "PowerKeyDoubleAction",
                                                    NULL);
    lens_cover_value = g_key_file_get_boolean (section->ini, "TKLock",
                                                "LensCoverUnlock", NULL);

    for (i = 0; i < SHORT_POWER_KEY_N; i++)
    {
        if (g_strcmp0 (short_power_key_value, spk[i].value) == 0)
        {
            hildon_picker_button_set_active
                (HILDON_PICKER_BUTTON (section->short_power_key), i);
        }
    }

    for (i = 0; i < LONG_POWER_KEY_N; i++)
    {
        if (g_strcmp0 (long_power_key_value, lpk[i].value) == 0)
        {
            hildon_picker_button_set_active
                (HILDON_PICKER_BUTTON (section->long_power_key), i);
        }
    }

    for (i = 0; i < DOUBLE_POWER_KEY_N; i++)
    {
        if (g_strcmp0 (double_power_key_value, dpk[i].value) == 0)
        {
            hildon_picker_button_set_active
                (HILDON_PICKER_BUTTON (section->double_power_key), i);
        }
    }

    hildon_check_button_set_active
        (HILDON_CHECK_BUTTON (section->lens_cover), lens_cover_value);

    section->value_changed = FALSE;

    iface = TWEAKR_SECTION (section);
    iface->name = _("Hardware keys");
    iface->widget = gtk_vbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (iface->widget), section->short_power_key,
                        FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (iface->widget), section->long_power_key,
                        FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (iface->widget), section->double_power_key,
                        FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (iface->widget), section->lens_cover,
                        FALSE, FALSE, 0);
}

static void
tweakr_mce_section_dispose (GObject *obj)
{
    TweakrMceSection *section = TWEAKR_MCE_SECTION (obj);
    if (section->ini)
    {
        g_key_file_free (section->ini);
        section->ini = NULL;
    }

    G_OBJECT_CLASS (tweakr_mce_section_parent_class)->dispose
        (obj);
}

static gboolean _save (TweakrSection *section,
                       gboolean *requires_restart)
{
    gchar *argv[6];
    gint short_active, long_active, double_active;
    gint i;
    gboolean lens_cover_value;

    if (!TWEAKR_MCE_SECTION (section)->value_changed)
        return TRUE;

    short_active = hildon_picker_button_get_active
        (HILDON_PICKER_BUTTON (TWEAKR_MCE_SECTION
                               (section)->short_power_key));

    long_active = hildon_picker_button_get_active
        (HILDON_PICKER_BUTTON (TWEAKR_MCE_SECTION
                               (section)->long_power_key));

    double_active = hildon_picker_button_get_active
        (HILDON_PICKER_BUTTON (TWEAKR_MCE_SECTION
                               (section)->double_power_key));

    lens_cover_value = hildon_check_button_get_active
        (HILDON_CHECK_BUTTON (TWEAKR_MCE_SECTION (section)->lens_cover));

    if (short_active  == SHORT_POWER_KEY_DISABLED &&
        long_active   == LONG_POWER_KEY_DISABLED  &&
        double_active == DOUBLE_POWER_KEY_DISABLED)
    {
        GtkWidget *note;
        gint retcode;

        note = hildon_note_new_confirmation
            (NULL,
             _("Setting all Power Key options to \"Do nothing\" means "
               "that the only way to turn the device off is the removal of "
               "the battery. Do you want to continue?"));
        retcode = gtk_dialog_run (GTK_DIALOG (note));
        gtk_widget_destroy (note);

        if (retcode == GTK_RESPONSE_CANCEL)
            return FALSE;
    }

    *requires_restart = TRUE;

    argv[0] = g_strdup ("/usr/bin/tweakr-mce-save");
    argv[1] = g_strdup_printf ("%s", spk[short_active].value);
    argv[2] = g_strdup_printf ("%s", lpk[long_active].value);
    argv[3] = g_strdup_printf ("%s", dpk[double_active].value);
    argv[4] = g_strdup_printf ("%d", lens_cover_value);
    argv[5] = NULL;

    g_spawn_sync ("/home/user", argv, NULL,
                  G_SPAWN_STDOUT_TO_DEV_NULL | G_SPAWN_STDERR_TO_DEV_NULL,
                  NULL, NULL, NULL, NULL, NULL, NULL);

    for (i = 0; i < (sizeof (argv) / sizeof (gchar *)) - 1; i++)
        g_free (argv[i]);

    return TRUE;
}

