# -*- coding: utf-8 -*-


import re
import feedparser
import httplib2
import os
import sys
import urllib
import mimetypes
import codecs
import time
import logging

logger = logging.getLogger("servers.model")

from socket import timeout as SocketTimeout
from httplib2 import ServerNotFoundError

#from simplejson import dumps as stringify_json
#from simplejson import loads as parse_json

from PyQt4.QtCore import QObject, QAbstractListModel, pyqtSignal

from languages.model import LanguageModel
from components.model import ComponentModel
from projects.model import ProjectModel
from files.model import RemoteFileModel

from config import config
from errors import *

def get_content_type(filename):
    return mimetypes.guess_type(filename)[0] or 'application/octet-stream'

def encode_multipart_formdata(fields, files):
    """
    fields is a sequence of (name, value) elements for regular form fields.
    files is a sequence of (name, filename, value) elements for data to be uploaded as files
    Return (content_type, body) ready for httplib.HTTP instance
    """
    BOUNDARY = u'----------ThIs_Is_tHe_bouNdaRY_$'
    CRLF = u'\r\n'
    L = []
    for (key, value) in fields:
        L.append(u'--%s' % BOUNDARY)
        L.append(u'Content-Disposition: form-data; name="%s"' % key)
        L.append(u'')
        L.append(value)
    for (key, filename, value) in files:
        L.append(u'--' + BOUNDARY)
        L.append(u'Content-Disposition: form-data; name="%s"; filename="%s"' % (key, filename))
        L.append(u'Content-Type: %s' % get_content_type(filename))
        L.append(u'')
        L.append(value)
    L.append(u'--%s--' % BOUNDARY)
    L.append('')
    body = CRLF.join(L)
    content_type = 'multipart/form-data; boundary=%s' % BOUNDARY
    return content_type, body.encode("utf-8")


#CACHE_JSON = "%s/.config/transifex-client/cache" % os.getenv("HOME")

class ServerModel(QObject):
    """
    Class for handling connections to Tx server and executing API calls
    """

    RE_CSRF = r"""<input\s+type='hidden'\s+name='csrfmiddlewaretoken'\s+value='(?P<token>.+)'\s*/>"""
    RE_PROJECT_LINK = r"""%s/projects/p/(?P<project_slug>[-\w]+)/$"""
    RE_COMPONENT_LINK = r"""%s/projects/p/(?P<project_slug>[-\w]+)/c/(?P<component_slug>[-\w]+)/$"""
    RE_PROJECT = r"""<a\s+href="/projects/p/(?P<slug>[-\w]+)/">\s*(?P<title>.+)\s*</a>"""
    RE_SEARCH_HIGHLIGHT = """(<span\s+class="highlight">)|(</span>)"""
    RE_VERSION = """<p\s+class="version">v(?P<major>\d+).(?P<minor>\d+).(?P<revision>\d+)</p>"""

    # Regex for /projects/p/PROJECT_SLUG/c/COMPONENT_SLUG
    #RE_RAW_FILE = r"""<td class="left name" id="(?P<language_code>[-\w]+?)">.*?href=\"/projects/p/(?P<project_slug>[-\w]+?)/c/(?P<component_slug>[-\w]+?)/raw/(?P<filename>.*?)\"></a>"""
    RE_RAW_FILE = r"""<td class="left name" id="(?P<language_code>[-\w]+?)">.*?<form action=\"/projects/p/(?P<project_slug>[-\w]+?)/c/(?P<component_slug>[-\w]+?)/(?P<locked>.+?)/(?P<filename>.*?)\" method="POST" class="microform">"""

    # Regex for /languages
    RE_LANGUAGE = r"""<li class="i16 language"><a href="/languages/l/(?P<language_code>[-\w]+?)/">(?P<language_name>.+?) \([-\w]+?\)</a></li>"""

    # Regex for /home
    RE_LOCK = r"""&raquo;.+?\s+?\((?P<language_code>[-\w]+?)\)\s*?<span style="float:right">\s*?<form action="/projects/p/(?P<project_slug>[-\w]+)/c/(?P<component_slug>[-\w]+)/unlock/(?P<filename>.*)" method="POST" class="microform">"""

    # Regexes for /accounts/profile/
    RE_ACCOUNT_USERNAME = r"""<tr><th\s+id=\"id_user\">Username:</th><td>\s*(?P<username>.+)</td></tr>"""
    RE_ACCOUNT_EMAIL = r"""<tr>\s*<th\s+id=\"id_email\">E-mail:\s*</th>\s*<td>\s*<span class="">\s*(?P<email>.*)\s*</span>\s*</td>\s*</tr>"""
    RE_ACCOUNT_FIRSTNAME = r"""<tr>\s*<th\s+id=\"id_firstname\">First Name:\s*</th>\s*<td>\s*(?P<firstname>.*)\s*</td>\s*</tr>"""
    RE_ACCOUNT_SURNAME = r"""<tr>\s*<th\s+id=\"id_surname\">Surname:\s*</th>\s*<td>\s*(?P<surname>.*)\s*</td>\s*</tr>"""
    RE_ACCOUNT_FIELD_BLANK = r"""<span\s+class="\s*quiet\s+red\s*">\s*-\s*</span>"""

    # Signals
    locksInvalidated = pyqtSignal()

    def __init__(self, hostname, title=None, parent=None, name=None, 
        username=None, password=None):
        QObject.__init__(self, parent)
        if not hostname in config.data['servers']:
            config.data['servers'][hostname] = {'last_access':time.time()}
        self.hostname = hostname
        if title:
            self.title = title
        else:
            self.title = self.hostname
        self.conn = None
        self.online = True
        self.last_seen = None
        #if not os.path.exists(CACHE_JSON):
            #os.makedirs(CACHE_JSON)
        #if not os.path.isdir(CACHE_JSON):
            #raise Exception("%s is not directory!" % CACHE_JSON)
        self.conn = httplib2.Http(timeout=120)
        self.token = None
        self.cookie = None
        self.secure_paths = []
        self.version = None
        self.cache_user_info = None
        self.pullLanguages()

    def __unicode__(self):
        return u"%s (%s)" % (self.name, self.hostname)
        
    def request(self, path="/", method="GET", body=None, headers={}):
        if type(body) == type({}):
            body = urllib.urlencode(body)

        retries = 3
        last_error = None
        url = "http://%s%s" % (self.hostname, path)
        while retries > 0:
            if url.startswith("http://") and \
                path in self.secure_paths:
                url = "https://%s%s" % (self.hostname, path)


            headers["Host"] = self.hostname
            headers["Connection"] = "close"
#            headers["Keep-Alive"] = "120"
#            headers["User-Agent"] = "Mozilla/5.0 (X11; U; Linux x86_64; et; rv:1.9.2.6) Gecko/20100628 Ubuntu/10.04 (lucid) Firefox/3.6.6"
            headers["User-Agent"] = "Transifex Mobile (httplib2)"
            headers["Accept"] = "text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8"
#            headers["Accept-Encoding"] = "gzip,deflate"
            headers["Accept-Charset"] = "UTF-8;q=0.7,*;q=0.7"
            headers["Accept-Language"] = "en;q=0.7,en-us;q=0.3"
#            headers["Cache-Control"] = "max-age=0"
            if self.cookie:
                headers["Cookie"] = self.cookie

            if retries == 3:
                logger.debug("Requesting: %s %s (headers=%s;body=%s)" % (method, url,headers,body))
            else:
                logger.debug("Retrying: %s" % url)
            retries -= 1

            try:
                response, content = self.conn.request(
                    url, method, headers=headers, body=body)
                if response.status == 302:
                    logger.debug("Response: %i (%s)" % (response.status, response['location']))
                else:
                    logger.debug("Response: %i" % response.status)
            except SocketTimeout:
                last_error = TransifexTimeout(self.hostname)
                continue
            except ServerNotFoundError:
                raise TransifexServerNotFound(self.hostname)
#            except Exception, e:
#                raise TransifexUnknownError("%s: %s" % (type(e), e))

            if response.status == 500:
                raise TransifexInternalServerError(self.hostname)
              
            if response.status == 403 and \
                "Cross Site Request Forgery detected. Request aborted." in content:
                    raise TransifexInvalidToken

            if url.startswith("http://") and \
                response.status == 302 and \
                response['location'].startswith("https://"):
                self.secure_paths.append(path)
                continue

            if response.status == 302 and \
                response['location'].endswith("/accounts/login/?next=%s" % path):
                    raise TransifexNeedsLogin

            if 'set-cookie' in response:
                cookie = response['set-cookie']
                if self.cookie != cookie:
                    logger.debug("New cookie: %s" % cookie.split(";")[0])
                self.cookie = cookie
                self.token = None
            if not self.version:
                m = re.search(self.RE_VERSION, content)
                if m:
                    self.version =  [int(m.group(j)) for j in ["major","minor","revision"]]
                else:
                    self.version = [0, 9, 0]
            return response, content, url
        if last_error:
            raise last_error
        raise TransifexUnknownError

    def csrf_token(self, secure=False, die=False):
        if not self.token:
            response, content, final_url = self.request("/accounts/login/")
            m = re.search(self.RE_CSRF, content)
            if m:
                self.token = m.group("token")
                logger.debug("New CSRF token: %s" % self.token)
            elif die:
                raise TransifexInvalidToken("Couldn't get CSRF token")
            else:
                return None
        return self.token

    def login(self, username, password):
        def mask(s):
            if s:
                return ''.join(['*' for c in s])
            else:
                return None
        logger.debug("Logging in as %s with %s" % (username, mask(password)))
        body = {
            'username' : username,
            'password' : password}
        token = self.csrf_token()
        if token:
            body['csrfmiddlewaretoken'] = token
        headers = {'Content-type': 'application/x-www-form-urlencoded'}
        response, content, _ = self.request("/accounts/login/?next=/home/",
            method='POST', headers=headers, body=body)
        if response.status == 302:
            if response['location'].endswith("/home/"):
                return True
        if "Please enter a correct username and password." in content:
            raise TransifexLoginFailed(self.hostname)
        raise TransifexUnknownError(self.hostname)

    def get_user_info(self):
        response, content, _ = self.request("/accounts/profile/")
        d = {'firstname':'John','surname':'Doe','email':'invalid@invalid.com'}
        for regex in (
#            self.RE_ACCOUNT_USERNAME,
            self.RE_ACCOUNT_EMAIL,
            self.RE_ACCOUNT_FIRSTNAME,
            self.RE_ACCOUNT_SURNAME):
            m = re.search(regex, content)
            if m:
                for key, value in m.groupdict().iteritems():
                    if not re.match(self.RE_ACCOUNT_FIELD_BLANK, value):
#                        d[key] = value
                        d[key] = u"%s" % value.decode('utf-8')
            else:
                raise TransifexInvalidProfile(self.hostname)
        return d

    def pullProjects(self, search=None, parent=None):
        def link_to_slug(url):
            m = re.search(self.RE_PROJECT_LINK % self.hostname, url)
            if m:
                return m.group("project_slug")
        if search:
            resp, content, _ = self.request("/search/?q=%s" % search)
            m = re.findall(self.RE_PROJECT, content)
            if m:
                return [ProjectModel(
                    slug,
                    re.sub(self.RE_SEARCH_HIGHLIGHT, "", title),
                    client=self, parent=parent) for slug,title in m]
            else:
                return []
        else:
            doc = feedparser.parse("http://%s/projects/feed/" % self.hostname)
            return [ProjectModel(
                link_to_slug(entry.link),
                entry.title,
                client=self, parent=parent) for entry in doc.entries]

    def pullComponents(self, project_slug):
        def link_to_slug(url):
            m = re.search(self.RE_COMPONENT_LINK % self.hostname, url)
            if m:
                return m.group("component_slug")
        doc = feedparser.parse("http://%s/projects/p/%s/components/feed/" % \
            (self.hostname, project_slug))
        return [ComponentModel(
            link_to_slug(entry.link),
            entry.title,
            client=self) for entry in doc.entries]

    def pullFiles(self, project_slug, component_slug):
        resp, content, _ = self.request("/projects/p/%s/c/%s/" %
            (project_slug, component_slug))
        m = re.findall(self.RE_RAW_FILE, content, re.MULTILINE|re.DOTALL)
        if m:
            return [RemoteFileModel(project_slug, component_slug, file_path,
                client=self, lang_code=language_code, locked=(locked=="unlock")) for language_code,
                project_slug, component_slug, locked, file_path in m]
        else:
            return []

    def pullFile(self, project_slug, component_slug, file_path):
        url = "/projects/p/%s/c/%s/raw/%s" % (
            project_slug,
            component_slug,
            file_path)
        _, content, _ = self.request(url)
        return content

    def pushFile(self, project_slug, component_slug, remote_path, local_path):
        url = "/projects/p/%s/c/%s/submit/%s" % (
            project_slug,
            component_slug,
            remote_path)
        logger.debug("Pushing file to %s" % remote_path)
        fields = ('csrfmiddlewaretoken', self.csrf_token(die=True)),
        files = ('submitted_file', remote_path, codecs.open(local_path, encoding="utf-8").read()),
        content_type, body = encode_multipart_formdata(fields, files)
        headers = {'content-type': content_type,
            'content-length': str (len (body))}
        response, content, _ = self.request(url,
            method='POST', headers=headers, body=body)
        logger.debug("Pushing %s finished" % remote_path)

    def lockFile(self, project_slug, component_slug, remote_path):
        url = "/projects/p/%s/c/%s/lock/%s" % (
            project_slug,
            component_slug,
            remote_path)
        body = {}
        token = self.csrf_token()
        if token:
            body['csrfmiddlewaretoken'] = token
        headers = {'Content-type': 'application/x-www-form-urlencoded'}
        logger.debug("Locking file via URL %s" % url)
        response, _, _ = self.request(url, method='POST', headers=headers, body=body)
        if response.status != 302 or not response['location'].endswith(
            "/projects/p/%s/c/%s/" % (project_slug, component_slug)):
            raise TransifexInvalidResponse("Locking request returned invalid response")

    def unlockFile(self, project_slug, component_slug, remote_path):
        url = "/projects/p/%s/c/%s/unlock/%s" % (
            project_slug,
            component_slug,
            remote_path)
        body = {}
        token = self.csrf_token()
        if token:
            body['csrfmiddlewaretoken'] = token
        headers = {'Content-type': 'application/x-www-form-urlencoded'}
        logger.debug("Unocking file via URL %s" % url)
        response, _, _ = self.request(url, method='POST', headers=headers, body=body)
        if response.status != 302 or not response['location'].endswith(
            "/projects/p/%s/c/%s/" % (project_slug, component_slug)):
            raise TransifexInvalidResponse("Unlocking request returned invalid response")


    def pullLocks(self):
        url = "/home/"
        response, content, _ = self.request(url)
        m = re.findall(self.RE_LOCK, content)
        if m:
            return [RemoteFileModel(project_slug, component_slug, file_path,
                client=self, locked=True, lang_code=language_code) for 
                language_code, project_slug, component_slug,
                file_path in m]
        else:
            return []

    def pullLanguages(self):
        logger.debug("Pulling languages")
        url = "/languages/"
        response, content, _ = self.request(url)
        m = re.findall(self.RE_LANGUAGE, content)
        if m:
            return [LanguageModel(language_code, language_name) for
                language_code, language_name in m]
        else:
            raise TransifexInvalidResponse("Couln't fetch any languages")

class ServerListModel(QAbstractListModel):
    """
    Model for Transifex servers list
    """
    def __init__(self, parent = None):
        QAbstractListModel.__init__(self, parent)
        self.objs = [ServerModel("beta.trasifex.net", "Transifex.net (beta)"), 
            ServerModel("http://translate.moblin.org", "Translate Moblin")]

    def rowCount(self, parent):
        return len(self.objs)

    def data(self, index, role):
        if not index.isValid():
            return QVariant()
        if index.row() < 0 or index.row() >= len(self.objs):
            return QVariant()
        if role == Qt.DisplayRole:
            return self.objs[index.row()].name
        if role == Qt.UserRole:
            return self.objs[index.row()]
        return QVariant()
