# -*- coding: utf-8 -*-
import shutil
import os
from time import sleep
from config import config

from PyQt4.QtCore import QModelIndex, QString, Qt, pyqtSlot, pyqtSignal, \
    QSize, QTimer, QThread, QAbstractItemModel
from PyQt4.QtGui import QSortFilterProxyModel, QTableView, QAbstractItemView, \
    QToolBar, QIcon, QPlainTextEdit, QHBoxLayout, QFileDialog, QMessageBox, \
    QWidget, QProgressDialog

try:
    from PyQt4.QtMaemo5 import QMaemo5InformationBox
except ImportError:
    QMaemo5InformationBox = None
  
from common import QCustomWindow, QIconCache
from views.search_string import SearchStringDialog
from views.login import LoginDialog
from views.push_dialog import PushDialog

from gettextfile.model import GettextFile
from files.model import LocalFileModel, RemoteFileModel
from stringeditor.view import StringView

HELP_ABOUT = u"""
<p>Mobile translation application</p>
<p>Author: Lauri Võsandi &lt;lauri.vosandi@gmail.com&gt;</p>
"""

class StringsetProxyModel(QSortFilterProxyModel):
    """
    Proxy model for filtering strings
    """
    def __init__(self, model):
        QSortFilterProxyModel.__init__(self)
        self.setSourceModel(model)
        self.setFilterKeyColumn(0)
        self.filterTranslated = True
        self.filterFuzzy = True
        self.filterUntranslated = True
        self.setFilterRole(-2)

    def filterAcceptsRow(self, source_row, source_parent):
        if QSortFilterProxyModel.filterAcceptsRow(self, source_row, source_parent):
            entry = self.sourceModel().objs[source_row]
            if entry.obsolete:
                return False
            elif entry.translated():
                return self.filterTranslated
            elif entry.fuzzy():
                return self.filterFuzzy
            else:
                return self.filterUntranslated
        return False

    @pyqtSlot(bool)
    def setFilterTranslated(self, flag):
        self.filterTranslated = flag
        self.invalidate()

    @pyqtSlot(bool)
    def setFilterFuzzy(self, flag):
        self.filterFuzzy = flag
        self.invalidate()

    @pyqtSlot(bool)
    def setFilterUntranslated(self, flag):
        self.filterUntranslated = flag
        self.invalidate()

class FilePullThread(QThread):
    setProgress = pyqtSignal(str, int)
    setSourceModel = pyqtSignal(QAbstractItemModel)
    def __init__(self, fp, parent):
        QThread.__init__(self, parent=parent)
        self.file_model = fp

    def run(self):
        target = self.parent()
        uri = self.file_model.uri()
        self.setProgress.emit(
            self.tr("Downloading..."), 1)
        self.file_model.pull()
        if not self.file_model.locked:
            self.setProgress.emit(
                self.tr("Locking..."), 2)
            self.file_model.lock()
        self.setProgress.emit(
            self.tr("Parsing..."), 3)
        source_model = GettextFile(self.file_model)
        self.setProgress.emit(
            self.tr("Processing..."), 4)
        source_model.moveToThread(self.thread())
        source_model.setParent(self)
        self.setSourceModel.emit(source_model)

        self.setProgress.emit(self.tr("Done"), 5)
        target.setWindowTitle(self.file_model.long_title)

class FilePushThread(QThread):
    setProgress = pyqtSignal(str, int)
    setSourceModel = pyqtSignal(QAbstractItemModel)
    finish = pyqtSignal()
    def __init__(self, source_model, file_model, push=True, unlock=True,
        close=False, parent=None):
        QThread.__init__(self, parent=parent)
        self.source_model = source_model
        self.file_model = file_model
        self.unlock = unlock
        self.push = push

    def run(self):
        self.setProgress.emit(self.tr("Generating file..."), 1)
        d = config.data['servers'][self.file_model.client.hostname]['profile']
        self.source_model.save(last_translator = "%(firstname)s %(surname)s <%(email)s>" % d)
        if self.push:
            self.setProgress.emit(
                self.tr("Uploading file. This might take a while if VCS backend is slow..."), 2)
            self.file_model.push() # Upload file from cache to server
        if self.unlock:
            self.setProgress.emit(
                self.tr("Unlocking..."), 3)
            self.file_model.unlock()
        self.setProgress.emit(self.tr("Done..."), 4)
        self.finish.emit()

class FileView(QCustomWindow):
    @pyqtSlot(QModelIndex)
    def editString(self, index):
        self.modified = True
#        if isinstance(self.model, QSortFilterProxyModel):
#            index = self.model.mapToSource(index)
        print "Going to edit string... model:", self.source_model
        win = StringView(self.model, index,
            parent=self)
        win.show()

    @pyqtSlot()
    def fileSave(self, push=True, unlock=True, close=True):
        if isinstance(self.file_model, RemoteFileModel):
            self.pd = QProgressDialog(parent=self)
            self.pd.setWindowTitle(self.tr("Saving file"))
            self.pd.setMaximum(5)
            # This fixes centering issues on desktop
            self.pd.setMinimumSize(QSize(500,0))
            self.pd.forceShow()
            self.pd.setWindowModality(Qt.WindowModal)
            
            thread = FilePushThread(self.source_model, self.file_model,
                push=push, unlock=unlock, parent=self)
            thread.setProgress.connect(self.setProgress, type=Qt.BlockingQueuedConnection)
            if close:
                thread.finish.connect(self.forceClose, type=Qt.QueuedConnection)
            thread.start()
        else:
            self.source_model.save()
            self.modified = False
            if QMaemo5InformationBox:
                QMaemo5InformationBox.information(self, 
                    "<font color='black'>%s saved</font>" % self.file_model.title())

    @pyqtSlot()
    def fileSaveAs(self):
        filename = QFileDialog.getSaveFileName(
            self, caption="Save file as...",
            filter=QString("Gettext PO (*.po)"))

    @pyqtSlot()
    def fileOpen(self):
        filename = QFileDialog.getOpenFileName(
            self, caption="Open translation file...",
            filter=QString("Gettext PO/POT (*.po *.pot)"),
            options=QFileDialog.DontUseSheet|QFileDialog.DontUseSheet)
#        if filename:
#            self.fileLoad(LocalFileModel(filename))
        self.modified = False

    @pyqtSlot()
    def helpAbout(self):
        QMessageBox.about(
            self,
            self.tr("About mobile traslator"),
            HELP_ABOUT)

    @pyqtSlot(str, int)
    def setProgress(self, text, value):
        self.pd.setValue(value)
        self.pd.setLabelText(text)
        sleep(0.05)

    @pyqtSlot(QAbstractItemModel)
    def setSourceModel(self, source_model):
        self.source_model = source_model
        if self.model:
            self.model.setSourceModel(source_model)
        else:
            self.model = StringsetProxyModel(source_model)

        self.tableview.setModel(self.model)
        for i in [1,2]:
            self.tableview.resizeColumnToContents(i)
        self.tableview.resizeRowsToContents()

        # Connect signals to slots
        if isinstance(self.model, QSortFilterProxyModel):
            self.actionFilterTranslated.toggled.connect(
                self.model.setFilterTranslated)
            self.actionFilterFuzzy.toggled.connect(
                self.model.setFilterFuzzy)
            self.actionFilterUntranslated.toggled.connect(
                self.model.setFilterUntranslated)

    @pyqtSlot(bool)
    def editFind(self, checked):
        """
        Open search string dialog and use the string in lowercase to filter 
        entries by custom datarole (-2)s
        """
        if checked:
            q = SearchStringDialog.getSearchString()
            if q:
                self.model.setFilterFixedString(q.lower())
            else:
                self.actionFilterString.setChecked(False)
        else:
            self.model.setFilterFixedString("")

    @pyqtSlot()
    def forceClose(self):
        self.force_close = True
        self.close()

    def closeEvent(self, e):
        if self.force_close:
            e.accept()
            return
        r = PushDialog().exec_()
        if r == PushDialog.Cancel:
            e.ignore()
        elif r == PushDialog.Forget:
            e.accept()
        else:
            if r == PushDialog.Save:
                self.fileSave(push=False, unlock=False, close=True)
            elif r == PushDialog.Push:
                self.fileSave(push=True, unlock=False, close=True)
            elif r == PushDialog.PushAndUnlock:
                self.fileSave(push=True, unlock=True, close=True)
            e.ignore()

    def __init__(self, fp=None, parent=None):
        QCustomWindow.__init__(self, parent, busy=False, 
            skip_central_widget=True)
        self.setWindowTitle(self.tr("Transifex-mobile"))
        self.modified = False # Wether file is modified 
        self.force_close = False
        self.model = None # Proxied string list model
        self.source_model = None # String list model
        self.tableview = QTableView()
        self.tableview.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self.tableview.setProperty("FingerScrollable", True)
        self.tableview.clicked.connect(self.editString)
        self.tableview.setSelectionMode(QAbstractItemView.NoSelection)
        self.setCentralWidget(self.tableview)
        toolbar = QToolBar(
            self.tr("Stringset"),
            self)

        # Filter translated button
        self.actionFilterTranslated = toolbar.addAction(
            QIconCache.get("flag-green.svgz"),
            self.tr("Translated"))
        self.actionFilterTranslated.setCheckable(True)
        self.actionFilterTranslated.setChecked(True)

        # Filter fuzzy button
        self.actionFilterFuzzy = toolbar.addAction(
            QIconCache.get("flag-blue.svgz"),
            self.tr("Fuzzy"))
        self.actionFilterFuzzy.setCheckable(True)
        self.actionFilterFuzzy.setChecked(True)

        # Filter untranslated button
        self.actionFilterUntranslated = toolbar.addAction(
            QIconCache.get("flag-red.svgz"),
            self.tr("Untranslated"))
        self.actionFilterUntranslated.setCheckable(True)
        self.actionFilterUntranslated.setChecked(True)

        # Search string button
        self.actionFilterString = toolbar.addAction(
            QIconCache.get("edit-find.svgz"),
            self.tr("Search string"))
        self.addToolBar(toolbar)
        self.actionFilterString.setCheckable(True)
        self.actionFilterString.triggered.connect(self.editFind)

        # File menu
        menuFile = self.menuBar().addMenu(self.tr("&File"))


        # Allow some actions only once in "local" editor mode
        if not fp or isinstance(fp, LocalFileModel):
            # Open button
            self.actionFileOpen = toolbar.addAction(
            QIconCache.get("document-open.svgz"),
            self.tr("Open..."),
            self.fileOpen)

            # Save button
            self.actionFileSave = toolbar.addAction(
            QIconCache.get("document-save-all.svgz"),
            self.tr("Save"),
            self.fileSave)

            # Save as button
            self.actionFileSaveAs = toolbar.addAction(
            QIconCache.get("document-save-as.svgz"),
            self.tr("Save as..."),
            self.fileSaveAs)
            menuFile.addAction(self.actionFileOpen)
            menuFile.addAction(self.actionFileSave)
            menuFile.addAction(self.actionFileSaveAs)
	    menuFile.addSeparator()

        menuFile.addAction(QIcon(), self.tr("E&xit"), self.close)

        # Edit menu
        menuEdit = self.menuBar().addMenu(self.tr("&Edit"))
        menuEdit.addAction(self.actionFilterString)

        # Help menu
        menuHelp = self.menuBar().addMenu(self.tr("&Help"))
        menuHelp.addAction(QIcon(), self.tr("&About"), self.helpAbout)

        # If we have file, load it
        if fp:
            self.pd = QProgressDialog(parent=self)
            self.pd.setWindowTitle(self.tr("Loading file"))
            self.pd.setMaximum(5)
            # This fixes centering issues on desktop
            self.pd.setMinimumSize(QSize(500,0))
            self.pd.forceShow()
            self.pd.setWindowModality(Qt.WindowModal)
            self.pd.setValue(0)

            thread = FilePullThread(fp, parent=self)
            self.file_model = fp
            thread.setProgress.connect(self.setProgress, type=Qt.BlockingQueuedConnection)
            thread.setSourceModel.connect(self.setSourceModel, type=Qt.BlockingQueuedConnection)
            thread.start()