/* Tracker Extract - extracts embedded metadata from files
 * Copyright (C) 2006, Mr Jamie McCracken (jamiemcc@gnome.org)
 * Flac support added by Michael Cronenworth (mike@cchtml.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA  02110-1301, USA.
 */

#include "config.h"

#include <stdio.h>
#include <string.h>
#include <fcntl.h>

#include <glib.h>

#include <FLAC/metadata.h>

#include "tracker-ripped-functions.h"
#include "tracker-extract-module-api.h"

static void extract_flac (const char *filename,
                          GHashTable *metadata);

static struct {
	gchar *name;
	gchar *meta_name;
} tags[] = {
	 { "TITLE", "Audio:Title" },
	 { "ARTIST", "Audio:Artist" },
	 { "ALBUM", "Audio:Album" },
	 { "ALBUM ARTIST", "Audio:AlbumArtist" },
	 { "TRACKNUMBER", "Audio:TrackNo" },
	 { "DISCNUMBER", "Audio:DiscNo" },
	 { "DATE", "Audio:ReleaseDate" },
	 { "Comment", "Audio:Comment" },
	 { "GENRE", "Audio:Genre" },
	 { "PUBLISHER", "File:Publisher" },
	 { "REPLAYGAIN_TRACK_GAIN", "Audio:TrackGain" },
	 { "REPLAYGAIN_TRACK_PEAK", "Audio:TrackPeakGain" },
	 { "REPLAYGAIN_ALBUM_GAIN", "Audio:AlbumGain" },
	 { "REPLAYGAIN_ALBUM_PEAK", "Audio:AlbumPeakGain" },
	 { "PUBLISHER", "File:Publisher" },
	 { NULL, NULL },
};

static TrackerExtractData extract_data[] = {
	{ "audio/x-flac", extract_flac },
	{ "application/flac", extract_flac },
	{ NULL, NULL }
};


static void
extract_flac (const char *filename,
              GHashTable *metadata)
{
    FLAC__Metadata_Chain *chain = FLAC__metadata_chain_new();
    FLAC__Metadata_Iterator *iterator = FLAC__metadata_iterator_new();
    FLAC__StreamMetadata *block;
    FLAC__uint64    duration;
	gint	        i, offset;
	unsigned int    bitrate;
	gchar          *str_bitrate;
	gchar          *str, *time_str, *utf_str;

    g_hash_table_insert (metadata, g_strdup ("Audio:Codec"), g_strdup ("flac"));

    if ( !FLAC__metadata_chain_read(chain, filename) )
        return;
    FLAC__metadata_iterator_init(iterator, chain);

    do {
        block = FLAC__metadata_iterator_get_block(iterator);

        switch ( block->type ) {
            case FLAC__METADATA_TYPE_STREAMINFO:
            // read bitstream information

            // calculate average bitrate
            bitrate = ( block->data.stream_info.max_framesize + block->data.stream_info.min_framesize ) / 2 * 8;
            str_bitrate = g_strdup_printf ("%d", bitrate);
            g_hash_table_insert (metadata, g_strdup ("Audio:Bitrate"),
                                 str_bitrate);
            // calculate duration
            duration = block->data.stream_info.total_samples / block->data.stream_info.sample_rate;
            time_str = g_strdup_printf ("%lld", duration);
            g_hash_table_insert (metadata, g_strdup ("Audio:Duration"), time_str);
            g_hash_table_insert (metadata, g_strdup ("Audio:Channels"),
                                 g_strdup_printf ("%d", block->data.stream_info.channels));
            g_hash_table_insert (metadata, g_strdup ("Audio:Samplerate"),
                                 g_strdup_printf ("%ld", block->data.stream_info.sample_rate));
            break;

            case FLAC__METADATA_TYPE_VORBIS_COMMENT:
            // read vorbis comments for tags
            for(i = 0; tags[i].name != NULL; ++i) {
                offset = FLAC__metadata_object_vorbiscomment_find_entry_from( block, 0, tags[i].name );
                if ( offset > -1 ) {
                    str = strchr( block->data.vorbis_comment.comments[offset].entry, '=' );
                    utf_str = g_locale_to_utf8( str + 1, -1, NULL, NULL, NULL );
                    g_hash_table_insert (metadata, g_strdup (tags[i].meta_name), utf_str);
                }
            }
            break;

            default:
            break;
        }
    }
    while ( FLAC__metadata_iterator_next(iterator) );

    // free memory
    FLAC__metadata_iterator_delete(iterator);
    FLAC__metadata_chain_delete(chain);
}

TrackerExtractData *
tracker_get_extract_data (void)
{
	return extract_data;
}
