#!/usr/bin/env python

LEVEL_PATH='/opt/togglegame/levels/'
VERSION='0.6'
VERSION_DATE='2013-04-23'

import pygtk
pygtk.require('2.0')
import gtk
import operator
import time
import string
#import code ## only for debug:  code.InteractiveConsole(locals()).interact()
try:
    import hildon
except ImportError:
    pass

class Colors:
    def __init__(self, colormap):
        self.bg=colormap.alloc_color(gtk.gdk.color_parse('#FFF'))
        self.line=colormap.alloc_color(gtk.gdk.color_parse('#00F'))
        self.toggle=[colormap.alloc_color(gtk.gdk.color_parse('#F00')),
                     colormap.alloc_color(gtk.gdk.color_parse('#0F0'))]


class ToggleBase:
    def __init__(self):
        self.state=0
        self.neighbours=[]
    def set_state(self, st):
        self.state=st
    def get_state(self):
        return self.state
    def set_neighbours(self, neighbours):
        self.neighbours=neighbours
    def add_neighbour(self, neighbour):
        self.neighbours.append(neighbour)
    def get_neighbours(self):
        return self.neighbours
    def toggle(self):
        self.toggle_one()
        for t in self.neighbours:
            t.toggle_one()
    def toggle_one(self):
        self.state=1-self.state

class ToggleRect(ToggleBase):
    def __init__(self,x,y,w,h):
        ToggleBase.__init__(self)
        self.x=x; self.y=y; self.w=w; self.h=h
    def draw(self,widget):
        drawable=widget.window
        mx,my=drawable.get_size()
        rx=int(self.x*mx); ry=int(self.y*my); rw=int(self.w*mx); rh=int(self.h*my)
        gc=drawable.new_gc(foreground=widget.my_colors.toggle[self.get_state()])
        drawable.draw_rectangle(gc, True, rx, ry, rw, rh)
        gc.set_foreground(widget.my_colors.line)
        drawable.draw_rectangle(gc, False, rx, ry, rw, rh)
    def is_inside(self,x,y):
        return (self.x <= x) and (x <= self.x+self.w) and (self.y <= y) and (y <= self.y+self.h)

class ToggleEllipsis(ToggleBase):
    def __init__(self,x,y,w,h):
        ToggleBase.__init__(self)
        self.x=x; self.y=y; self.w=w; self.h=h
    def draw(self,widget):
        drawable=widget.window
        mx,my=drawable.get_size()
        rx=int(self.x*mx); ry=int(self.y*my); rw=int(self.w*mx); rh=int(self.h*my)
        gc=drawable.new_gc(foreground=widget.my_colors.toggle[self.get_state()], line_width=2)
        drawable.draw_arc(gc, True, rx, ry, rw, rh, 0, 64*360)
        gc.set_foreground(widget.my_colors.line)
        drawable.draw_arc(gc, False, rx, ry, rw, rh, 0, 64*360)
    def is_inside(self,x,y):
        # todo: test actual ellipse
        return (self.x <= x) and (x <= self.x+self.w) and (self.y <= y) and (y <= self.y+self.h)

class ConnectPolygon:
    def __init__(self, points):
        self.points=points
    def draw(self,widget):
        drawable=widget.window
        mx,my=drawable.get_size()
        rpoints=[(int(x*mx),int(y*my)) for x,y in self.points]
        gc=drawable.new_gc(foreground=widget.my_colors.line, line_width=2)
        for i in range(len(rpoints)-1):
            drawable.draw_line(gc, rpoints[i][0], rpoints[i][1], rpoints[i+1][0], rpoints[i+1][1])

class ConnectLine(ConnectPolygon):
    def __init__(self,x1,y1,x2,y2):
        ConnectPolygon.__init__(self,[(x1,y1),(x2,y2)])

class ConnectDot:
    def __init__(self,x,y):
        self.x=x; self.y=y
    def draw(self,widget):
        drawable=widget.window
        mx,my=drawable.get_size()
        gc=drawable.new_gc(foreground=widget.my_colors.line, line_width=5, cap_style=gtk.gdk.CAP_ROUND)
        rx=int(mx*self.x); ry=int(my*self.y)
        drawable.draw_line(gc, rx, ry, rx, ry)


class GameModel:
    def __init__(self):
        self.levelnr=1
        self.loadlevel()
    def loadlevel(self):
        levelfile=LEVEL_PATH+'level%.2d.fig' % self.levelnr
        self.toggles=[]
        self.drawables=[]
        self.parse(levelfile)
    def reset(self):
        for t in self.toggles:
            t.set_state(0)
    def press(self, x, y):
        obj=self.find_toggle_at(x,y)
        if obj!=None:
            obj.toggle()
    def next_level(self):
        self.levelnr+=1
        self.loadlevel()
    def get_level(self):
        return self.levelnr
    def compute_password(self, n):
        return '%.8x'%((n<<24)+((((n+1)*(n+2)*(n+5))&0xff)<<16)+(((n^0xa5)**3^0x2712)&0xffff))
    def get_level_password(self):
        return self.compute_password(self.levelnr)
    def try_level(self, password):
        try:
            n=int(password,16)>>24
            if n==0: # allow direct level number when current level nr is higher
                n=int(password)
                #if n > 0: #use this to CHEAT
                if n > 0 and n < self.levelnr:
                    self.levelnr=n
                    self.loadlevel()
                    return True
                else:
                    return False
            if self.compute_password(n)==password:
                self.levelnr=n
                self.loadlevel()
            return True
        except:
            pass
        return False

    def draw(self, widget):
        for d in self.drawables:
            d.draw(widget)
        for t in self.toggles:
            t.draw(widget)
    def is_level_done(self):
        states=[t.get_state() for t in self.toggles]
        return 0==states.count(0) and len(states)>0
    def find_toggle_at(self,x,y):
        for t in self.toggles:
            if t.is_inside(x,y):
                return t
        return None
    def parse(self, levelfile):
        # very incomplete parsing of xfig files, see http://www.xfig.org/userman/fig-format.html
        f=file(levelfile, 'rU')
        lines=f.readlines()
        f.close()
        lineno=9
        circles=[]; boxes=[]; bluelinks=[]; greenlinks=[]; blacklines=[]; blackpoints=[]
        border=None
        while lineno<len(lines):
            line=lines[lineno]
            lineno+=1
            linelist=[float(s) for s in line.split(' ')]
            if line.startswith('1 3'):
                # Ellipsis (toggle)
                assert len(linelist)==20
                cx,cy,rx,ry=linelist[12:16]
                circles.append((cx-rx,cy-ry,cx+rx,cy+ry)) # (x1,y1,x2,y2)
            elif line.startswith('2 2 0 1 0 4'):
                # Box (toggle)
                pointlist=[float(s) for s in lines[lineno].lstrip().split(' ')]; lineno+=1
                assert len(pointlist)==10
                boxes.append((pointlist[0],pointlist[1],pointlist[4],pointlist[5])) # (x1,y1,x2,y2)
            elif line.startswith('2 1 0 1 1'):
                # Blue link drawed connector
                pointlist=[float(s) for s in lines[lineno].lstrip().split(' ')]; lineno+=1
                assert len(pointlist)==4
                bluelinks.append(tuple(pointlist)) # (x1,y1,x2,y2)
            elif line.startswith('2 1 0 1 2'):
                # Green link connector only
                pointlist=[float(s) for s in lines[lineno].lstrip().split(' ')]; lineno+=1
                assert len(pointlist)==4
                greenlinks.append(tuple(pointlist)) # (x1,y1,x2,y2)
            elif line.startswith('2 1 0 1 0'):
                # Black polyline drawed only
                pointlist=[float(s) for s in lines[lineno].lstrip().split(' ')]; lineno+=1
                blacklines.append(pointlist) # [x1,y1, ... ,xn,yn]
            elif line.startswith('2 1 0 5 0'):
                # Black point drawed only
                pointlist=[float(s) for s in lines[lineno].lstrip().split(' ')]; lineno+=1
                assert len(pointlist)==2
                blackpoints.append(tuple(pointlist)) # (x1,y1)
            elif line.startswith('2 2 0 1 0 7'):
                # Border box
                pointlist=[float(s) for s in lines[lineno].lstrip().split(' ')]; lineno+=1
                assert len(pointlist)==10
                border=(pointlist[0],pointlist[1],pointlist[4],pointlist[5]) # (x1,y1,x2,y2)
        assert border!=None
        x0=border[0]; y0=border[1]
        rw=float(border[2]-x0)
        rh=float(border[3]-y0)
        tx=lambda x: (x-x0)/rw
        ty=lambda y: (y-y0)/rh
        connections=[]
        for x1,y1,x2,y2 in circles:
            self.toggles.append(ToggleEllipsis(tx(x1),ty(y1),tx(x2)-tx(x1),ty(y2)-ty(y1)))
        for x1,y1,x2,y2 in boxes:
            self.toggles.append(ToggleRect(tx(x1),ty(y1),tx(x2)-tx(x1),ty(y2)-ty(y1)))
        for x1,y1,x2,y2 in bluelinks:
            self.drawables.append(ConnectLine(tx(x1),ty(y1),tx(x2),ty(y2)))
            connections.append((tx(x1),ty(y1),tx(x2),ty(y2)))
        for x1,y1,x2,y2 in greenlinks:
            connections.append((tx(x1),ty(y1),tx(x2),ty(y2)))
        for pointlist in blacklines:
            self.drawables.append(ConnectPolygon(zip(map(tx,pointlist[::2]),map(ty,pointlist[1::2]))))
        for x,y in blackpoints:
            self.drawables.append(ConnectDot(tx(x),ty(y)))
        for x1,y1,x2,y2 in connections:
            obj1=self.find_toggle_at(x1,y1)
            assert obj1!=None, 'Connection is not terminated in toggle object'
            obj2=self.find_toggle_at(x2,y2)
            assert obj2!=None, 'Connection is not terminated in toggle object'
            obj1.add_neighbour(obj2)
            obj2.add_neighbour(obj1)


class ToggleGame:
    def __init__(self):
        try:
            self.window = hildon.Window()
        except NameError:
            self.window = gtk.Window(gtk.WINDOW_TOPLEVEL)
        self.window.set_title("Toggle Game")
        self.window.connect("destroy", lambda w: gtk.main_quit())
        self.area = gtk.DrawingArea()
        self.area.set_size_request(480, 396)
        self.area.connect("expose-event", self.area_expose_cb)
        self.area.set_events(gtk.gdk.BUTTON_PRESS_MASK)
        self.area.connect("button_press_event", self.area_press_event)
        self.button_reset = gtk.Button(label="Reset")
        self.button_load = gtk.Button(label="Load")
        self.button_about = gtk.Button(label="About")
        self.model = GameModel()
        self.button_reset.connect("button_press_event", self.action_reset)
        self.button_reset.show()
        self.button_load.connect("button_press_event", self.action_load)
        self.button_load.show()
        self.button_about.connect("button_press_event", self.action_about)
        self.button_about.show()
        self.label_level=gtk.Label('')
        self.label_level.show()
        self.area.my_colors=Colors(self.area.get_colormap())
        self.area.show()
        hbox=gtk.HBox()
        self.window.add(hbox)
        hbox.pack_start(self.area)
        buttonbox=gtk.VBox()
        buttonbox.pack_start(self.label_level, False, False,5)
        buttonbox.pack_start(self.button_reset, False, False,5)
        buttonbox.pack_start(self.button_load, False, False,5)
        buttonbox.pack_start(self.button_about, False, False,5)
        buttonbox.show()
        hbox.pack_end(buttonbox, False, False, 5)
        hbox.show()
        self.window.show()
        self.redraw()
    def redraw(self):
        self.label_level.set_text("Level %d\nPassword:\n%s"% (self.model.get_level(),self.model.get_level_password()))
        self.area.queue_draw()
    def action_about(self,widget,event):
        dialog = gtk.Dialog(title='About Toggle Game', parent=self.window, flags=gtk.DIALOG_MODAL)
                            #buttons=(gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        dialog.set_size_request(720,350)
        text1 = """
Toggle Game - a logical game with interconnected switches
Copyright (C) 2007 Jakub Travnik, <jtra@seznam.cz>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2 of
the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public
License along with this program; if not, write to the Free
Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
MA 02110-1301 USA

Full text of the license may also be available at
/usr/share/togglegame/LICENSE (plain ASCII text file)
"""
        text2 = """
Goal of each level is to change all toggles from red to green.
This can be difficult because switch changes not only its color
but also a color of connected neighbors when pressed.

If you want to continue from the same level next time, remember
the level password. You can enter this password in Load dialog.
You can also get to lower levels (than current) by entering
level number there directly.

Reset button changes all toggles to back to red.
"""
        text3 = """
Version: 0.6
Aapo Rantalainen slightly modified game for Nokia N900/Maemo5.
----------------
Version: 0.5
Jakub Travnik:
I saw idea of this game somewhere in mid 90's in a DOS game
called Marionet or something like that. I liked the idea and
re-implemented the game for MS Windows in Delphi in 1998 and
created a set of 12 levels with increasing complexity. I have
solution for all levels and I have proved by program that in
some levels there is exactly one solution of switch state. That
version was not widely available, except for few friends in
closed source form.

This Python implementation was written in January 2007 specially
for Nokia 770 Internet tablet device. It is available with full
source code under GNU GPL v2 (see License tab). Levels are
identical to those in 1998 version, however level files has been
change to different format which allow rapid level creation :-)

You can make your own levels with xfig vector editor. Each level
file is ordinary .fig file with some conventions and many
limitations. Only rectangle and circle shapes are supported for
toggles and several kind of lines are distinguished by their
color. Load existing levels to learn how to create new ones.
"""
        text4 = "Version "+VERSION+" from "+VERSION_DATE
        notebook = gtk.Notebook()
        notebook.show()
        def add_scrolled_text_view(text, labeltext):
            scroll = gtk.ScrolledWindow()
            scroll.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
            tv = gtk.TextView()
            tv.set_editable(False)
            tb = tv.get_buffer()
            scroll.add(tv)
            scroll.show()
            tv.show()
            tb.set_text(text)
            label = gtk.Label(labeltext)
            label.show()
            notebook.append_page(scroll, label)
        add_scrolled_text_view(text1, 'License')
        add_scrolled_text_view(text2, 'Instructions')
        add_scrolled_text_view(text3, 'History')
        add_scrolled_text_view(text4, 'Version')
        dialog.vbox.pack_start(notebook, True, True, 0)
        dialog.run()
        dialog.destroy()
    def action_load(self,widget,event):
        dialog = gtk.Dialog(title='Load level', parent=self.window, flags=gtk.DIALOG_MODAL,
                            buttons=(gtk.STOCK_CANCEL, gtk.RESPONSE_REJECT, gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        label = gtk.Label("Enter level password or\ndirect level number for previous levels.")
        dialog.vbox.pack_start(label, True, True, 0)
        label.show()
        entry = gtk.Entry(max=8)
        dialog.vbox.pack_start(entry, True, True, 0)
        entry.show()
        if dialog.run()==gtk.RESPONSE_ACCEPT:
            if self.model.try_level(entry.get_text()):
                self.redraw()
            else:
                dialog2 = gtk.MessageDialog(self.window,
                                            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                                            gtk.MESSAGE_WARNING, gtk.BUTTONS_OK,
                                            "Bad password or level number.")
                dialog2.run()
                dialog2.destroy()
        dialog.destroy()
    def action_reset(self,widget,event):
        dialog = gtk.MessageDialog(self.window,
                                   gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                                   gtk.MESSAGE_INFO, gtk.BUTTONS_OK_CANCEL,
                                   "Do you want to reset this level?")
        if dialog.run()==gtk.RESPONSE_OK:
            self.model.reset()
            self.redraw()
        dialog.destroy()
    def area_press_event(self, widget, event):
        size=widget.window.get_size()
        self.model.press(event.x/float(size[0]), event.y/float(size[1]))
        self.redraw()
        if self.model.is_level_done():
            dialog = gtk.MessageDialog(self.window,
                                       gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                                       gtk.MESSAGE_INFO, gtk.BUTTONS_OK,
                                       "Level done.")
            dialog.run()
            dialog.destroy()
            try:
                self.model.next_level()
            except IOError:
                dialog = gtk.MessageDialog(self.window,
                                           gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                                           gtk.MESSAGE_INFO, gtk.BUTTONS_OK,
                                           "Congratulations! You have completed all levels.")
                dialog.run()
                dialog.destroy()
            self.redraw()

    def area_expose_cb(self, area, event):
        self.model.draw(self.area)

def main():
    gtk.main()
    return 0

if __name__ == "__main__":
    ToggleGame()
    main()
