#include "todolistwidget.h"
#include "ui_todolistwidget.h"

TodoListWidget::TodoListWidget(QWidget *parent) :
    QWidget(parent),
    ui(new Ui::TodoListWidget)
{
    ui->setupUi(this);

    organizerDBChanged = false;

    defaultManager = new QOrganizerManager;
    connect(defaultManager, SIGNAL(dataChanged ()), this, SLOT(dataChanged()));

    readSettings();

    populateList();

    fixSize();

    this->setAttribute(Qt::WA_TranslucentBackground);
}

TodoListWidget::~TodoListWidget()
{
    delete ui;
}

void TodoListWidget::dataChanged()
{
    organizerDBChanged = true;
}

void TodoListWidget::settingsChanged()
{
    readSettings();
    clearList();
    populateList();
    fixSize();
    repaint();
}

void TodoListWidget::settingsRequested()
{
    SettingsDialog dialog;
    connect(&dialog, SIGNAL(saveButtonClicked()), this, SLOT(settingsChanged()));
    dialog.exec();
}

void TodoListWidget::populateList()
{
    QOrganizerItemDetailFilter itemDetailFilter;
    itemDetailFilter.setDetailDefinitionName("Type", "Type");
    itemDetailFilter.setValue("Todo");

    QList<QOrganizerItem> entries;
    entries = defaultManager->items(itemDetailFilter);
    qDebug()<<"Number of total Todo items: "<<entries.size();

    int numberOfTodosToShow;

    if ( entries.size() > numberOfRows )
    {
        numberOfTodosToShow = numberOfRows;
    }
    else
    {
        numberOfTodosToShow = entries.size();
    }

    qDebug() << "Number of todos to show: " << numberOfTodosToShow;

    for (int i = 0; i < numberOfTodosToShow; i++)
    {
        QOrganizerTodo todoItem;
        QOrganizerItem orgItem;
        orgItem = entries[i];
        todoItem = orgItem;

        QOrganizerCollectionId collectionId = todoItem.collectionId();
        QOrganizerCollection collection = defaultManager->collection(collectionId);

        TodoWidget* todoWidget = new TodoWidget;
        todoWidget->setManagerPointer(defaultManager);
        todoWidget->setTitle( entries.at(i).displayLabel() );

        QFont font("Nokia Sans", fontSize);
        todoWidget->setFont(font);

        todoWidget->setCollectionColor(collection.metaData("Color").toString());


        qDebug()<<"Status is: "<<todoItem.status();
        qDebug() << "Color is: " << collection.metaData("Color").toString();

        if ( todoItem.status() == QOrganizerTodoProgress::StatusInProgress )
        {
            todoWidget->setStatus(TodoWidget::Done);
        }
        else if ( todoItem.status() == QOrganizerTodoProgress::StatusNotStarted )
        {
            todoWidget->setStatus(TodoWidget::NotDone);
        }

        todoWidget->setIconSize(rowHeight);
        todoWidget->drawIcon(rowHeight-3);


        todoWidget->setFixedSize(QSize(rowWidth, rowHeight));
        todoWidget->setId( entries.at(i).id().toString() );


        ui->verticalLayout->addWidget(todoWidget);
        todoWidgetsVector.append(todoWidget);

        connect(todoWidget, SIGNAL(todoDeleted()), this, SLOT(todoDeleted()));

        QPixmap whiteLinePixmap = QPixmap(rowWidth, 1);
        whiteLinePixmap.fill();

        QLabel *lineLabel = new QLabel;
        lineLabel->setPixmap(whiteLinePixmap);
        ui->verticalLayout->addWidget(lineLabel);
        todoWidgetsVector.append(lineLabel);




    }
}

void TodoListWidget::readSettings()
{
    QSettings settings("todolistwidget", "settings", this);
    numberOfRows = settings.value("numberOfRows", 7).toInt();
    fontSize = settings.value("fontSize", 15).toInt();
    rowHeight = settings.value("rowHeight", 40).toInt();
    rowWidth = settings.value("rowWidth", 400).toInt();
    autoResize = settings.value("autoResize", false).toBool();
    alphaValue = settings.value("alphaValue", 60).toInt();
    widgetCornersRoundness = settings.value("widgetCornersRoundness", 8).toInt();


}

void TodoListWidget::todoDeleted()
{
    clearList();
    populateList();

    fixSize();
}

void TodoListWidget::clearList()
{
    for ( int i = 0; i < todoWidgetsVector.size(); i++)
    {
        delete todoWidgetsVector.at(i);
    }

    todoWidgetsVector.clear();


}

void TodoListWidget::on_newTodoButton_clicked()
{
    AddTodoDialog dialog(defaultManager);
    connect(&dialog, SIGNAL(todoAdded()), this, SLOT(todoAdded()));
    dialog.exec();


    /*QOrganizerTodo todoItem;
    bool ok;
    QString string = QInputDialog::getText(0, "New Todo", "Todo:", QLineEdit::Normal, "", &ok , 0);
    if ( ok == false ) return;
    todoItem.setDisplayLabel(string);
    todoItem.setDueDateTime(QDateTime::currentDateTime());
    defaultManager->saveItem(&todoItem);

    clearList();
    populateList();*/
}

void TodoListWidget::todoAdded()
{
    clearList();
    populateList();
    fixSize();
}

void TodoListWidget::fixSize()
{
    ui->newTodoButton->setFixedSize(rowWidth, rowHeight);
    ui->newTodoButton->setIconSize(QSize(rowWidth-3, rowHeight-3));
    ui->newTodoButton->setIcon(QIcon(":/icons/addButton.png"));

    qDebug() << "Fixing size, vector size is: " << todoWidgetsVector.size();

    for ( int i = 0; i < todoWidgetsVector.size(); i++)
    {
        if ( i%2 == 1 ) continue;

        todoWidgetsVector.at(i)->setFixedSize(QSize(rowWidth, rowHeight));
    }

    if ( autoResize )
    {
        adjustSize();
    }
    else
    {
        this->setFixedSize( rowWidth, (numberOfRows+1) * (rowHeight+1) );
    }
}
