#include "playlistmodel.h"
#include "music.h"
#include "notifications.h"

PlaylistModel::PlaylistModel(QObject *parent) :
    QAbstractListModel(parent),
    m_loading(false)
{
    m_roleNames[PlaylistRoles::IdRole] = "id";
    m_roleNames[PlaylistRoles::NameRole] = "name";
    m_roleNames[PlaylistRoles::UrlRole] = "url";
    m_roleNames[PlaylistRoles::SongCountRole] = "songCount";
#if QT_VERSION < 0x050000
    this->setRoleNames(m_roleNames);
#endif
}

PlaylistModel::~PlaylistModel() {
    qDeleteAll(m_list);
    m_list.clear();
}

#if QT_VERSION >= 0x050000
QHash<int, QByteArray> PlaylistModel::roleNames() const {
    return m_roleNames;
}
#endif

int PlaylistModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant PlaylistModel::data(const QModelIndex &index, int role) const {
    if (Playlist *playlist = this->get(index)) {
        switch (role) {
        case PlaylistRoles::IdRole:
            return playlist->id();
        case PlaylistRoles::NameRole:
            return playlist->name();
        case PlaylistRoles::UrlRole:
            return playlist->url();
        case PlaylistRoles::SongCountRole:
            return playlist->songCount();
        default:
            break;
        }
    }

    return QVariant();
}

QVariant PlaylistModel::data(int row, int role) const {
    return this->data(this->index(row), role);
}

bool PlaylistModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    Q_UNUSED(index)
    Q_UNUSED(value)
    Q_UNUSED(role)

    return false;
}

bool PlaylistModel::setData(int row, const QVariant &value, int role) {
    return this->setData(this->index(row), value, role);
}

QMap<int, QVariant> PlaylistModel::itemData(const QModelIndex &index) const {
    QMap<int, QVariant> map;

    for (int role = PlaylistRoles::IdRole; role <= PlaylistRoles::SongCountRole; role++) {
        map[role] = this->data(index, role);
    }

    return map;
}

QMap<int, QVariant> PlaylistModel::itemData(int row) const {
    return this->itemData(this->index(row));
}

Playlist* PlaylistModel::get(const QModelIndex &index) const {
    if ((index.row() >= 0) && (index.row() < m_list.size())) {
        return m_list.at(index.row());
    }

    return 0;
}

Playlist* PlaylistModel::get(int row) const {
    return this->get(this->index(row));
}

bool PlaylistModel::loading() const {
    return m_loading;
}

void PlaylistModel::setLoading(bool loading) {
    if (loading != this->loading()) {
        m_loading = loading;
        emit loadingChanged(loading);
    }
}

QList<Playlist*> PlaylistModel::playlists() const {
    return m_list;
}

void PlaylistModel::appendPlaylist(Playlist *playlist) {
    this->beginInsertRows(QModelIndex(), m_list.size(), m_list.size());
    m_list.append(playlist);
    this->endInsertRows();

    emit countChanged(this->rowCount());
}

void PlaylistModel::appendPlaylists(QList<Playlist *> playlists) {
    this->beginInsertRows(QModelIndex(), m_list.size(), m_list.size() + playlists.size() - 1);
    m_list.append(playlists);
    this->endInsertRows();

    emit countChanged(this->rowCount());
}

void PlaylistModel::insertPlaylist(int i, Playlist *playlist) {
    if ((i >= 0) && (i < m_list.size())) {
        this->beginInsertRows(QModelIndex(), i, i);
        m_list.insert(i, playlist);
        this->endInsertRows();
    }
    else {
        this->appendPlaylist(playlist);
    }

    emit countChanged(this->rowCount());
}

void PlaylistModel::insertPlaylists(int i, QList<Playlist *> playlists) {
    if ((i >= 0) && (i < m_list.size())) {
        this->beginInsertRows(QModelIndex(), i, i + playlists.size() - 1);

        foreach (Playlist *playlist, playlists) {
            m_list.insert(i, playlist);
            i++;
        }

        this->endInsertRows();
    }
    else {
        this->appendPlaylists(playlists);
    }

    emit countChanged(this->rowCount());
}

void PlaylistModel::removePlaylist(int i) {
    if ((i >= 0) && (i < m_list.size())) {
        this->beginRemoveRows(QModelIndex(), i, i);
        m_list.takeAt(i)->deleteLater();
        this->endRemoveRows();
    }

    emit countChanged(this->rowCount());
}

void PlaylistModel::removePlaylists(int i, int count) {
    if ((i >= 0) && ((i + count) <= m_list.size())) {
        this->beginRemoveRows(QModelIndex(), i, i + count - 1);

        for (int ii = i; ii < (i + count); ii++) {
            m_list.takeAt(i)->deleteLater();
        }

        this->endRemoveRows();
    }

    emit countChanged(this->rowCount());
}

void PlaylistModel::reload() {
    this->clear();
    this->getPlaylists();
}

void PlaylistModel::clear() {
    this->removePlaylists(0, m_list.size());
}

void PlaylistModel::getPlaylists() {
    this->setLoading(true);

    PlaylistList *list = Music::getPlaylists();
    this->connect(list, SIGNAL(ready(PlaylistList*)), this, SLOT(addPlaylists(PlaylistList*)));
}

void PlaylistModel::addPlaylists(PlaylistList *list) {
    switch (list->error()) {
    case ReplyError::NoError:
        if (list->count() > 0) {
            this->appendPlaylists(list->playlists());
        }

        break;
    default:
        Notifications::showError(list->errorString());
        break;
    }

    this->setLoading(false);

    list->deleteLater();
}
