#include "albummodel.h"
#include "music.h"
#include "notifications.h"
#include "definitions.h"
#include "artworkcache.h"

AlbumModel::AlbumModel(QObject *parent) :
    QAbstractListModel(parent),
    m_cache(new ArtworkCache(this)),
    m_loading(false)
{
    m_roleNames[AlbumRoles::IdRole] = "id";
    m_roleNames[AlbumRoles::TitleRole] = "title";
    m_roleNames[AlbumRoles::ArtistRole] = "artist";
    m_roleNames[AlbumRoles::ArtistIdRole] = "artistId";
    m_roleNames[AlbumRoles::UrlRole] = "url";
    m_roleNames[AlbumRoles::ArtworkUrlRole] = "artworkUrl";
    m_roleNames[AlbumRoles::ArtworkRole] = "artwork";
    m_roleNames[AlbumRoles::YearRole] = "year";
    m_roleNames[AlbumRoles::DateRole] = "date";
#if QT_VERSION < 0x050000
    this->setRoleNames(m_roleNames);
#endif
    this->connect(m_cache, SIGNAL(artworkReady()), this, SLOT(onArtworkReady()));
}

AlbumModel::~AlbumModel() {
    qDeleteAll(m_list);
    m_list.clear();
}

#if QT_VERSION >= 0x050000
QHash<int, QByteArray> AlbumModel::roleNames() const {
    return m_roleNames;
}
#endif

int AlbumModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant AlbumModel::data(const QModelIndex &index, int role) const {
    if (Album *album = this->get(index)) {
        switch (role) {
        case AlbumRoles::IdRole:
            return album->id();
        case AlbumRoles::TitleRole:
            return album->title();
        case AlbumRoles::ArtistRole:
            return album->artist();
        case AlbumRoles::ArtistIdRole:
            return album->artistId();
        case AlbumRoles::UrlRole:
            return album->url();
        case AlbumRoles::ArtworkUrlRole:
            return album->artworkUrl();
        case AlbumRoles::ArtworkRole:
            return m_cache->artwork(album->artworkUrl(), THUMBNAIL_SIZE);
        case AlbumRoles::YearRole:
            return album->year();
        case AlbumRoles::DateRole:
            return album->date();
        default:
            break;
        }
    }

    return QVariant();
}

QVariant AlbumModel::data(int row, int role) const {
    return this->data(this->index(row), role);
}

bool AlbumModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    Q_UNUSED(index)
    Q_UNUSED(value)
    Q_UNUSED(role)

    return false;
}

bool AlbumModel::setData(int row, const QVariant &value, int role) {
    return this->setData(this->index(row), value, role);
}

QMap<int, QVariant> AlbumModel::itemData(const QModelIndex &index) const {
    QMap<int, QVariant> map;

    for (int role = AlbumRoles::IdRole; role <= AlbumRoles::DateRole; role++) {
        map[role] = this->data(index, role);
    }

    return map;
}

QMap<int, QVariant> AlbumModel::itemData(int row) const {
    return this->itemData(this->index(row));
}

Album* AlbumModel::get(const QModelIndex &index) const {
    if ((index.row() >= 0) && (index.row() < m_list.size())) {
        return m_list.at(index.row());
    }

    return 0;
}

Album* AlbumModel::get(int row) const {
    return this->get(this->index(row));
}

bool AlbumModel::loading() const {
    return m_loading;
}

void AlbumModel::setLoading(bool loading) {
    if (loading != this->loading()) {
        m_loading = loading;
        emit loadingChanged(loading);
    }
}

QList<Album*> AlbumModel::albums() const {
    return m_list;
}

QString AlbumModel::artistId() const {
    return m_artistId;
}

void AlbumModel::setArtistId(const QString &id) {
    m_artistId = id;
}

void AlbumModel::appendAlbum(Album *album) {
    this->beginInsertRows(QModelIndex(), m_list.size(), m_list.size());
    m_list.append(album);
    this->endInsertRows();

    emit countChanged(this->rowCount());
}

void AlbumModel::appendAlbums(QList<Album *> albums) {
    this->beginInsertRows(QModelIndex(), m_list.size(), m_list.size() + albums.size() - 1);
    m_list.append(albums);
    this->endInsertRows();

    emit countChanged(this->rowCount());
}

void AlbumModel::insertAlbum(int i, Album *album) {
    if ((i >= 0) && (i < m_list.size())) {
        this->beginInsertRows(QModelIndex(), i, i);
        m_list.insert(i, album);
        this->endInsertRows();
    }
    else {
        this->appendAlbum(album);
    }

    emit countChanged(this->rowCount());
}

void AlbumModel::insertAlbums(int i, QList<Album *> albums) {
    if ((i >= 0) && (i < m_list.size())) {
        this->beginInsertRows(QModelIndex(), i, i + albums.size() - 1);

        foreach (Album *album, albums) {
            m_list.insert(i, album);
            i++;
        }

        this->endInsertRows();
    }
    else {
        this->appendAlbums(albums);
    }

    emit countChanged(this->rowCount());
}

void AlbumModel::removeAlbum(int i) {
    if ((i >= 0) && (i < m_list.size())) {
        this->beginRemoveRows(QModelIndex(), i, i);
        m_list.takeAt(i)->deleteLater();
        this->endRemoveRows();
    }

    emit countChanged(this->rowCount());
}

void AlbumModel::removeAlbums(int i, int count) {
    if ((i >= 0) && ((i + count) <= m_list.size())) {
        this->beginRemoveRows(QModelIndex(), i, i + count - 1);

        for (int ii = i; ii < (i + count); ii++) {
            m_list.takeAt(i)->deleteLater();
        }

        this->endRemoveRows();
    }

    emit countChanged(this->rowCount());
}

void AlbumModel::reload() {
    this->clear();

    if (this->artistId().isEmpty()) {
        this->getAlbums();
    }
    else {
        this->getArtistAlbums(this->artistId());
    }
}

void AlbumModel::clear() {
    this->removeAlbums(0, m_list.size());
}

void AlbumModel::getAlbums() {
    this->setLoading(true);
    this->setArtistId(QString());

    AlbumList *list = Music::getAlbums();
    this->connect(list, SIGNAL(ready(AlbumList*)), this, SLOT(addAlbums(AlbumList*)));
}

void AlbumModel::getArtistAlbums(const QString &artistId) {
    this->setLoading(true);
    this->setArtistId(artistId);

    AlbumList *list = Music::getArtistAlbums(artistId);
    this->connect(list, SIGNAL(ready(AlbumList*)), this, SLOT(addAlbums(AlbumList*)));
}

void AlbumModel::addAlbums(AlbumList *list) {
    switch (list->error()) {
    case ReplyError::NoError:
        if (list->count() > 0) {
            this->appendAlbums(list->albums());
        }

        break;
    default:
        Notifications::showError(list->errorString());
        break;
    }

    this->setLoading(false);

    list->deleteLater();
}

void AlbumModel::onArtworkReady() {
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}
