#include "mediaplayer.h"
#include "music.h"
#include "../../shared/playbackqueuemodel.h"
#include "../../shared/settings.h"
#include "../../shared/notifications.h"
#include <QDateTime>
#include <qplatformdefs.h>
#if QT_VERSION >= 0x050000
#include <QStandardPaths>
#else
#include <QDesktopServices>
#endif
#include <QDebug>

#if (defined Q_WS_MAEMO_5) || (defined MEEGO_EDITION_HARMATTAN) || (defined SAILFISH_OS)
static const QString CACHE_DIRECTORY("/home/user/MyDocs/toBuntu/.music_cache/");
#elif (defined Q_OS_SYMBIAN)
static const QString CACHE_DIRECTORY("E:/toBuntu/.music_cache/");
#elif QT_VERSION >= 0x050000
static const QString CACHE_DIRECTORY(QStandardPaths::writableLocation(QStandardPaths::HomeLocation) + "/toBuntu/.music_cache/");
#else
static const QString CACHE_DIRECTORY(QDesktopServices::storageLocation(QDesktopServices::HomeLocation) + "/toBuntu/.music_cache/");
#endif

MediaPlayer* MediaPlayer::m_instance = 0;

MediaPlayer::MediaPlayer() :
    QObject(),
    m_player(new QMediaPlayer(this, QMediaPlayer::StreamPlayback)),
    m_queue(new PlaybackQueueModel(this)),
    m_song(0),
    m_stream(0),
    m_index(0),
    m_playbackMode(Media::Sequential)
{
    if (!m_instance) {
        m_instance = this;
    }

    this->connect(m_player, SIGNAL(bufferStatusChanged(int)), this, SIGNAL(bufferStatusChanged(int)));
    this->connect(m_player, SIGNAL(positionChanged(qint64)), this, SIGNAL(positionChanged(qint64)));
    this->connect(m_player, SIGNAL(volumeChanged(int)), this, SIGNAL(volumeChanged(int)));
    this->connect(m_player, SIGNAL(mediaStatusChanged(QMediaPlayer::MediaStatus)), this, SLOT(onMediaStatusChanged()));
    this->connect(m_player, SIGNAL(stateChanged(QMediaPlayer::State)), this, SLOT(onStateChanged()));
    this->connect(m_player, SIGNAL(error(QMediaPlayer::Error)), this, SLOT(onError()));
    this->connect(m_queue, SIGNAL(countChanged(int)), this, SLOT(onQueueCountChanged(int)));
    this->connect(Settings::instance(), SIGNAL(cacheMusicStreamsChanged(bool)), this, SLOT(onCacheMusicStreamsChanged(bool)));
}

MediaPlayer::~MediaPlayer() {}

MediaPlayer* MediaPlayer::instance() {
    return !m_instance ? new MediaPlayer : m_instance;
}

PlaybackQueueModel* MediaPlayer::queue() const {
    return m_queue;
}

int MediaPlayer::currentIndex() const {
    return m_index;
}

void MediaPlayer::setCurrentIndex(int index) {
    if ((index >= 0) && (index <= m_queue->rowCount())) {
        m_index = index;
        emit currentIndexChanged(index);
    }
}

int MediaPlayer::count() const {
    return m_queue->rowCount();
}

Song* MediaPlayer::currentSong() const {
    return m_song;
}

void MediaPlayer::setCurrentSong(Song *song) {
    if (song != this->currentSong()) {
        m_song = song;
        emit currentSongChanged(song);
    }
}

qint64 MediaPlayer::streamPosition() const {
    return !m_stream ? 0 : m_stream->streamPosition();
}

qint64 MediaPlayer::streamSize() const {
    return !m_song ? 0 : m_song->size();
}

qint64 MediaPlayer::position() const {
    return m_player->position();
}

void MediaPlayer::setPosition(qint64 position) {
    m_player->setPosition(position);
}

qint64 MediaPlayer::duration() const {
    return !m_song ? m_player->duration() : m_song->duration();
}

int MediaPlayer::volume() const {
    return m_player->volume();
}

void MediaPlayer::setVolume(int volume) {
    m_player->setVolume(volume);
}

int MediaPlayer::bufferStatus() const {
    return m_player->bufferStatus();
}

Media::PlaybackMode MediaPlayer::playbackMode() const {
    return m_playbackMode;
}

void MediaPlayer::setPlaybackMode(Media::PlaybackMode mode) {
    if (mode != this->playbackMode()) {
        m_playbackMode = mode;
        emit playbackModeChanged(mode);
    }
}

void MediaPlayer::togglePlaybackMode() {
    switch (this->playbackMode()) {
    case Media::Sequential:
        this->setPlaybackMode(Media::RepeatAll);
        return;
    case Media::RepeatAll:
        this->setPlaybackMode(Media::RepeatOne);
        return;
    default:
        this->setPlaybackMode(Media::Sequential);
        return;
    }
}

Media::MediaStatus MediaPlayer::mediaStatus() const {
    return Media::MediaStatus(m_player->mediaStatus());
}

Media::State MediaPlayer::state() const {
    return Media::State(m_player->state());
}

Media::Error MediaPlayer::error() const {
    return Media::Error(m_player->error());
}

QString MediaPlayer::errorString() const {
    return m_player->errorString();
}

void MediaPlayer::play(Song *song) {
    m_queue->clear();
    m_queue->appendSong(song);
    this->play_p(song);
}

void MediaPlayer::play_p(Song *song) {
    this->setCurrentSong(song);
    this->stop();

    m_stream = m_cache.value(song->url(), 0);

    if (!m_stream) {
        m_stream = Music::getMusicStream(song->url(), CACHE_DIRECTORY + QString("%1.%2").arg(QDateTime::currentMSecsSinceEpoch()).arg(song->fileSuffix()));
        m_stream->setParent(this);

        if (Settings::instance()->cacheMusicStreams()) {
            m_cache.insert(song->url(), m_stream);
        }

        this->connect(m_stream, SIGNAL(streamPositionChanged(qint64)), this, SIGNAL(streamPositionChanged(qint64)));
        this->connect(m_stream, SIGNAL(statusChanged(StreamStatus::Status)), this, SLOT(onStreamStatusChanged(StreamStatus::Status)));
    }

    emit streamPositionChanged(m_stream->streamPosition());

    switch (m_stream->status()) {
    case StreamStatus::Finished:
        m_player->play();
        return;
    default:
        m_stream->start();
        return;
    }
}

void MediaPlayer::play(const QList<Song *> &songs) {
    if (!songs.isEmpty()) {
        m_queue->clear();
        m_queue->appendSongs(songs);
        this->play_p(songs.first());
    }
}

void MediaPlayer::play(int index) {
    this->setCurrentIndex(index);
    this->play_p(m_queue->get(index));
}

void MediaPlayer::play() {
    if (!m_song) {
        return;
    }

    if (!m_stream) {
        m_stream = Music::getMusicStream(m_song->url(), CACHE_DIRECTORY + QString("%1.%2").arg(QDateTime::currentMSecsSinceEpoch()).arg(m_song->fileSuffix()));
        m_stream->setParent(this);

        if (Settings::instance()->cacheMusicStreams()) {
            m_cache.insert(m_song->url(), m_stream);
        }
    }

    switch (m_stream->status()) {
    case StreamStatus::Finished:
        m_player->play();
        return;
    default:
        m_stream->start();
        return;
    }
}

void MediaPlayer::pause() {
    m_player->pause();
}

void MediaPlayer::togglePlayPause() {
    switch (this->state()) {
    case Media::PlayingState:
        this->pause();
        return;
    default:
        this->play();
        return;
    }
}

void MediaPlayer::stop() {
    m_player->stop();

    if (m_stream) {
        m_stream->stop();

        if (!Settings::instance()->cacheMusicStreams()) {
            m_cache.remove(m_stream->url());
            m_stream->deleteLater();
            m_stream = 0;
        }
    }
}

void MediaPlayer::next() {
    this->play(this->currentIndex() + 1);
}

void MediaPlayer::previous() {
    this->play(this->currentIndex() - 1);
}

void MediaPlayer::addSong(Song *song) {
    m_queue->appendSong(song);
    Notifications::showInformation(tr("Song added to playback queue."));
}

void MediaPlayer::addSongs(const QList<Song *> &songs) {
    m_queue->appendSongs(songs);
    Notifications::showInformation(tr("Songs added to playback queue."));
}

void MediaPlayer::removeSong(int index) {
    m_queue->removeSong(index);

    if ((index == this->currentIndex()) && (m_queue->rowCount() > 0)) {
        this->setCurrentIndex(this->currentIndex() - 1);
        this->next();
    }
    else if (index < this->currentIndex()) {
        this->setCurrentIndex(this->currentIndex() - 1);
    }
}

void MediaPlayer::clearSongs() {
    m_queue->clear();
}

void MediaPlayer::onQueueCountChanged(int count) {
    if (count == 0) {
        this->stop();
        this->setCurrentSong(0);
    }
    else if (!m_song) {
        this->setCurrentSong(m_queue->get(0));
    }

    emit countChanged(count);
}

void MediaPlayer::onStreamStatusChanged(StreamStatus::Status status) {
    switch (status) {
    case StreamStatus::Ready:
        switch (this->state()) {
        case Media::PausedState:
            return;
        default:
            if (m_stream) {
                m_stream->open(QIODevice::ReadOnly);
                m_player->setMedia(QMediaContent(), m_stream);
                m_player->play();
            }

            return;
        }
    case StreamStatus::Error:
        m_player->stop();

        if (m_stream) {
            Notifications::showError(m_stream->errorString());
        }

        return;
    default:
        return;
    }
}

void MediaPlayer::onMediaStatusChanged() {
    emit mediaStatusChanged(this->mediaStatus());
}

void MediaPlayer::onStateChanged() {
    emit stateChanged(this->state());
}

void MediaPlayer::onError() {
    emit error(this->error());
    Notifications::showError(this->errorString());
}

void MediaPlayer::onCacheMusicStreamsChanged(bool cache) {
    if (cache) {
        if (m_stream) {
            m_cache.insert(m_stream->url(), m_stream);
        }
    }
    else {
        foreach (MusicStream *stream, m_cache.values()) {
            if ((stream) && (stream != m_stream)) {
                stream->deleteLater();
            }
        }

        m_cache.clear();
    }
}
