#ifndef UBUNTUONE_H
#define UBUNTUONE_H

#include <QObject>
#include <QMap>
#include <QVariantMap>
#include <QStringList>
#include <QFile>
#include <QProcess>

class QNetworkAccessManager;
class QNetworkReply;

class UbuntuOne : public QObject
{
    Q_OBJECT

public:
    explicit UbuntuOne(QObject *parent = 0);
    ~UbuntuOne();
    inline QNetworkAccessManager* networkAccessManager() { return m_nam; }
    inline void setNetworkAccessManager(QNetworkAccessManager *manager) { m_nam = manager; }
    inline bool userSignedIn() { return ((!m_accessToken.isEmpty()) && (!m_tokenSecret.isEmpty())); }
    inline QString rootNodePath() const { return m_rootNode; }
    QByteArray getOAuthHeader(const QString &method, const QString &url, QMap<QString, QString> params);
    QUrl getMusicUrl(const QString &method, const QMap<QString, QString> params);
    QUrl getMusicPlaybackUrl(const QString &id);
    QUrl getMusicDownloadUrl(const QString &id);

public slots:
    inline void setFileHandlers(const QVariantMap &handlers) { m_fileHandlers = handlers; }
    void setCredentials(const QVariantMap &token);
    void signIn(const QString &user, const QString &pass);
    void signOut();
    void getUserProfile();
    void getMobileCredentials();
    void getFiles(const QString &resourcePath);
    void createFolder(QString path);
    void renameItem(const QString &resourcePath, const QString &newPath);
    void moveItem(const QString &resourcePath, const QString &newPath);
    void deleteItem(const QString &resourcePath);
    void setPublished(const QString &resourcePath, bool publish);
    void openFile(const QString &contentPath);
    void createPlaylist(const QString &name);
    void deletePlaylist(const QString &id);

private slots:
    void checkIfSignedIn();
    void checkMobileCredentials();
    void checkTokenSubmission();
    void checkUserProfile();
    void checkUserAccount();
    void checkFiles();
    void checkIfCreated();
    void checkIfMoved();
    void checkIfRenamed();
    void checkIfDeleted();
    void checkPublished();
    void updateDownloadProgress(qint64 bytesReceived, qint64 bytesTotal);
    void downloadReadyRead();
    void onFileDownloaded();
    void onProcessError(QProcess::ProcessError error);
    void onProcessFinished(int exitCode);
    void checkPlaylistCreated();
    void checkPlaylistDeleted();

private:
    void submitTokenToUbuntuOne();
    void getUserAccount();

signals:
    void busy(const QString &message, int numberOfOperations = 1);
    void busyProgressChanged(int progress);
    void alert(const QString &message);
    void warning(const QString &message);
    void filesError();
    void signedIn(const QVariantMap &token);
    void signedOut();
    void signedInToMusic();
    void noMusicAccount();
    void gotUserProfile(const QVariantMap &profile);
    void gotFiles(const QVariantMap &files);
    void folderCreated(const QVariantMap &folder);
    void itemRenamed(const QVariantMap &folder);
    void itemMoved(const QVariantMap &folder);
    void itemDeleted();
    void publishedChanged(const QVariantMap &file);
    void publishError();
    void playlistCreated();
    void playlistDeleted();

private:
    QNetworkAccessManager *m_nam;
    QString m_appName;
    QString m_consumerKey;
    QString m_consumerSecret;
    QString m_accessToken;
    QString m_tokenSecret;
    QString m_mobileUser;
    QString m_mobilePass;
    QString m_email;
    QString m_rootNode;
    QVariantMap m_profile;
    QNetworkReply *m_tempReply;
    QFile m_tempFile;
    QProcess *m_fileProcess;
    QVariantMap m_fileHandlers;
};

#endif // UBUNTUONE_H
