#include "ubuntuone.h"
#include "baseurls.h"
#include "json.h"
#include "oauth.h"
#include "utils.h"
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QUrl>
#include <QDomDocument>
#include <QDomElement>
#include <QDesktopServices>
#include <QTimer>
#include <QDir>
#include <QProcess>
#include <QDebug>

#define TEMP_DIR "/home/user/toBuntu/.temp/"

using namespace QtOAuth;
using namespace QtJson;

UbuntuOne::UbuntuOne(QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_rootNode("/~/Ubuntu One"),
    m_tempReply(0),
    m_fileProcess(new QProcess(this))
{
    connect(m_fileProcess, SIGNAL(error(QProcess::ProcessError)), this, SLOT(onProcessError(QProcess::ProcessError)));
    connect(m_fileProcess, SIGNAL(finished(int)), this, SLOT(onProcessFinished(int)));
}

UbuntuOne::~UbuntuOne() {
    QDir dir(TEMP_DIR);
    QStringList tempFiles = dir.entryList();

    while (!tempFiles.isEmpty()) {
        dir.remove(TEMP_DIR + tempFiles.takeFirst());
    }
}

void UbuntuOne::signIn(const QString &user, const QString &pass) {
    emit busy(tr("Signing in..."));
    m_email = user;
    QUrl url("https://login.ubuntu.com/api/1.0/authentications?ws.op=authenticate&token_name=Ubuntu%20One%20@%20tobuntu");
    QByteArray credentials(user.toUtf8() + ":" + pass.toUtf8());
    QNetworkRequest request(url);
    request.setRawHeader("Authorization", "Basic " +  credentials.toBase64());
    request.setRawHeader("User-Agent", QString("toBuntu/%1 (Maemo; Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkIfSignedIn()));
}

void UbuntuOne::checkIfSignedIn() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error. Cannot sign in"));
        return;
    }

    bool ok;
    QString response(reply->readAll());
    QVariantMap result = Json::parse(response, ok).toMap();

    if (!ok) {
        emit warning(tr("Error parsing server response"));
        return;
    }

    QString name(result.value("name").toString());
    QString key(result.value("consumer_key").toString());
    QString secret(result.value("consumer_secret").toString());
    QString token(result.value("token").toString());
    QString tSecret(result.value("token_secret").toString());

    if ((!token.isEmpty()) && (!tSecret.isEmpty())) {
        m_appName = name;
        m_consumerKey = key;
        m_consumerSecret = secret;
        m_accessToken = token;
        m_tokenSecret = tSecret;
        submitTokenToUbuntuOne();
    }
    else {
        int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

        if (statusCode == 401) {
            emit warning(tr("Username and/or password rejected. Please check and try again."));
        }
        else if (statusCode == 503) {
            emit warning(tr("Server is temporarily inaccessible. Please try again later."));
        }
        else {
            QString status(reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
            emit warning(status);
        }
    }

    reply->deleteLater();
}

void UbuntuOne::submitTokenToUbuntuOne() {
    QUrl url("https://one.ubuntu.com/oauth/sso-finished-so-get-tokens/" + m_email);
    m_email.clear();
    QNetworkRequest request(url);
    request.setRawHeader("Host", "one.ubuntu.com");
    request.setRawHeader("Authorization", getOAuthHeader("GET", url.toString(), QMap<QString, QString>()));
    request.setRawHeader("User-Agent", QString("toBuntu/%1 (Maemo; Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkTokenSubmission()));
}

void UbuntuOne::checkTokenSubmission() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    QVariantMap token;
    int status = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (status == 200) {
        emit alert(tr("You have added your Ubuntu One account"));
        token.insert("appName", m_appName);
        token.insert("consumerKey", m_consumerKey);
        token.insert("consumerSecret", m_consumerSecret);
        token.insert("accessToken", m_accessToken);
        token.insert("tokenSecret", m_tokenSecret);
        emit signedIn(token);
    }
    else {
        emit warning(tr("Unable to add your Ubuntu One account"));
        setCredentials(token);
    }

    reply->deleteLater();
}

void UbuntuOne::getMobileCredentials() {
    if ((!m_mobileUser.isEmpty()) && (!m_mobilePass.isEmpty())) {
        return;
    }

    busy(tr("Signing in..."));
    QString urlString(BASE_URL_MOBILE_LOGIN);
    QString url(urlString + "?platform=maemo&manufacturer=nokia&model=n900");
    QMap<QString, QString> params;
    params["platform"] = "maemo";
    params["manufacturer"] = "nokia";
    params["model"] = "n900";
    QNetworkRequest request(url);
    request.setRawHeader("Host", "one.ubuntu.com");
    request.setRawHeader("Authorization", getOAuthHeader("GET", urlString, params));
    request.setRawHeader("User-Agent", QString("toBuntu/%1 (Maemo; Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkMobileCredentials()));
}

void UbuntuOne::checkMobileCredentials() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error. Cannot sign in"));
        return;
    }

    int status = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if ((status == 200) || (status == 201)) {
        QByteArray response(reply->readAll());
        QDomDocument doc;
        doc.setContent(response);
        QDomElement element = doc.documentElement();
        QUrl loginUrl(element.attribute("action"));

        m_mobileUser = loginUrl.userName();
        m_mobilePass = loginUrl.password();
        emit signedInToMusic();
    }
    else if (status == 401) {
        emit warning(tr("Access to Ubunut One Music is forbidden"));
        emit noMusicAccount();
    }
    else {
        QString statusText(reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
        emit warning(statusText);
    }

    reply->deleteLater();
}

void UbuntuOne::signOut() {
    setCredentials(QVariantMap());
    QDesktopServices::openUrl(QUrl("https://one.ubuntu.com/account/"));
    emit signedOut();
}

void UbuntuOne::setCredentials(const QVariantMap &token) {
    m_appName = token.value("appName").toString();
    m_consumerKey = token.value("consumerKey").toString();
    m_consumerSecret = token.value("consumerSecret").toString();
    m_accessToken = token.value("accessToken").toString();
    m_tokenSecret = token.value("tokenSecret").toString();
}

void UbuntuOne::getUserProfile() {
    emit busy("Retrieving profile...");
    QUrl url(BASE_URL_FILES);
    QNetworkRequest request(url);
    request.setRawHeader("Host", "one.ubuntu.com");
    request.setRawHeader("Authorization", getOAuthHeader("GET", url.toString(), QMap<QString, QString>()));
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkUserProfile()));
}

void UbuntuOne::checkUserProfile() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    bool ok;
    QString response(reply->readAll());
    QVariantMap result = Json::parse(response, ok).toMap();

    if (!ok) {
        emit warning(tr("Unable to retrieve account details"));
        return;
    }

    m_rootNode = result.value("root_node_path").toString();
    m_profile = result;
    getUserAccount();

    reply->deleteLater();
}

void UbuntuOne::getUserAccount() {
    QUrl url("https://one.ubuntu.com/api/account/");
    QNetworkRequest request(url);
    request.setRawHeader("Host", "one.ubuntu.com");
    request.setRawHeader("Authorization", getOAuthHeader("GET", url.toString(), QMap<QString, QString>()));
    request.setRawHeader("User-Agent", QString("toBuntu/%1 (Maemo; Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkUserAccount()));
}

void UbuntuOne::checkUserAccount() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    bool ok;
    QString response(reply->readAll());
    QVariantMap result = Json::parse(response, ok).toMap();

    if (!ok) {
        emit warning(tr("Unable to retrieve account details"));
        return;
    }

    m_profile.insert("account", result);
    emit gotUserProfile(m_profile);

    reply->deleteLater();
}

void UbuntuOne::getFiles(const QString &resourcePath) {
    QString urlString(BASE_URL_FILES + resourcePath.toUtf8().toPercentEncoding(":/~_?="));
    QUrl url(urlString);
    url.addQueryItem("include_children", "true");
    QString method("GET");
    QMap<QString, QString> params;
    params["include_children"] = "true";
    QNetworkRequest request(url);
    request.setRawHeader("Host", "one.ubuntu.com");
    request.setRawHeader("Authorization", getOAuthHeader(method, urlString, params));
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkFiles()));
}

void UbuntuOne::checkFiles() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit filesError();
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QString response(reply->readAll());
        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (ok) {
            emit gotFiles(result);
        }
        else {
            emit filesError();
            emit warning(tr("Unable to parse server response"));
        }
    }
    else {
        QString status(reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
        emit filesError();
        emit warning(status);
    }

    reply->deleteLater();
}

void UbuntuOne::deleteItem(const QString &resourcePath) {
    QString urlString(BASE_URL_FILES + resourcePath.toUtf8().toPercentEncoding(":/~_?="));
    QUrl url(urlString);
    QNetworkRequest request(url);
    request.setRawHeader("Authorization", getOAuthHeader("DELETE", urlString, QMap<QString, QString>()));
    request.setRawHeader("User-Agent", QString("toBuntu/%1 (Maemo; Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = networkAccessManager()->deleteResource(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkIfDeleted()));
}

void UbuntuOne::checkIfDeleted() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        emit itemDeleted();
    }
    else {
        QString status(reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
        emit warning(status);
    }

    reply->deleteLater();
}

void UbuntuOne::createFolder(QString path) {
    QString urlString(BASE_URL_FILES + path.toUtf8().toPercentEncoding(":/~_?="));
    QUrl url(urlString);
    QByteArray json("{ \"kind\": \"directory\" }");
    QNetworkRequest request(url);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");
    request.setRawHeader("Authorization", getOAuthHeader("PUT", urlString, QMap<QString, QString>()));
    request.setRawHeader("User-Agent", QString("toBuntu/%1 (Maemo; Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = networkAccessManager()->put(request, json);
    connect(reply, SIGNAL(finished()), this, SLOT(checkIfCreated()));
}

void UbuntuOne::checkIfCreated() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QString response(reply->readAll());
        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (ok) {
            emit folderCreated(result);
        }
        else {
            emit warning(tr("Unable to parse server response"));
        }
    }
    else {
        QString status(reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
        emit warning(status);
    }

    reply->deleteLater();
}

void UbuntuOne::renameItem(const QString &resourcePath, const QString &newPath) {
    QString urlString(BASE_URL_FILES + resourcePath.toUtf8().toPercentEncoding(":/~_?="));
    QUrl url(urlString);
    QByteArray json("{ \"path\": \"" + newPath.toUtf8() + "\" }");
    QNetworkRequest request(url);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");
    request.setRawHeader("Authorization", getOAuthHeader("PUT", urlString, QMap<QString, QString>()));
    request.setRawHeader("User-Agent", QString("toBuntu/%1 (Maemo; Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = networkAccessManager()->put(request, json);
    connect(reply, SIGNAL(finished()), this, SLOT(checkIfRenamed()));
}

void UbuntuOne::checkIfRenamed() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QString response(reply->readAll());
        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (ok) {
            emit itemRenamed(result);
        }
        else {
            emit warning(tr("Unable to parse server response"));
        }
    }
    else {
        QString status(reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
        emit warning(status);
    }

    reply->deleteLater();
}

void UbuntuOne::moveItem(const QString &resourcePath, const QString &newPath) {
    QString urlString(BASE_URL_FILES + resourcePath.toUtf8().toPercentEncoding(":/~_?="));
    QUrl url(urlString);
    QByteArray json("{ \"path\": \"" + newPath.toUtf8() + "\" }");
    QNetworkRequest request(url);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");
    request.setRawHeader("Authorization", getOAuthHeader("PUT", urlString, QMap<QString, QString>()));
    request.setRawHeader("User-Agent", QString("toBuntu/%1 (Maemo; Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = networkAccessManager()->put(request, json);
    connect(reply, SIGNAL(finished()), this, SLOT(checkIfMoved()));
}

void UbuntuOne::checkIfMoved() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QString response(reply->readAll());
        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (ok) {
            emit itemMoved(result);
        }
        else {
            emit warning(tr("Unable to parse server response"));
        }
    }
    else {
        QString status(reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
        emit warning(status);
    }

    reply->deleteLater();
}

void UbuntuOne::setPublished(const QString &resourcePath, bool publish) {
    QString urlString(BASE_URL_FILES + resourcePath.toUtf8().toPercentEncoding(":/~_?="));
    QUrl url(urlString);
    QByteArray isPublic = (publish ? "true" : "false");
    QByteArray json("{ \"is_public\": " + isPublic + "}");
    QNetworkRequest request(url);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");
    request.setRawHeader("Authorization", getOAuthHeader("PUT", urlString, QMap<QString, QString>()));
    request.setRawHeader("User-Agent", QString("toBuntu/%1 (Maemo; Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = networkAccessManager()->put(request, json);
    connect(reply, SIGNAL(finished()), this, SLOT(checkPublished()));
}

void UbuntuOne::checkPublished() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QString response(reply->readAll());
        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (ok) {
            emit publishedChanged(result);
        }
        else {
            emit publishError();
            QString status(reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
            emit warning(status);
        }
    }
    else {
        emit publishError();
        QString status(reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
        emit warning(status);
    }

    reply->deleteLater();
}

void UbuntuOne::openFile(const QString &contentPath) {
    emit busy(tr("Retrieving file"), 100);
    m_tempFile.setFileName(QString("%1temp_%2_%3").arg(TEMP_DIR).arg(QDateTime::currentMSecsSinceEpoch()).arg(contentPath.section('/', -1)));

    if (!m_tempFile.open(QIODevice::WriteOnly)) {
        emit warning(tr("Unable to open temporaray file"));
        return;
    }

    QString urlString(CONTENT_ROOT_FILES + contentPath.toUtf8().toPercentEncoding(":/~_?="));
    QUrl url(urlString);
    QNetworkRequest request(url);
    request.setRawHeader("Authorization", getOAuthHeader("GET", urlString, QMap<QString, QString>()));
    request.setRawHeader("User-Agent", QString("toBuntu/%1 (Maemo; Qt)").arg(Utils::versionNumberString()).toUtf8());
    m_tempReply = networkAccessManager()->get(request);
    connect(m_tempReply, SIGNAL(downloadProgress(qint64,qint64)), this, SLOT(updateDownloadProgress(qint64,qint64)));
    connect(m_tempReply, SIGNAL(readyRead()), this, SLOT(downloadReadyRead()));
    connect(m_tempReply, SIGNAL(finished()), this, SLOT(onFileDownloaded()));
}

void UbuntuOne::updateDownloadProgress(qint64 bytesReceived, qint64 bytesTotal) {
    if (bytesReceived) {
        emit busyProgressChanged(bytesReceived * 100 / bytesTotal);
    }
}

void UbuntuOne::downloadReadyRead() {
    m_tempFile.write(m_tempReply->readAll());
}

void UbuntuOne::onFileDownloaded() {
    m_tempFile.close();

    if (!m_tempReply) {
        emit warning(tr("Network error. Cannot retrieve file"));
        return;
    }

    int statusCode = m_tempReply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode != 200) {
        emit warning(tr("Unable to open file."));
        return;
    }

    QDir dir;
    dir.mkpath(TEMP_DIR);
    QString command(m_fileHandlers.value(m_tempFile.fileName().section('.', -1)).toString());

    if (!command.isEmpty()) {
        m_fileProcess->start(QString("%1 '%2'").arg(command).arg(m_tempFile.fileName()));
    }
    else if (!QDesktopServices::openUrl(QUrl::fromLocalFile(m_tempFile.fileName()))) {
        emit warning(tr("Unable to open file using default handler"));
    }

    m_tempReply->deleteLater();
}

void UbuntuOne::onProcessError(QProcess::ProcessError error) {
    Q_UNUSED(error);

    emit warning(m_fileProcess->errorString());
}

void UbuntuOne::onProcessFinished(int exitCode) {
    if (exitCode != 0) {
        emit warning(QString("%1: %2").arg(tr("Process exited with code")).arg(exitCode));
    }
}

void UbuntuOne::createPlaylist(const QString &name) {
    QMap<QString, QString> params;
    params["name"] = name;
    QUrl url = getMusicUrl("createPlaylist.view", params);
    qDebug() << url;
    QNetworkRequest request;
    request.setUrl(url);
    request.setRawHeader("User-Agent", QString("toBuntu/%1 (Maemo; Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkPlaylistCreated()));
}

void UbuntuOne::checkPlaylistCreated() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    qDebug() << response;

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();
    qDebug() << statusCode;

    if ((statusCode == 200) || (statusCode == 201)) {
        emit alert(tr("New playlist created"));
        emit playlistCreated();
    }
    else {
        QString status(reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
        emit alert(status);
    }

    reply->deleteLater();
}

void UbuntuOne::deletePlaylist(const QString &id) {
    QMap<QString, QString> params;
    params["id"] = id;
    QUrl url = getMusicUrl("deletePlaylist.view", params);
    qDebug() << url;
    QNetworkRequest request;
    request.setUrl(url);
    request.setRawHeader("User-Agent", QString("toBuntu/%1 (Maemo; Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkPlaylistDeleted()));
}

void UbuntuOne::checkPlaylistDeleted() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit warning(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    qDebug() << response;

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();
    qDebug() << statusCode;

    if ((statusCode == 200) || (statusCode == 201)) {
        emit alert(tr("Playlist deleted"));
        emit playlistDeleted();
    }
    else {
        QString status(reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
        emit alert(status);
    }

    reply->deleteLater();
}

QByteArray UbuntuOne::getOAuthHeader(const QString &method, const QString &url, QMap<QString, QString> params) {
    params.insert("oauth_consumer_key", m_consumerKey);
    params.insert("oauth_consumer_secret", m_consumerSecret);
    params.insert("oauth_token", m_accessToken);
    params.insert("oauth_token_secret", m_tokenSecret);

    return OAuth::createOAuthHeader(method, url, params);
}

QUrl UbuntuOne::getMusicUrl(const QString &method, const QMap<QString, QString> params) {
    QUrl url(BASE_URL_MUSIC + method);
    QMapIterator<QString, QString> iterator(params);

    while (iterator.hasNext()) {
        iterator.next();
        url.addQueryItem(iterator.key(), iterator.value());
    }

    url.addQueryItem("u", m_mobileUser);
    url.addQueryItem("p", m_mobilePass);
    url.addQueryItem("v", "1.2.0");
    url.addQueryItem("c", "tobuntu");

    return url;
}

QUrl UbuntuOne::getMusicPlaybackUrl(const QString &id) {
    QUrl url(BASE_URL_MUSIC + QString("stream.view"));
    url.addQueryItem("id", id);
    url.addQueryItem("u", m_mobileUser);
    url.addQueryItem("p", m_mobilePass);
    url.addQueryItem("v", "1.2.0");
    url.addQueryItem("c", "tobuntu");

    return url;
}

QUrl UbuntuOne::getMusicDownloadUrl(const QString &id) {
    QUrl url(BASE_URL_MUSIC + QString("stream.view"));
    url.addQueryItem("id", id);
    url.addQueryItem("u", m_mobileUser);
    url.addQueryItem("p", m_mobilePass);
    url.addQueryItem("v", "1.2.0");
    url.addQueryItem("c", "tobuntu");

    return url;
}
