#ifndef TRANSFERWORKER_H
#define TRANSFERWORKER_H

#include "enums.h"
#include "ubuntuone.h"
#include "transferitem.h"
#include <QUrl>
#include <QFile>
#include <QTime>

class QNetworkAccessManager;
class QNetworkReply;
class QNetworkRequest;

class TransferWorker : public QObject
{
    Q_OBJECT

public:
    explicit TransferWorker(QObject *parent = 0);
    inline QNetworkAccessManager *networkAccessManager() const { return m_nam; }
    inline void setNetworkAccessManager(QNetworkAccessManager *manager) { m_nam = manager; }
    inline void setUbuntuOneSession(UbuntuOne *session) { m_ubuntu = session; }
    inline bool transferInProgress() const { return m_busy; }

public slots:
    void downloadFile(QSharedPointer<TransferItem> transfer);
    void uploadFile(QSharedPointer<TransferItem> transfer);
    void pauseDownload();
    void cancelDownload();
    void cancelUpload();
    void deleteIncompleteDownload(const QString &filePath);

private slots:
    void performDownload(const QUrl &url);
    void updateSize();
    void updateDownloadProgress(qint64 bytesReceived, qint64 bytesTotal);
    void updateUploadProgress(qint64 bytesSent, qint64 bytesTotal);
    void downloadFinished();
    void downloadReadyRead();
    void uploadFinished();
    void uploadPublished(const QVariantMap &file);

private:
    inline void setTransferInProgress(bool busy) { m_busy = busy; emit transferInProgressChanged(); }

signals:
    void transferCompleted(QSharedPointer<TransferItem> transfer);
    void transferStarted(QSharedPointer<TransferItem> transfer);
    void transferCancelled(QSharedPointer<TransferItem> transfer);
    void transferPaused(QSharedPointer<TransferItem> transfer);
    void transferFailed(QSharedPointer<TransferItem> transfer, const QString &reason);
    void progressChanged(float progress, int eta);
    void sizeChanged(qint64 size);
    void transferInProgressChanged();
    void fileCreated(const QVariantMap &file);

private:
    QNetworkAccessManager *m_nam;
    UbuntuOne *m_ubuntu;
    QFile m_downloadFile;
    QFile m_uploadFile;
    bool m_busy;
    QNetworkReply *m_dreply;
    QNetworkReply *m_ureply;
    QTime m_transferTime;
    QSharedPointer<TransferItem> m_transfer;
};

#endif // TRANSFERWORKER_H
