/*
    TipQalc - A tip, tax and total calculator for touchscreen.
    Copyright (C) 2010 Ianaré Sévi.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "cmath"
#include "tipqalc.h"
#include "ui_tipqalc.h"

TipQalc::TipQalc(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::TipQalc)
{
    ui->setupUi(this);

    // filter events on editable lines
    ui->taxLine->installEventFilter(this);
    ui->billLine->installEventFilter(this);

    connect(ui->numberPad, SIGNAL(numberClickedSignal(QString)),
            this, SLOT(numberClicked(QString)));
    connect(ui->numberPad, SIGNAL(clearClickedSignal()),
            this, SLOT(clearClicked()));
    connect(ui->numberPad, SIGNAL(deleteClickedSignal()),
            this, SLOT(deleteClicked()));
    connect(ui->numberPad, SIGNAL(periodClickedSignal()),
            this, SLOT(periodClicked()));
    connect(ui->actionSettings, SIGNAL(triggered()),
            this, SLOT(showSettingsDialog()));
    connect(ui->actionAbout, SIGNAL(triggered()),
            this, SLOT(showAboutDialog()));

    QCoreApplication::setOrganizationName("SeviSoft");
    QCoreApplication::setOrganizationDomain("sevisoft.com");
    QCoreApplication::setApplicationName("TipQalc");

    QSettings settings;
    //settings.clear();

    QString taxPercent = settings.value("taxPercent").toString();
    ui->taxLine->clear();
    ui->taxLine->insert(taxPercent);

    // tip can never be empty
    if (settings.contains("tipPercent"))
    {
        QString tipPercent = settings.value("tipPercent").toString();
        ui->tipLine->clear();
        ui->tipLine->insert(tipPercent);
    }

    manageTaxInputWidgets();
    // remove as soon as functionality is complete
    ui->roundButton->hide();
}

TipQalc::~TipQalc()
{
    QSettings settings;
    settings.setValue("taxPercent", ui->taxLine->text());
    settings.setValue("tipPercent", ui->tipLine->text());
    delete ui;
}

void TipQalc::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    case QEvent::FocusIn:

    default:
        break;
    }
}

/*
  Show the settings dialog, adjust calculation if needed.
*/
void TipQalc::showSettingsDialog()
{
    SettingsDialog settingsDialog(this);
    settingsDialog.exec();

    if (total) calculateTip();
    manageTaxInputWidgets();
}

/*
  Show the about dialog.
*/
void TipQalc::showAboutDialog()
{
    AboutDialog aboutDialog(this);
    aboutDialog.exec();
}

/*
  Point to currently selected editable line.
*/
bool TipQalc::eventFilter(QObject *object, QEvent *event)
{
    if (event->type() == QEvent::FocusIn)
    {
        editableLine = qFindChild<QLineEdit*>(this, object->objectName());
        return true;
    }
    else
    {
        // standard event processing
        return QObject::eventFilter(object, event);
    }
    

}


/*
  Show or hide tax line inputs.
*/
void TipQalc::manageTaxInputWidgets()
{
    QSettings settings;
    bool useTax = settings.value("subtractTaxFromTotal").toBool();
    if (!useTax)
    {
        ui->taxLabel->hide();
        ui->taxLine->hide();
        editableLine = ui->billLine;
    }
    else
    {
        ui->taxLabel->show();
        ui->taxLine->show();
    }
}

/*
  Increase or decrease the number of persons.
*/
void TipQalc::adjustPersons(int value)
{
    QString text = ui->personsLine->text();
    int currentValue = text.toInt();

    value = currentValue + value;
    if (value < 1) value = 1;

    ui->personsLine->clear();
    ui->personsLine->insert(QString::number(value));

    if (total) calculateTip();
}

/*
  Increase or decrease the tip percentage.
*/
void TipQalc::adjustTip(int value)
{
    QString text = ui->tipLine->text();
    int currentValue = text.toInt();

    value = currentValue + value;
    if (value < 0) value = 0;

    ui->tipLine->clear();
    ui->tipLine->insert(QString::number(value));

    if (total) calculateTip();
}

/*
  Rounding method, precision defaults to 2.
*/
float TipQalc::round( float x, int precision=2)
{
    float sd = pow(10, precision);
    return int(x*sd + (x<0? -0.5 : 0.5))/sd;
}

/*
  Calculate tax and tip, set values in UI.
*/
void TipQalc::calculateTip()
{
    int persons = ui->personsLine->text().toInt();
    float taxPercent = ui->taxLine->text().toFloat();
    float tipPercent = ui->tipLine->text().toFloat();
    float bill = ui->billLine->text().toFloat();

    QSettings settings;
    bool useTax = settings.value("subtractTaxFromTotal").toBool();

    float tip;
    if (useTax)
    {
        float tax = (bill * taxPercent) / 100;
        tip = ( (bill - tax) * tipPercent ) / 100;
    }
    else
    {
        tip = ( bill * tipPercent ) / 100;
    }

    total = round(bill + tip);
    personTotal = round(total / persons);
    personTip = round(tip / persons);

    tip = round(tip);

    ui->totalTipResultLabel->setText(QString::number(tip));
    ui->totalResultLabel->setText(QString::number(total));
    ui->personTotalResultLabel->setText(QString::number(personTotal));
    ui->personTipResultLabel->setText(QString::number(personTip));
}

void TipQalc::roundTotal()
{

}

void TipQalc::roundTip()
{

}

/*
  Clear.
*/
void TipQalc::clearClicked()
{
    editableLine->clear();
    if (total) calculateTip();
}

void TipQalc::adjustCursorPosition()
{
    int length = editableLine->text().length();
    editableLine->setCursorPosition(length);
}

/*
  Delete last character.
*/
void TipQalc::deleteClicked()
{
    adjustCursorPosition();
    editableLine->backspace();
}

/*
  Insert digit.
*/
void TipQalc::numberClicked(QString value)
{
    QString objectName = editableLine->objectName();
    QRegExp rx("");

    // set decimal precision according to field
    if (objectName == "billLine")
    {
        rx.setPattern("\\.\\d\\d");
    }
    else if (objectName == "taxLine")
    {
        rx.setPattern("\\.\\d\\d\\d");
    }

    if (!editableLine->text().contains(rx))
    {
        adjustCursorPosition();
        editableLine->insert(value);
    }

    // remove leading 0
    rx.setPattern("^0\\d");
    QString lineValue = editableLine->text();
    if (lineValue.contains(rx))
    {
        lineValue.remove(0,1);
    }
    editableLine->clear();
    editableLine->insert(lineValue);
}

/*
  Insert period.
*/
void TipQalc::periodClicked()
{
    if (!editableLine->text().contains("."))
    {
        adjustCursorPosition();
        editableLine->insert(".");
    }
}

void TipQalc::on_personsAddButton_clicked()
{
    adjustPersons(1);
}

void TipQalc::on_personsMinusButton_clicked()
{
    adjustPersons(-1);
}

void TipQalc::on_tipAddButton_clicked()
{
    adjustTip(1);
}

void TipQalc::on_tipMinusButton_clicked()
{
    adjustTip(-1);
}

void TipQalc::on_calculateButton_clicked()
{
    calculateTip();
}
