/*
 * This file is part of TimedSilencer.
 *
 *  TimedSilencer is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TimedSilencer is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TimedSilencer.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <QStandardItemModel>
#include <QHeaderView>
#include <QSettings>
#include <QHash>
#include "profileevent.h"
#include "switchingeventlist.h"
#include "newalarmdlg.h"
#include "eventlistdelegate.h"
#include "alarmd_backend.h"

const int ROW_HEIGHT = 60;

SwitchingEventList::SwitchingEventList(QWidget *parent) :
    QTableView(parent)
{
  setSelectionBehavior(QAbstractItemView::SelectRows);
  model = new QStandardItemModel(0, 5);
  // Set Header
  model->setHeaderData(EV_STATUS, Qt::Horizontal, tr("Status"));
  model->setHeaderData(EV_FROM, Qt::Horizontal, tr("From"));
  model->setHeaderData(EV_TO, Qt::Horizontal, tr("To"));
  model->setHeaderData(EV_REPEAT, Qt::Horizontal, tr("Repeat"));
  setModel(model);
  setItemDelegate(new EventListDelegate);
  connect(this, SIGNAL(activated(QModelIndex)), this, SLOT(editEvent(QModelIndex)));
  // Hide vertical header
  verticalHeader()->setVisible(false);
  horizontalHeader()->setStretchLastSection(true);
  hideColumn(EV_ID);
  // Load saved events
  loadSavedEvents();
  // Height hint
  if(verticalHeader()->defaultSectionSize() < ROW_HEIGHT)
    verticalHeader()->setDefaultSectionSize(ROW_HEIGHT);
  if (sizeHintForRow(0)>0)
    setMinimumHeight(sizeHintForRow(0) * 5);
}

SwitchingEventList::~SwitchingEventList() {
  delete model;
}

void SwitchingEventList::updateRow(int row, ProfileEvent *pe) {
  model->setData(model->index(row, EV_STATUS), pe->activated, Qt::UserRole);
  model->setData(model->index(row, EV_FROM), pe->from_time.toString("HH:mm"));
  model->setData(model->index(row, EV_TO), pe->to_time.toString("HH:mm"));
  model->setData(model->index(row, EV_REPEAT), ProfileEvent::formatDays(pe->days));
  model->setData(model->index(row, EV_ID), pe->getID());
}

void SwitchingEventList::editEvent(QModelIndex index) {
  if(!index.isValid()) return;
  QByteArray edited_id = model->data(model->index(index.row(), EV_ID)).toByteArray();
  if(index.column() == EV_STATUS) {
    // Toggle activated state
    const bool new_status = !index.data(Qt::UserRole).toBool();
    model->setData(index, new_status, Qt::UserRole);
    ProfileEvent::setStatus(edited_id, new_status);
    // Alter Alarmd events
    if(new_status) {
      // Was activated
      AlarmdBackend::setProfileEvents(edited_id);
    } else {
      // Was deactivated
      AlarmdBackend::deleteEvents(edited_id);
    }
  } else {
    NewAlarmDlg dlg(this, edited_id);
    connect(&dlg, SIGNAL(editedEvent(QByteArray,bool)), this, SLOT(editEvent(QByteArray,bool)));
    connect(&dlg, SIGNAL(deletedEvent(QByteArray)), this, SLOT(deleteEvent(QByteArray)));
    connect(&dlg, SIGNAL(newEvent(QVariant)), this, SLOT(addNewEvent(QVariant)));
    dlg.exec();
  }
}

void SwitchingEventList::loadSavedEvents() {
  qDebug("Loading saved events");
  QSettings settings("TimedSilencer", "TimedSilencer");
  QHash<QString, QVariant> events = settings.value("events").toHash();
  // Check for < v0.6 settings
  if(!settings.value("from_time", QTime()).toTime().isNull()) {
    qDebug("Old settings from < v0.6 were detected, importing...");
    ProfileEvent pe;
    pe.activated = settings.value("enabled", false).toBool();
    pe.from_time = settings.value("from_time").toTime();
    pe.to_time = settings.value("to_time").toTime();
    pe.days << EVERY_DAY;
    // Cookies
    long from_cookie = settings.value("silencing_cookie", 0).toLongLong();
    if(from_cookie > 0)
      pe.alarmd_cookies << from_cookie;
    long to_cookie = settings.value("unsilencing_cookie", 0).toLongLong();
    if(to_cookie > 0)
      pe.alarmd_cookies << to_cookie;
    if(to_cookie <= 0 || from_cookie <= 0)
      pe.activated = false;
    events.insert(pe.getID(), pe.save());
    // Remove old format values
    settings.clear();
    // Save in new format
    settings.setValue("events", events);
  }
  // Load >= v0.6 settings
  bool settings_change = false;
  foreach(QVariant e, events.values()) {
    ProfileEvent *pe = ProfileEvent::load(e);
    // Check if still active
    if(pe->activated && !AlarmdBackend::checkIfStillActive(pe)) {
      qDebug("An existing profile switching event is no longer active, updating its status");
      pe->activated = false;
      events[pe->getID()] = pe->save();
      settings_change = true;
    }
    // Add new model row
    const int nb_rows = model->rowCount();
    model->setRowCount(nb_rows+1);
    updateRow(nb_rows, pe);
    // Clean up
    delete pe;
  }
  if(settings_change)
    settings.setValue("events", events);
}

void SwitchingEventList::addNewEvent(QVariant var_event) {
  qDebug("Adding a new event to the list");
  ProfileEvent *pe = ProfileEvent::load(var_event);
  // Add new model row
  const int nb_rows = model->rowCount();
  model->setRowCount(nb_rows+1);
  updateRow(nb_rows, pe);
  delete pe;
}

void SwitchingEventList::editEvent(QByteArray id, bool new_status) {
  const int row = getRowFromID(id);
  qDebug("Editing event at row %d", row);
  Q_ASSERT(row >= 0);
  model->setData(model->index(row, EV_STATUS), new_status, Qt::UserRole);
}

void SwitchingEventList::deleteEvent(QByteArray id) {
  const int row = getRowFromID(id);
  qDebug("Deleting an event (row: %d)", row);
  Q_ASSERT(row >= 0);
  model->removeRow(row);
}

int SwitchingEventList::getRowFromID(QByteArray id) {
  for(int i=0; i<model->rowCount(); ++i) {
    if(model->data(model->index(i, EV_ID)).toByteArray() == id)
      return i;
  }
  return -1;
}
