/*
 * This file is part of TimedSilencer.
 *
 *  TimedSilencer is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TimedSilencer is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TimedSilencer.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef ALARMD_TALKER_H
#define ALARMD_TALKER_H

#include <QTime>
#include <QString>
#include <QSettings>
#include <QPair>
#include <alarmd/libalarm.h>
#include <dbus-1.0/dbus/dbus-protocol.h>
#include <time.h>

#include "phone_profile.h"
#include "dbus_backend.h"
#include "profileevent.h"

// Alarmd documentation found at:
// http://wiki.maemo.org/Documentation/Maemo_5_Developer_Guide/Using_Generic_Platform_Components/Alarm_Framework
class AlarmdBackend : public QObject {
  Q_OBJECT
private:
  static time_t toTime_t(const QTime &t) {
    int time_diff = QTime::currentTime().secsTo(t);
    if(time_diff < 0) {
      // Add 24 hours
      time_diff += 86400;
    }
    qDebug("time diff: %d", time_diff);
    return (time_t) time(0) + time_diff;
  }

  static uint32_t daysToMask(QList<int> days) {
    uint32_t mask = 0;
    foreach(const int& d, days) {
      switch(d) {
      case MON:
        mask |= ALARM_RECUR_WDAY_MON;
        break;
      case TUE:
        mask |= ALARM_RECUR_WDAY_TUE;
        break;
      case WED:
        mask |= ALARM_RECUR_WDAY_WED;
        break;
      case THU:
        mask |= ALARM_RECUR_WDAY_THU;
        break;
      case FRI:
        mask |= ALARM_RECUR_WDAY_FRI;
        break;
      case SAT:
        mask |= ALARM_RECUR_WDAY_SAT;
        break;
      case SUN:
        mask |= ALARM_RECUR_WDAY_SUN;
        break;
      default:
        Q_ASSERT(0); // Should never go here
        mask |= ALARM_RECUR_WDAY_ALL;
        break;
      }
    }
    return mask;
  }

public:
  // Is only called on program uninstall
  static void deleteAllEvents() {
    QSettings settings("TimedSilencer", "TimedSilencer");
    QHash<QString, QVariant> events = settings.value("events").toHash();
    foreach(QVariant var_ev, events) {
      ProfileEvent *pe = ProfileEvent::load(var_ev);
      foreach(const long &cookie, pe->alarmd_cookies) {
        qDebug("Unregistering event with cookie %ld", cookie);
        alarmd_event_del(cookie);
      }
      delete pe;
    }
    // Save in QSettings
    events.clear();
    settings.setValue("events", events);
  }

  static void deleteEvents(QByteArray event_id) {
    deleteEvents(ProfileEvent::findByID(event_id));
  }

  static void deleteEvents(ProfileEvent *pe) {
    // unregistering events
    foreach(const long &cookie, pe->alarmd_cookies) {
      qDebug("Unregistering event with cookie %ld", cookie);
      alarmd_event_del(cookie);
    }
    pe->alarmd_cookies.clear();
    ProfileEvent::clearCookies(pe->getID());
  }

  static void setProfileEvents(QByteArray event_id) {
    setProfileEvents(ProfileEvent::findByID(event_id));
  }

  static bool checkIfStillActive(ProfileEvent *pe) {
    Q_ASSERT(pe->activated);
    foreach(const long &cookie, pe->alarmd_cookies) {
      alarm_event_t *eve = 0;
      if((eve = alarmd_event_get(cookie)) != 0) {
        // Free all dynamic memory associated with the alarm event
        alarm_event_delete(eve);
        return true;
      }
    }
    return false;
  }

  static void setProfileEvents(ProfileEvent *pe) {
    Q_ASSERT(pe->activated);
    // First clear old alarmd events
    foreach(const long &cookie, pe->alarmd_cookies) {
      qDebug("Unregistering event with cookie %ld", cookie);
      alarmd_event_del(cookie);
    }
    pe->alarmd_cookies.clear();
    // Then setting new events
    long c1 = newProfileEvent(SILENT, pe->from_time, pe->days);
    Q_ASSERT(c1 > 0);
    if(c1 > 0)
      pe->alarmd_cookies << c1;
    long c2 = newProfileEvent(GENERAL, pe->to_time, pe->days);
    Q_ASSERT(c2 > 0);
    if(c2 > 0)
      pe->alarmd_cookies << c2;
    // Save in QSettings
    ProfileEvent::setCookies(pe->getID(), pe->alarmd_cookies);
    // Set Profile to SILENT if we are currently in the silent time slot
    if(pe->affectsCurrentTime())
      DBusBackend::setProfile(SILENT);
  }

protected:
  static long newProfileEvent(Profile p, const QTime &event_time, QList<int> days) {
    Q_ASSERT(!days.empty());
    if(days.empty()) days << NEVER;
    // Create the default alarm struct.
    alarm_event_t *newEvent = alarm_event_create();
    // Set the APP ID
    alarm_event_set_alarm_appid(newEvent, "TimedSilencer");
    // Set the title
    if(p == SILENT)
      alarm_event_set_title(newEvent, "silent_profile");
    else
      alarm_event_set_title(newEvent, "general_profile");
    // Timing
    if(days.first() == EVERY_DAY) {
      newEvent->recur_secs = 86400; // 24 hours
      newEvent->recur_count = -1; // Reoccur infinitely
      newEvent->alarm_time = toTime_t(event_time); // Set event time
    } else {
      if(days.first() == NEVER) {
        newEvent->alarm_time = toTime_t(event_time); // Set event time
      } else {
        newEvent->recur_secs = 0; // We re not using this way for recurrence
        alarm_recur_t* recur = alarm_event_add_recurrences(newEvent, 1);
        // Set event time
        recur->mask_hour |= (1ul << event_time.hour());
        recur->mask_min |= (1ull << event_time.minute());
        recur->mask_wday |= daysToMask(days);
      }
    }
    //Add 1 action to our alarm event, and assign it to the "act" variable
    alarm_action_t *act = alarm_event_add_actions(newEvent, 1);
    // Actions are documented here:
    // http://maemo.org/api_refs/5.0/5.0-final/libalarm/libalarm_8h.html#cc8e6f439d134448001132132476683c910f7626ec85a4170659b53fa2f0abc7
    //Setup this action to be an "DBus command" one; also set it up to use DBUS auto-activation.
    act->flags = ALARM_ACTION_WHEN_TRIGGERED | ALARM_ACTION_DBUS_USE_ACTIVATION | ALARM_ACTION_TYPE_DBUS;

    //Setup the DBus params for this action
    alarm_action_set_dbus_interface(act, "com.nokia.profiled");
    alarm_action_set_dbus_service(act, "com.nokia.profiled");
    alarm_action_set_dbus_path(act, "/com/nokia/profiled");
    alarm_action_set_dbus_name(act, "set_profile");

    if(p == SILENT) {
      const char* param = "silent";
      alarm_action_set_dbus_args(act, DBUS_TYPE_STRING, &param, DBUS_TYPE_INVALID);
    } else {
      const char* param = "general";
      alarm_action_set_dbus_args(act, DBUS_TYPE_STRING, &param, DBUS_TYPE_INVALID);
    }

    // Finally with everything setup, try to add your event to the alarm queue
    long cookie = alarmd_event_add(newEvent);
    // Free all dynamic memory associated with the alarm event
    alarm_event_delete(newEvent);
    return cookie;
  }
};

#endif // ALARMD_TALKER_H
