/*
 * This file is part of TimedSilencer.
 *
 *  TimedSilencer is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  TimedSilencer is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with TimedSilencer.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef ALARMD_TALKER_H
#define ALARMD_TALKER_H

#include <QTime>
#include <QString>
#include <QSettings>
#include <alarmd/libalarm.h>
#include <dbus-1.0/dbus/dbus-protocol.h>
#include <time.h>

#include "phone_profile.h"

// Alarmd documentation found at:
// http://wiki.maemo.org/Documentation/Maemo_5_Developer_Guide/Using_Generic_Platform_Components/Alarm_Framework
class AlarmdBackend : public QObject {
  Q_OBJECT
private:
  static time_t toTime_t(const QTime &t) {
    int time_diff = QTime::currentTime().secsTo(t);
    if(time_diff < 0) {
      // Add 24 hours
      time_diff += 86400;
    }
    qDebug("time diff: %d", time_diff);
    return (time_t) time(0) + time_diff;
  }

public:
  static void deleteEvents() {
    // Get events cookies
    QSettings settings("TimedSilencer", "TimedSilencer");
    const cookie_t silent_cookie = settings.value("silencing_cookie", 0).toLongLong();
    if(silent_cookie != 0) {
      qDebug("Deleting silent profile event with cookie %ld", (long) silent_cookie);
      alarmd_event_del(silent_cookie);
    }
    const cookie_t general_cookie = settings.value("unsilencing_cookie", 0).toLongLong();
    if(general_cookie != 0) {
      qDebug("Deleting general profile event with cookie %ld", (long) general_cookie);
      alarmd_event_del(general_cookie);
    }
  }

  static void setProfileEvent(Profile p, const QTime &event_time) {
    // Get event cookie
    QSettings settings("TimedSilencer", "TimedSilencer");
    cookie_t cookie;
    if(p == SILENT)
      cookie = settings.value("silencing_cookie", 0).toLongLong();
    else
      cookie = settings.value("unsilencing_cookie", 0).toLongLong();
    alarm_event_t *eve = 0;
    if(cookie == 0 || (eve = alarmd_event_get(cookie)) == 0) {
      qDebug("Profile event does not exist yet, creating it...");
      // The event does not exist yet
      newProfileEvent(p, event_time);
      return;
    }
    // Update existing event
    qDebug("Updating profile event with cookie %ld", (long)cookie);
    eve->alarm_time = toTime_t(event_time);
    alarmd_event_update(eve);
  }

protected:
  static void newProfileEvent(Profile p, const QTime &event_time) {
    // Create the default alarm struct.
    alarm_event_t *newEvent = alarm_event_create();
    // Set the APP ID
    alarm_event_set_alarm_appid(newEvent, "TimedSilencer");
    // Set the title
    if(p == SILENT)
      alarm_event_set_title(newEvent, "silent_profile");
    else
      alarm_event_set_title(newEvent, "general_profile");
    // Timing
    newEvent->recur_secs = 86400; // 24 hours interval
    newEvent->recur_count = -1; // Reoccur infinitely
    newEvent->alarm_time = toTime_t(event_time);
    //Add 1 action to our alarm event, and assign it to the "act" variable
    alarm_action_t *act = alarm_event_add_actions(newEvent, 1);
    // Actions are documented here:
    // http://maemo.org/api_refs/5.0/5.0-final/libalarm/libalarm_8h.html#cc8e6f439d134448001132132476683c910f7626ec85a4170659b53fa2f0abc7
    //Setup this action to be an "DBus command" one; also set it up to use DBUS auto-activation.
    act->flags = ALARM_ACTION_WHEN_TRIGGERED | ALARM_ACTION_DBUS_USE_ACTIVATION | ALARM_ACTION_TYPE_DBUS;

    //Setup the DBus params for this action
    alarm_action_set_dbus_interface(act, "com.nokia.profiled");
    alarm_action_set_dbus_service(act, "com.nokia.profiled");
    alarm_action_set_dbus_path(act, "/com/nokia/profiled");
    alarm_action_set_dbus_name(act, "set_profile");

    if(p == SILENT) {
      const char* param = "silent";
      alarm_action_set_dbus_args(act, DBUS_TYPE_STRING, &param, DBUS_TYPE_INVALID);
    } else {
      const char* param = "general";
      alarm_action_set_dbus_args(act, DBUS_TYPE_STRING, &param, DBUS_TYPE_INVALID);
    }

    // Finally with everything setup, try to add your event to the alarm queue
    const cookie_t &cookie = alarmd_event_add(newEvent);
    if(cookie != 0) {
      // Save cookie
      QSettings settings("TimedSilencer", "TimedSilencer");
      if(p == SILENT) {
        qDebug("Saving silent profile event cookie: %ld", (long)cookie);
        settings.setValue("silencing_cookie", static_cast<const qlonglong>(cookie));
      } else {
        qDebug("Saving silent general event cookie: %ld", (long)cookie);
        settings.setValue("unsilencing_cookie", static_cast<const qlonglong>(cookie));
      }
      return;
    }
    qDebug("ERROR: Failed to add profile event to the queue!");
  }
};

#endif // ALARMD_TALKER_H
