/*
 * longcat: An account plugin for a telepathy connection manager
 * Copyright (C) 2009 Collabora, Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <config.h>
#include <gtk/gtk.h>
#include <glade/glade.h>

#include "longcat.h"

#include <librtcom-accounts-widgets/rtcom-account-item.h>
#include <librtcom-accounts-widgets/rtcom-login.h>
#include <librtcom-accounts-widgets/rtcom-edit.h>
#include <librtcom-accounts-widgets/rtcom-param-int.h>
#include <librtcom-accounts-widgets/rtcom-param-string.h>
#include <librtcom-accounts-widgets/rtcom-param-bool.h>

#include <hildon/hildon-check-button.h>
#include <hildon/hildon-banner.h>
#include <hildon/hildon-entry.h>

#include <libmcclient/mc-profile.h>

#define H_(x) dgettext("hildon-libs", x)

G_DEFINE_TYPE (Longcat, longcat, RTCOM_TYPE_ACCOUNT_PLUGIN);

#define LONGCAT_PRIV(ap) (G_TYPE_INSTANCE_GET_PRIVATE ((ap), \
            LONGCAT_TYPE, LongcatPrivate))

typedef struct {
  gchar *name;
  guint caps;
} LongcatPrivate;

enum /* properties */
{
  PROP_NAME = 1,
  PROP_CAPS,
  LAST_PROP
};

static void
longcat_init (Longcat *self)
{
}

static void
longcat_set_property (GObject *object,
    guint prop_id,
    const GValue *value,
    GParamSpec *pspec)
{
  Longcat *self = LONGCAT (object);
  LongcatPrivate *priv = LONGCAT_PRIV (self);

  switch (prop_id)
    {
      case PROP_NAME:
        g_free (priv->name);
        priv->name = g_value_dup_string (value);
        break;
      case PROP_CAPS:
        priv->caps = g_value_get_uint (value);
        break;
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
longcat_get_property (GObject *object,
    guint prop_id,
    GValue *value,
    GParamSpec *pspec)
{
  Longcat *self = LONGCAT (object);
  LongcatPrivate *priv = LONGCAT_PRIV (self);

  switch (prop_id)
    {
      case PROP_NAME:
        g_value_set_string (value, priv->name);
        break;
      case PROP_CAPS:
        g_value_set_uint (value, priv->caps);
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
longcat_finalize (GObject *object)
{
  LongcatPrivate *priv = LONGCAT_PRIV (object);

  if (priv->name != NULL)
    {
      g_free (priv->name);
      priv->name = NULL;
    }
}

static void
longcat_constructed (GObject *object)
{
  RtcomAccountPlugin *rtcom_plugin = RTCOM_ACCOUNT_PLUGIN (object);
  LongcatPrivate *priv = LONGCAT_PRIV (object);
  GList *profiles, *list;
  gchar *cfg_name;

  rtcom_plugin->name = priv->name;
  rtcom_plugin->username_prefill = "";
  rtcom_plugin->capabilities = priv->caps;

  cfg_name = g_strdup_printf ("%s-plugin", priv->name);

  profiles = mc_profiles_list ();
  for (list = profiles; list; list = list->next)
    {
      McProfile *profile = MC_PROFILE (list->data);
      const gchar *name, *cfg_ui;

      cfg_ui = mc_profile_get_configuration_ui (profile);
      if (cfg_ui != NULL && strcmp (cfg_ui, cfg_name) == 0)
        {
          name = mc_profile_get_unique_name (profile);
          rtcom_account_plugin_add_service (rtcom_plugin, name);
        }
    }

  mc_profiles_free_list (profiles);
  g_free (cfg_name);

  glade_init ();
}

static void
longcat_fill_page (Longcat *plugin,
    GtkWidget *page)
{
  GList *children;
  GtkWidget *area, *main_align, *viewport, *main_pane, *login_table;
  GtkWidget *table;

  if (LONGCAT_GET_CLASS (plugin)->fill_page == NULL)
    return;

  /* Kind of hacky, but means we can use RtcomEdit/Login functions. */

  /* Get the HildonPannableArea from the RtcomPage object */
  children = gtk_container_get_children (GTK_CONTAINER (page));
  area = children->data;
  g_list_free (children);

  /* Get the GtkViewport from the HildonPannableArea */
  children = gtk_container_get_children (GTK_CONTAINER (area));
  viewport = children->data;
  g_list_free (children);

  /* Get the GtkAlignment from the GtkViewport */
  children = gtk_container_get_children (GTK_CONTAINER (viewport));
  main_align = children->data;
  g_list_free (children);

  /* Get the GtkVBox from the GtkAlignment */
  children = gtk_container_get_children (GTK_CONTAINER (main_align));
  main_pane = children->data;
  g_list_free (children);

  /* Get the GtkTable from the GtkVBox */
  children = gtk_container_get_children (GTK_CONTAINER (main_pane));
  login_table = children->data;
  g_list_free (children);

  /* Keep a ref to the login table but remove it from the main pane */
  g_object_ref (login_table);
  g_object_weak_ref (G_OBJECT (page), (GWeakNotify) g_object_unref,
      login_table);
  gtk_container_remove (GTK_CONTAINER (main_pane), login_table);

  table = gtk_table_new (1, 2, FALSE);
  gtk_widget_show (table);

  gtk_box_pack_start (GTK_BOX (main_pane), table, FALSE, FALSE, 0);
  gtk_box_reorder_child (GTK_BOX (main_pane), table, GTK_PACK_START);

  if (LONGCAT_GET_CLASS (plugin)->fill_page != NULL)
    LONGCAT_GET_CLASS (plugin)->fill_page (plugin, GTK_TABLE (table));
}

static void
get_advanced_settings (GtkWidget *widget,
    GHashTable *advanced_settings)
{
  const gchar *name;

  name = gtk_widget_get_name (widget);

  if (RTCOM_IS_PARAM_INT (widget))
    {
      gint value = rtcom_param_int_get_value (RTCOM_PARAM_INT (widget));
      g_hash_table_replace (advanced_settings, widget,
          GINT_TO_POINTER (value));
    }
  else if (GTK_IS_ENTRY (widget))
    {
      gchar *data = g_strdup (gtk_entry_get_text (GTK_ENTRY (widget)));
      g_object_set_data_full (G_OBJECT (widget), "adv_data", data, g_free);
      g_hash_table_replace (advanced_settings, widget, data);
    }
  else if (HILDON_IS_CHECK_BUTTON (widget))
    {
      gboolean active =
        hildon_check_button_get_active (HILDON_CHECK_BUTTON (widget));
      g_hash_table_replace (advanced_settings, widget,
          GINT_TO_POINTER (active));
    }
  else if (GTK_IS_CONTAINER (widget))
    {
      gtk_container_foreach (GTK_CONTAINER (widget),
          (GtkCallback)get_advanced_settings,
          advanced_settings);
    }
  else if (!GTK_IS_LABEL (widget))
    {
      g_warning ("%s: unhandled widget type %s (%s)", G_STRFUNC,
          g_type_name (G_TYPE_FROM_INSTANCE (widget)), name);
    }
}

static void
set_widget_setting (gpointer key,
    gpointer value,
    gpointer userdata)
{
  GtkWidget *widget = key;

  if (RTCOM_IS_PARAM_INT (widget))
    {
      rtcom_param_int_set_value (RTCOM_PARAM_INT (widget),
          GPOINTER_TO_INT (value));
    }
  else if (GTK_IS_ENTRY (widget))
    {
      gtk_entry_set_text (GTK_ENTRY (widget), value);
    }
  else if (HILDON_IS_CHECK_BUTTON (widget))
    {
      hildon_check_button_set_active (HILDON_CHECK_BUTTON (widget),
          GPOINTER_TO_INT (value));
    }
  else
    {
      g_warning ("%s: unhandled widget type %s (%s)", G_STRFUNC,
          g_type_name (G_TYPE_FROM_INSTANCE (widget)),
          gtk_widget_get_name (widget));
    }
}

static void
set_advanced_settings (GtkWidget *dialog, GHashTable *advanced_settings)
{
  g_hash_table_foreach (advanced_settings, set_widget_setting, dialog);
}

static GtkWindow *
_get_main_window (RtcomDialogContext *context)
{
  GtkWindow *main_window = NULL;
  GtkWidget *context_start_page = NULL;

  context_start_page = rtcom_dialog_context_get_start_page (context);
  if (context_start_page)
    main_window = GTK_WINDOW (gtk_widget_get_toplevel (context_start_page));

  return main_window;
}

static void
on_advanced_settings_response (GtkWidget *dialog,
    gint response,
    RtcomDialogContext *context)
{
  GHashTable *advanced_settings;

  advanced_settings = g_object_get_data (G_OBJECT (context), "settings");

  if (response == GTK_RESPONSE_OK)
    {
      GError *error = NULL;
      GtkWidget *page;
      GladeXML *xml;

      xml = glade_get_widget_tree (dialog);
      page = glade_xml_get_widget (xml, "page");
      if (!rtcom_page_validate (RTCOM_PAGE (page), &error))
        {
          g_warning ("advanced page validation failed");
          if (error)
            {
              g_warning ("%s: error \"%s\"", G_STRFUNC, error->message);
              hildon_banner_show_information (GTK_WIDGET (dialog), NULL,
                  error->message);
              g_error_free (error);
            }
          return;
        }

      get_advanced_settings (dialog, advanced_settings);
    }
  else
    {
      /* dialog has been cancelled; overwrite the widget's data with those
       * stored in the advanced_settings hash table */
      set_advanced_settings (dialog, advanced_settings);
    }

  gtk_widget_hide (dialog);
}

static GtkWidget *
create_advanced_settings_page (RtcomDialogContext *context)
{
  GtkWidget *dialog;

  dialog = g_object_get_data (G_OBJECT (context), "page_advanced");
  if (!dialog)
    {
      GladeXML *xml;
      const gchar *format, *profile_name;
      gchar text[200];
      AccountService *service;
      AccountItem *account;
      GtkWidget *page;
      GtkWidget *vbox;
      GtkWidget *table;
      GHashTable *advanced_settings = NULL;
      Longcat *plugin;
      LongcatPrivate *priv;

      xml = glade_xml_new (PLUGIN_XML_DIR "/longcat-advanced.glade", NULL,
          GETTEXT_PACKAGE);
      rtcom_dialog_context_take_obj (context, G_OBJECT (xml));
      dialog = glade_xml_get_widget (xml, "advanced");

      if (!dialog)
        {
          g_warning ("Unable to load Advanced settings dialog");
          return NULL;
        }

      gtk_dialog_add_buttons (GTK_DIALOG (dialog),
          H_("wdgt_bd_done"), GTK_RESPONSE_OK,
          NULL);

      g_object_ref (dialog);
      g_object_set_data_full (G_OBJECT (context), "page_advanced", dialog,
          g_object_unref);

      account = account_edit_context_get_account
        (ACCOUNT_EDIT_CONTEXT (context));

      page = glade_xml_get_widget (xml, "page");
      rtcom_page_set_account (RTCOM_PAGE (page),
          RTCOM_ACCOUNT_ITEM (account));

      service = account_item_get_service (account);
      profile_name = account_service_get_display_name (service);

      format = _("accountwizard_ti_advanced_settings");
      g_snprintf (text, sizeof (text), format, profile_name);

      gtk_window_set_title (GTK_WINDOW (dialog), text);
      gtk_window_set_transient_for (GTK_WINDOW (dialog),
          _get_main_window (context));
      gtk_window_set_destroy_with_parent (GTK_WINDOW (dialog), TRUE);

      /* Create the table for param settings and add it to the vbox */
      table = gtk_table_new (1, 2, FALSE);
      gtk_table_set_row_spacings (GTK_TABLE (table), 5);
      gtk_table_set_col_spacings (GTK_TABLE (table), 16);
      gtk_widget_show (table);

      vbox = glade_xml_get_widget (xml, "vbox");
      gtk_box_pack_start (GTK_BOX (vbox), table, TRUE, TRUE, 0);

      plugin = LONGCAT (account_item_get_plugin (account));
      priv = LONGCAT_PRIV (plugin);

      if (LONGCAT_GET_CLASS (plugin)->fill_advanced_table != NULL)
        LONGCAT_GET_CLASS (plugin)->fill_advanced_table (plugin,
            RTCOM_ACCOUNT_SERVICE (service), account, table);

      advanced_settings = g_hash_table_new (g_direct_hash, g_direct_equal);
      get_advanced_settings (dialog, advanced_settings);
      g_object_set_data_full (G_OBJECT (context), "settings",
          advanced_settings, (GDestroyNotify) g_hash_table_destroy);
    }

  g_signal_connect (dialog, "response",
      G_CALLBACK (on_advanced_settings_response), context);
  /* Prevent the dialog from being destroyed on delete-event */
  g_signal_connect (dialog, "delete-event",
      G_CALLBACK (gtk_true), NULL);

  return dialog;
}

static gboolean
create_advanced_settings_page_idle (gpointer data)
{
  create_advanced_settings_page (RTCOM_DIALOG_CONTEXT (data));
  return FALSE;
}

static void
on_advanced_clicked (gpointer data)
{
  GtkWidget *dialog;
  RtcomDialogContext *context = RTCOM_DIALOG_CONTEXT(data);

  dialog = create_advanced_settings_page (context);
  gtk_widget_show (dialog);
}

static void
context_init (RtcomAccountPlugin *plugin,
    RtcomDialogContext *context)
{
  gboolean editing;
  GtkWidget *page = NULL;
  RtcomAccountItem *account;

  g_idle_add (create_advanced_settings_page_idle, context);

  editing = account_edit_context_get_editing (ACCOUNT_EDIT_CONTEXT (context));
  account = RTCOM_ACCOUNT_ITEM (
      account_edit_context_get_account (ACCOUNT_EDIT_CONTEXT (context)));

  if (editing)
    {
      g_debug ("%s plugin builds edit page...", plugin->name);

      if (LONGCAT_GET_CLASS (plugin)->get_edit_page != NULL)
        page = LONGCAT_GET_CLASS (plugin)->get_edit_page (
            LONGCAT (plugin), account);

      g_assert (RTCOM_IS_EDIT (page));

      if (LONGCAT_GET_CLASS (plugin)->fill_page != NULL)
        longcat_fill_page (LONGCAT (plugin), page);

      rtcom_edit_connect_on_advanced (RTCOM_EDIT (page),
          G_CALLBACK (on_advanced_clicked), context);
    }
  else
    {
      g_debug ("%s plugin builds start page...", plugin->name);

      if (LONGCAT_GET_CLASS (plugin)->get_login_page != NULL)
        page = LONGCAT_GET_CLASS (plugin)->get_login_page (
            LONGCAT (plugin), account);

      g_assert (RTCOM_IS_LOGIN (page));

      if (LONGCAT_GET_CLASS (plugin)->fill_page != NULL)
        longcat_fill_page (LONGCAT (plugin), page);

      rtcom_login_connect_on_advanced (RTCOM_LOGIN (page),
          G_CALLBACK (on_advanced_clicked), context);
    }

  rtcom_dialog_context_set_start_page (context, page);
}

static void
longcat_class_init (LongcatClass *klass)
{
  GObjectClass* object_class = G_OBJECT_CLASS (klass);
  RtcomAccountPluginClass *rtcom_class = RTCOM_ACCOUNT_PLUGIN_CLASS (klass);

  object_class->get_property = longcat_get_property;
  object_class->set_property = longcat_set_property;
  object_class->constructed = longcat_constructed;
  object_class->finalize = longcat_finalize;

  rtcom_class->context_init = context_init;

  g_object_class_install_property (object_class, PROP_NAME,
      g_param_spec_string ("name", "name", "name", "",
          G_PARAM_STATIC_STRINGS | G_PARAM_READWRITE));

  g_object_class_install_property (object_class, PROP_CAPS,
      g_param_spec_uint ("caps", "caps", "caps", 0, G_MAXUINT, 0,
          G_PARAM_STATIC_STRINGS | G_PARAM_READWRITE));

  g_type_class_add_private (object_class, sizeof (LongcatPrivate));
}
