using GLib;
using Gtk;

public class Tear.OvBox : Gtk.Box { // This is translation to Vala from libview, copyright of VMWare with some changes to support top and bottom.
	
	private Gdk.Window underWin;
	private Gtk.Widget under;
	private Gdk.Window overWin;
	private Gtk.Widget over;
	private Gtk.Requisition overR;
	private double fraction;
	private int verticalOffset;
	
	public int min;
	public string orientation;
	
	construct {
		if (DEBUG)
			warning ( "construct" );
		
		this.underWin = null;
		this.under = null;
		this.overWin = null;
		this.over = null;
		this.overR.height = -1;
		this.overR.width = -1;
		this.min = 0;
		this.fraction = 0;
		this.verticalOffset = 0;
	}

	public override void map () {
		if (DEBUG)
			warning ( "map" );
		this.window.show();
		base.map();
	}

  	public override void unmap () {
		if (DEBUG)
			warning ( "unmap" );
		this.window.hide();
		base.unmap();
	}
	
	public int get_actual_min () {
		if (DEBUG)
			warning ( "get_actual_min" );
		return int.min(this.min, this.overR.height);
	}

 	public void get_under_geometry ( out int x, out int y, out int width, out int height) {
		if (DEBUG)
			warning ( "get_under_geometry" );
 		int min;
 	
 		min = this.get_actual_min();
 	
 		x = 0;
		if ( this.orientation == "top" )
	 		y = min;
	 	else
	 		y = 0;
 		width = this.allocation.width;
 		height = this.allocation.height - min;
 	}
 	
 	public void get_over_geometry ( out int x, out int y, out int width, out int height) {
		if (DEBUG)
			warning ( "get_over_geometry" );
		bool expand, fill;
 		int padding, boxWidth, sub;
 	
 		if (this.over != null) {
 			/*
 			 * When a child's expand or fill property changes, GtkBox queues
 			 * a resize for the child.
 			 */
 	      this.child_get( this.over, "expand", out expand, "fill", out fill, "padding", out padding );
 		} else {
 			/* Default values used by GtkBox. */
 			expand = true;
 			fill = true;
 			padding = 0;
 		}

		boxWidth = this.allocation.width;
		
		if (!expand) {
			width = int.min( this.overR.width, boxWidth - padding );
			x = padding;
		} else 
			if (!fill) {
				width = int.min( this.overR.width, boxWidth );
				x = (int) Math.round((boxWidth - width) / 2);
 	   		} else {
				width = boxWidth;
				x = 0;
			}
			
		if ( this.orientation == "top" )
			y = (int) Math.round((this.overR.height - this.get_actual_min()) * (this.fraction - 1) + this.verticalOffset);
		else {
			sub = this.get_actual_min();
			sub = (this.min == 9999 ? this.overR.height : (int) Math.round((this.overR.height - sub) * this.fraction) );
			y = this.allocation.height - sub;
		}
		height = this.overR.height;
 	}
 	
	public void set_background () {
		if (DEBUG)
			warning ( "set_background" );
		this.style.set_background ( this.window, Gtk.StateType.NORMAL);
		this.style.set_background ( this.underWin, Gtk.StateType.NORMAL);
		this.style.set_background ( this.overWin, Gtk.StateType.NORMAL);
	}
	
	public override void realize () {
		Gdk.WindowAttr attributes = Gdk.WindowAttr();
		int mask;
		
		if (DEBUG)
			warning ( "realize" );
		
		this.set_flags( Gtk.WidgetFlags.REALIZED );
		
		attributes.window_type = Gdk.WindowType.CHILD;
		attributes.wclass = Gdk.WindowClass.INPUT_OUTPUT;
		attributes.visual = this.get_visual ();
		attributes.colormap = this.get_colormap ();
		attributes.event_mask = this.get_events () | Gdk.EventMask.EXPOSURE_MASK;
		mask = Gdk.WindowAttributesType.VISUAL | Gdk.WindowAttributesType.COLORMAP | Gdk.WindowAttributesType.X | Gdk.WindowAttributesType.Y;
		
		attributes.x = this.allocation.x;
		attributes.y = this.allocation.y;
		attributes.width = this.allocation.width;
		attributes.height = this.allocation.height;
		this.window = new Gdk.Window ( this.get_parent_window(), attributes, mask );
		
		this.window.set_user_data ( this );
		this.style = this.style.attach ( this.window );
		
		/*
		 * The order in which we create the children X window matters: the child
		 * created last is stacked on top. --hpreg
		 */
		
		this.get_under_geometry ( out attributes.x, out attributes.y, out attributes.width, out attributes.height);
		this.underWin = new Gdk.Window ( this.window, attributes, mask);
		this.underWin.set_user_data ( this );
		if (this.under != null) {
		   this.under.set_parent_window( this.underWin );
		}
		this.underWin.show();
		
		this.get_over_geometry ( out attributes.x, out attributes.y, out attributes.width, out attributes.height);
		this.overWin = new Gdk.Window ( this.window, attributes, mask);
		this.overWin.set_user_data ( this );
		if (this.over != null) {
		   this.over.set_parent_window ( this.overWin );
		}
		this.overWin.show ();
		
		this.set_background();
	}
	
	public override void unrealize () {
		if (DEBUG)
			warning ( "unrealize" );
		/*
		 * Unrealize the parent before destroying the windows so that we end up
		 * unrealizing all the child widgets before destroying the child windows,
		 * giving them a chance to reparent their windows before we clobber them.
		 */
		base.unrealize ();
	
		this.underWin.set_user_data ( null );
		this.underWin.destroy ();
		this.underWin = null;
	
		this.overWin.set_user_data ( null );
		this.overWin.destroy ();
		this.overWin = null;
	}
	
	public override void size_request( out Gtk.Requisition requisition ) {
		if (DEBUG)
			warning ( "size_request" );
		Gtk.Requisition underR;
		bool expand, fill;
		int padding, min;
	
		this.under.size_request ( out underR );
		this.over.size_request ( out this.overR );
	
		this.child_get( this.over, "expand", out expand, "fill", out fill, "padding", out padding );
		requisition.width = int.max ( underR.width, this.overR.width + ((expand || fill) ? 0 : padding) );
		min = this.get_actual_min ();
		requisition.height = int.max ( underR.height + min, this.overR.height);
	}
	
	public override void size_allocate ( Gdk.Rectangle allocation ) {
		Gtk.Allocation under = Gtk.Allocation();
		Gtk.Allocation over = Gtk.Allocation();
	
		this.allocation = (Gtk.Allocation) allocation;
	
		this.get_under_geometry ( out under.x, out under.y, out under.width, out under.height );
		this.get_over_geometry ( out over.x, out over.y, out over.width, out over.height);
	
		if (this.is_realized()) {
			this.window.move_resize ( allocation.x, allocation.y, allocation.width, allocation.height );
			this.underWin.move_resize ( under.x, under.y, under.width, under.height );
			this.overWin.move_resize ( over.x, over.y, over.width, over.height );
		}
	
		under.x = 0;
		under.y = 0;
		this.under.size_allocate ( (Gdk.Rectangle) under );
		over.x = 0;
		over.y = 0;
		this.over.size_allocate ( (Gdk.Rectangle) over );
	}
	
	public override void style_set ( Gtk.Style? previousStyle ) {
		if (DEBUG)
			warning ( "size_set" );
		if (this.is_realized()) {
			this.set_background();
		}

		base.style_set( previousStyle );
	}
	
	public void set_child ( ref Gtk.Widget child, Gdk.Window childWin, Gtk.Widget widget )	{
		if (DEBUG)
			warning ( "set_child" );
		Gtk.Widget oldChild = child;
	
		if ( oldChild != null ) {
			oldChild.destroy();
		}

		child = widget;
		if ( child != null ) {
	 		widget.set_parent_window( childWin );
			this.add( child );
		}

	}
	
	public void set_over ( Gtk.Widget widget ) {
		if (DEBUG)
			warning ( "set_over" );
		this.set_child( ref this.over, this.overWin, widget);
	}
	
	public void set_under( Gtk.Widget widget ) {
		return_if_fail(this != null);
		if (DEBUG)
			warning ( "set_under" );
	   
		this.set_child ( ref this.under, this.underWin, widget );
	}
	
	public void set_min ( int min ) {
		return_if_fail(this != null);
		if (DEBUG)
			warning ( "set_min" );
	
   		this.min = (min < 0) ? 9999 : min;
		this.queue_resize();
	}
	
	public void set_orientation ( string orientation ) {
		return_if_fail(this != null);
		return_if_fail ( orientation == "top" || orientation == "bottom" );
		
		this.orientation = orientation;
		if (this.is_realized()) {
			int x, y, width, height;

			this.get_over_geometry ( out x, out y, out width, out height );
			overWin.move ( x, y );
			this.queue_resize();
		}
	}

	public void set_fraction ( double fraction ) {
		return_if_fail(this != null);
		return_if_fail ( fraction >=0 && fraction <= 1 );
		if (DEBUG)
			warning ( "set_fraction" );
	
		this.fraction = fraction;
		if (this.is_realized()) {
			int x, y, width, height;

			this.get_over_geometry ( out x, out y, out width, out height );
			overWin.move ( x, y );
		}
	}
	
	public double get_fraction() {
		return_val_if_fail(this != null, 0);
		if (DEBUG)
			warning ( "get_fraction" );
	
		return this.fraction;
	}
	
	public void set_vertical_offset( int offset ) {
		return_if_fail(this != null);
		if (DEBUG)
			warning ( "set_vertical_offset" );
	
		this.verticalOffset = offset;
		if (this.is_realized()) {
			int x, y, width, height;

			this.get_over_geometry ( out x, out y, out width, out height );
			overWin.move ( x, y );
		}
	}

}
